﻿#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Reflection;
using System.Windows.Media.Imaging; // for ribbon, requires references to PresentationCore and WindowsBase .NET assemblies
using Autodesk.Revit;
#endregion // Namespaces

namespace RibbonEmbedImage
{
  public class App : IExternalApplication
  {
    /// <summary>
    /// Message used as ribbon panel button tooltip 
    /// and displayed by the external command.
    /// </summary>
    public const string Message =
      "Ribbon Embed Image says 'Hello world'"
      + " via an embedded bitmap resource.";

    /// <summary>
    /// Return embedded bitmap resource.
    /// The 'name' argument is the fully qualified resource name, 
    /// i.e. includes the application namespace etc. 
    /// For example, for an application namespace "MyNamespace" 
    /// and an image "MyImage.png" you would call the function 
    /// like this: 
    /// 
    ///   MyPushButton.LargeImage = GetEmebeddedImage( 
    ///     "MyNamespace.MyImage.png" );
    /// </summary>
    /// <param name="name">Fully qualified resource name, e.g. "MyNamespace.MyImage.png"</param>
    /// <returns>Embedded bitmap resource</returns>
    static BitmapSource GetEmbeddedImage( string name )
    {
      try
      {
        Assembly a = Assembly.GetExecutingAssembly();
        Stream s = a.GetManifestResourceStream( name );
        return BitmapFrame.Create( s );
      }
      catch
      {
        return null;
      }
    }

    static void AddRibbonPanel(
      ControlledApplication a )
    {
      string path = Assembly.GetExecutingAssembly().Location;

      PushButtonData data = new PushButtonData(
        "Hello", "Hello", path, "RibbonEmbedImage.Command" );

      BitmapSource bitmap16 = GetEmbeddedImage(
        "RibbonEmbedImage.Bitmap16.bmp" );

      BitmapSource bitmap32 = GetEmbeddedImage( 
        "RibbonEmbedImage.Bitmap32.bmp" );

      data.Image = bitmap16;
      data.LargeImage = bitmap32;
      data.ToolTip = Message;

      RibbonPanel panel = a.CreateRibbonPanel(
        "Ribbon Embed Image" );

      RibbonItem item = panel.AddButton( data );
    }

    public IExternalApplication.Result OnStartup( 
      ControlledApplication a )
    {
      AddRibbonPanel( a );
      return IExternalApplication.Result.Succeeded;
    }

    public IExternalApplication.Result OnShutdown(
      ControlledApplication a )
    {
      return IExternalApplication.Result.Succeeded;
    }
  }

  public class Command : IExternalCommand
  {
    public IExternalCommand.Result Execute( 
      ExternalCommandData commandData, 
      ref string message, 
      ElementSet elements )
    {
      message = App.Message;
      return IExternalCommand.Result.Failed;
    }
  }
}
