﻿using System;
using System.Collections.Generic;
using System.Reflection;
using System.Windows.Forms;

using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.UI;

using adWin = Autodesk.Windows;

namespace RibbonMoveExample
{
	/// <summary>	
	/// Demonstrates the use of Autodesk.Windows.RibbonControl to move API created ribbon buttons and / or panels from thier
	/// original location to any other tab / panel including system created ones.
	/// I'm pretty sure that this technique would also work in reverse to allow grabbing system panels / buttons and adding
	/// them to API created Tabs and even other system tabs, meaning that some drastic customisation of Revit's ribbon menu is possible.
	/// Revit CUI addin anybody? :)
	/// 
	/// Created by Scott Wilson, released into the public domain as demonstration of concept only. Use at own risk.
	/// </summary>
	public class RibbonMoveExampleApp : IExternalApplication
	{

		// System tab and panel ids to use as targets for the demo
		// To find other valid ids for system tabs and panels, simply dump
		// out the id of each item visited in the foreach loops below
		String SystemTabId = "Manage";
		String SystemPanelId = "settings_shr";		

		// Example names of API created tabs, panels and buttons
		// To try this demo on ribbon items already created by another loaded addin,
		// enter the appropriate details here and comment out everything in
		// the OnStartup method except for the ApplicationInitialized registration
		String ApiTabName = "New Tab";
		String ApiPanelName = "New Panel";
		String ApiButtonName = "NewButton";
		String ApiButtonText = "New\nButton";

		public Autodesk.Revit.UI.Result OnStartup(UIControlledApplication uiConApp)
		{

			// Add our own tab, panel and command button to the ribbon using the API
			uiConApp.CreateRibbonTab(ApiTabName);

			RibbonPanel apiRibbonPanel = uiConApp.CreateRibbonPanel(ApiTabName, ApiPanelName);

			PushButtonData buttonDataTest = new PushButtonData(ApiButtonName, ApiButtonText, Assembly.GetExecutingAssembly().Location, "RibbonMoveExample.ExampleCommand");
			buttonDataTest.AvailabilityClassName = "RibbonMoveExample.ExampleCommand_Availability";
			apiRibbonPanel.AddItem(buttonDataTest);

			// Subscribe to the "ApplicationInitialized" event then continue from there once it is fired.
			// This is to ensure that the ribbon is fully populated before we mess with it.
			uiConApp.ControlledApplication.ApplicationInitialized += OnApplicationInitialized;
			
			return Result.Succeeded;
		}


		void OnApplicationInitialized(object sender, Autodesk.Revit.DB.Events.ApplicationInitializedEventArgs e)
		{
			// Find a system ribbon tab and panel to house our API items
			// Also find our API tab, panel and button within the Autodesk.Windows.RibbonControl context

			adWin.RibbonControl adWinRibbon = adWin.ComponentManager.Ribbon;

			adWin.RibbonTab adWinSysTab = null;
			adWin.RibbonPanel adWinSysPanel = null;

			adWin.RibbonTab adWinApiTab = null;
			adWin.RibbonPanel adWinApiPanel = null;
			adWin.RibbonItem adWinApiItem = null;

			foreach(adWin.RibbonTab ribbonTab in adWinRibbon.Tabs)
			{
				// Look for the specified system tab
				if(ribbonTab.Id == SystemTabId)
				{
					adWinSysTab = ribbonTab;

					foreach(adWin.RibbonPanel ribbonPanel in ribbonTab.Panels)
					{
						// Look for the specified panel within the system tab
						if(ribbonPanel.Source.Id == SystemPanelId)
						{
							adWinSysPanel = ribbonPanel;
						}
					}
				}
				else
				{
					// Look for our API tab
					if(ribbonTab.Id == ApiTabName)
					{
						adWinApiTab = ribbonTab;

						foreach(adWin.RibbonPanel ribbonPanel in ribbonTab.Panels)
						{
							// Look for our API panel.							

							// The Source.Id property of an API created ribbon panel has the following format: CustomCtrl_%[TabName]%[PanelName]
							// Where PanelName correlates with the string entered as the name of the panel at creation
							// The Source.AutomationName property can also be used as it is also a direct correlation of the panel name, but without all the cruft
							// Be sure to include any new line characters (\n) used for the panel name at creation as they still form part of the Id & AutomationName

							//if(ribbonPanel.Source.AutomationName == ApiPanelName) // Alternative method
							if(ribbonPanel.Source.Id == "CustomCtrl_%" + ApiTabName + "%" + ApiPanelName)
							{								
								adWinApiPanel = ribbonPanel;

								foreach(adWin.RibbonItem ribbonItem in ribbonPanel.Source.Items)
								{
									// Look for our command button

									// The Id property of an API created ribbon item has the following format: CustomCtrl_%CustomCtrl_%[TabName]%[PanelName]%[ItemName]
									// Where ItemName correlates with the string entered as the first parameter (name) of the PushButtonData() constructor
									// While AutomationName correlates with the string entered as the second parameter (text) of the PushButtonData() constructor
									// Be sure to include any new line characters (\n) used for the button name and text at creation as they still form part of the ItemName & AutomationName
									
									//if(ribbonItem.AutomationName == ApiButtonText) // alternative method
									if(ribbonItem.Id == "CustomCtrl_%CustomCtrl_%" + ApiTabName + "%" + ApiPanelName + "%" + ApiButtonName)
									{
										adWinApiItem = ribbonItem;
									}
								}

							}
						}
					}
				}
			}

			// Make sure we got everything we need
			if(adWinSysTab != null && adWinSysPanel != null && adWinApiTab != null && adWinApiPanel != null && adWinApiItem != null)
			{
				// First we'll add the whole panel including the button to the system tab
				adWinSysTab.Panels.Add(adWinApiPanel);

				// now lets also add the button itself to a system panel
				adWinSysPanel.Source.Items.Add(adWinApiItem);


				// Remove panel from original API tab
				// It can also be left there if needed, there doesn't seem to be any problems with
				// duplicate panels / buttons on seperate tabs / panels respectively
				adWinApiTab.Panels.Remove(adWinApiPanel); 
				
				adWinRibbon.Tabs.Remove(adWinApiTab); // Remove our original API tab from the ribbon				

			}

			// A new panel should now be added to the specified system tab. Its command buttons will behave as they normally would, including API access and
			// ExternalCommandAvailability tests. There will also be a second copy of the command button from the panel added to the specified system panel.
			
		}




		public Result OnShutdown(UIControlledApplication application)
		{
			return Result.Succeeded;
		}

	}
}
