//
// (C) Copyright 2005 by Autodesk, Inc. 
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted, 
// provided that the above copyright notice appears in all copies and 
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting 
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC. 
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to 
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//

using System;
using System.Collections.Generic;
using System.Collections;
using System.Text;
using System.Diagnostics;

using RvtMgdDbg.Snoop.Collectors;

using Autodesk.Revit.MEP;
using Autodesk.Revit.MEP.Enums;


namespace RvtMgdDbg.Snoop.CollectorExts {

  public class CollectorExtMEP : CollectorExt {

    public
    CollectorExtMEP()
    {
    }

    protected override void
    CollectEvent(object sender, CollectorEventArgs e)
    {
        // cast the sender object to the SnoopCollector we are expecting
      Collector snoopCollector = sender as Collector;
      if (snoopCollector == null) {
        Debug.Assert(false);  // why did someone else send us the message?
        return;
      }


        // see if it is a type we are responsible for      
      
      Connector connector = e.ObjToSnoop as Connector;
      if (connector != null) {
        Stream(snoopCollector.Data(), connector);
        return;
      }

      ConnectorManager connectorMgr = e.ObjToSnoop as ConnectorManager;
      if (connectorMgr != null) {
        Stream(snoopCollector.Data(), connectorMgr);
        return;
      }

      CorrectionFactor correctionFactor = e.ObjToSnoop as CorrectionFactor;
      if (correctionFactor != null) {
        Stream(snoopCollector.Data(), correctionFactor);
        return;
      }      

      DemandFactorData demandFactorData = e.ObjToSnoop as DemandFactorData;
      if (demandFactorData != null) {
        Stream(snoopCollector.Data(), demandFactorData);
        return;
      }

      ElectricalSetting elecSetting = e.ObjToSnoop as ElectricalSetting;
      if (elecSetting != null) {
        Stream(snoopCollector.Data(), elecSetting);
        return;
      }

      GroundConductorSize groundConductorSize = e.ObjToSnoop as GroundConductorSize;
      if (groundConductorSize != null) {
        Stream(snoopCollector.Data(), groundConductorSize);
        return;
      }

      MEPModel mepModel = e.ObjToSnoop as MEPModel;
      if (mepModel != null) {
        Stream(snoopCollector.Data(), mepModel);
        return;
      }

      WireSize wireSize = e.ObjToSnoop as WireSize;
      if (wireSize != null) {
        Stream(snoopCollector.Data(), wireSize);
        return;
      }
    }
    
    private void
    Stream(ArrayList data, Connector connector) // exception handlers added for all properties by jeremy; todo: replace them by if statement checks
    {
      data.Add(new Snoop.Data.ClassSeparator(typeof(Connector)));

      try
      {
        data.Add( new Snoop.Data.Enumerable( "All refs", connector.AllRefs ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "AllRefs: " + ex.Message );
      }
      if( Domain.DomainHvac == connector.Domain 
        || Domain.DomainPiping == connector.Domain ) // Domain.DomainElectrical != connector.Domain // added by jeremy
      {
        data.Add( new Snoop.Data.Double( "Angle", connector.Angle ) );
      }
      //if( Domain.DomainPiping == connector.Domain ) // added by jeremy
      try
      {
        data.Add( new Snoop.Data.String( "Duct flow configuration type", connector.AssignedDuctFlowConfiguration.ToString() ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "AssignedDuctFlowConfiguration: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.String( "Duct loss method type", connector.AssignedDuctLossMethod.ToString() ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "AssignedDuctLossMethod: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Double( "Assigned fixture units", connector.AssignedFixtureUnits ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "AssignedFixtureUnits: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Double( "Assigned flow", connector.AssignedFlow ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "AssignedFlow: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.String( "Assigned flow direction", connector.AssignedFlowDirection.ToString() ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "AssignedFlowDirection: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Double( "Assigned flow factor", connector.AssignedFlowFactor ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "AssignedFlowFactor: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Double( "Assigned K coefficient", connector.AssignedKCoefficient ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "AssignedKCoefficient: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Double( "Assigned loss coefficient", connector.AssignedLossCoefficient ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "AssignedLossCoefficient: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.String( "Assigned pipe flow configuration", connector.AssignedPipeFlowConfiguration.ToString() ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "AssignedPipeFlowConfiguration: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.String( "Assigned pipe loss method", connector.AssignedPipeLossMethod.ToString() ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "AssignedPipeLossMethod: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Double( "Assigned pressure drop", connector.AssignedPressureDrop ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "AssignedPressureDrop: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Double( "Coefficient", connector.Coefficient ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "Coefficient: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Object( "Connector manager", connector.ConnectorManager ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "ConnectorManager: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.String( "Connector type", connector.ConnectorType.ToString() ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "ConnectorType: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Object( "Coordinate system", connector.CoordinateSystem ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "CoordinateSystem: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Double( "Demand", connector.Demand ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "Demand: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.String( "Direction", connector.Direction.ToString() ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "Direction: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.String( "Domain", connector.Domain.ToString() ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "Domain: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.String( "Duct system type", connector.DuctSystemType.ToString() ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "DuctSystemType: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.String( "Electrical system type", connector.ElectricalSystemType.ToString() ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "ElectricalSystemType: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Double( "Flow", connector.Flow ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "Flow: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Double( "Height", connector.Height ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "Height: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Bool( "Is connected", connector.IsConnected ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "IsConnected: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Bool( "Is movable", connector.IsMovable ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "IsMovable: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Object( "MEP system", connector.MEPSystem ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "MEPSystem: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Xyz( "Origin", connector.Origin ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "Origin: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.String( "Pipe system type", connector.PipeSystemType.ToString() ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "PipeSystemType: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Double( "Pressure drop", connector.PressureDrop ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "PressureDrop: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Double( "Velocity pressure", connector.VelocityPressure ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "VelocityPressure: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Double( "Width", connector.Width ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "Width: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Double( "Radius", connector.Radius ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "Radius: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.String( "Shape", connector.Shape.ToString() ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "Shape: " + ex.Message );
      }
      try
      {
        data.Add( new Snoop.Data.Object( "Owner", connector.Owner ) );
      }
      catch( System.NotSupportedException ex )
      {
        Debug.Print( "Owner: " + ex.Message );
      }
    }

    private void
    Stream(ArrayList data, ConnectorManager connectorMgr)
    {
      data.Add(new Snoop.Data.ClassSeparator(typeof(ConnectorManager)));

      data.Add(new Snoop.Data.Enumerable("Connectors", connectorMgr.Connectors));
      data.Add(new Snoop.Data.Enumerable("Unused connectors", connectorMgr.UnusedConnectors));
      data.Add(new Snoop.Data.Object("Owner", connectorMgr.Owner));
    }

    private void
    Stream(ArrayList data, CorrectionFactor correctionFactor)
    {
      data.Add(new Snoop.Data.ClassSeparator(typeof(CorrectionFactor)));

      data.Add(new Snoop.Data.Double("Factor", correctionFactor.Factor));
      data.Add(new Snoop.Data.Double("Temperature", correctionFactor.Temperature));
    }    

    private void
    Stream(ArrayList data, DemandFactorData demandFactorData)
    {
      data.Add(new Snoop.Data.ClassSeparator(typeof(DemandFactorData)));

      data.Add(new Snoop.Data.Bool("Is valid", demandFactorData.IsValid));
      data.Add(new Snoop.Data.Double("Max factor", demandFactorData.MaxFactor));
      data.Add(new Snoop.Data.Double("Min factor", demandFactorData.MinFactor));
      data.Add(new Snoop.Data.Double("Percentage", demandFactorData.Percentage));
    }

    private void
    Stream(ArrayList data, MEPModel mepModel)
    {
      data.Add(new Snoop.Data.ClassSeparator(typeof(MEPModel)));

      data.Add(new Snoop.Data.Object("Connector manager", mepModel.ConnectorManager));
      data.Add(new Snoop.Data.Enumerable("Electrical systems", mepModel.ElectricalSystems));

      ElectricalEquipment elecEquip = mepModel as ElectricalEquipment;
      if (elecEquip != null) {
        Stream(data, elecEquip);
        return;
      }

      LightingDevice lightDevice = mepModel as LightingDevice;
      if (lightDevice != null) {
        Stream(data, lightDevice);
        return;
      }

      LightingFixture lightFixture = mepModel as LightingFixture;
      if (lightFixture != null) {
        Stream(data, lightFixture);
        return;
      }

      MechanicalEquipment mechEquip = mepModel as MechanicalEquipment;
      if (mechEquip != null) {
        Stream(data, mechEquip);
        return;
      }

      MechanicalFitting mechFitting = mepModel as MechanicalFitting;
      if (mechFitting != null) {
        Stream(data, mechFitting);
        return;
      }
    }

    private void
    Stream(ArrayList data, ElectricalEquipment elecEquip)
    {
      data.Add(new Snoop.Data.ClassSeparator(typeof(ElectricalEquipment)));

      data.Add(new Snoop.Data.Object("Distribution system", elecEquip.DistributionSystem));
    }

    private void
    Stream(ArrayList data, LightingDevice lightDevice)
    {
      data.Add(new Snoop.Data.ClassSeparator(typeof(LightingDevice)));

        // no data at this level
    }

    private void
    Stream(ArrayList data, LightingFixture lightFixture)
    {
      data.Add(new Snoop.Data.ClassSeparator(typeof(LightingFixture)));

        // no data at this level
    }

    private void
    Stream(ArrayList data, MechanicalEquipment mechEquip)
    {
      data.Add(new Snoop.Data.ClassSeparator(typeof(MechanicalEquipment)));

        // no data at this level
    }

    private void
    Stream(ArrayList data, MechanicalFitting mechFitting)
    {
      data.Add(new Snoop.Data.ClassSeparator(typeof(MechanicalFitting)));

      data.Add(new Snoop.Data.String("Part type", mechFitting.PartType.ToString()));
    }

    private void
    Stream(ArrayList data, ElectricalSetting elecSetting)
    {
      data.Add(new Snoop.Data.ClassSeparator(typeof(ElectricalSetting)));

      data.Add(new Snoop.Data.Enumerable("Conduit types", elecSetting.ConduitTypes));
      data.Add(new Snoop.Data.Enumerable("Distribution system types", elecSetting.DistributionSysTypes));
      data.Add(new Snoop.Data.Enumerable("Voltage types", elecSetting.VoltageTypes));
      data.Add(new Snoop.Data.Enumerable("Wire material types", elecSetting.WireMaterialTypes));
      data.Add(new Snoop.Data.Enumerable("Wire types", elecSetting.WireTypes));

      data.Add(new Snoop.Data.CategorySeparator("Demand Factors"));
      foreach (LoadClassification lc in System.Enum.GetValues(typeof(LoadClassification))) {
        data.Add(new Snoop.Data.Object(string.Format("Demand Factor ({0})", lc.ToString()), elecSetting.GetDemandFactor(lc)));
      }
    }

    private void
    Stream(ArrayList data, GroundConductorSize grndCondSize)
    {
      data.Add(new Snoop.Data.ClassSeparator(typeof(GroundConductorSize)));

      data.Add(new Snoop.Data.Double("Ampacity", grndCondSize.Ampacity));
      data.Add(new Snoop.Data.String("Conductor size", grndCondSize.ConductorSize));
      data.Add(new Snoop.Data.Object("Material belong to", grndCondSize.MaterialBelongTo));
    }

    private void
    Stream(ArrayList data, WireSize wireSize)
    {
      data.Add(new Snoop.Data.ClassSeparator(typeof(WireSize)));

      data.Add(new Snoop.Data.Double("Ampacity", wireSize.Ampacity));
      data.Add(new Snoop.Data.Double("Diameter", wireSize.Diameter));
      data.Add(new Snoop.Data.Bool("In use", wireSize.InUse));
      data.Add(new Snoop.Data.String("Size", wireSize.Size));
    }
  }
}
