//
// (C) Copyright 2005 by Autodesk, Inc. 
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted, 
// provided that the above copyright notice appears in all copies and 
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting 
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC. 
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to 
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//

using System;
using System.Collections;
using System.Diagnostics;

using Autodesk.Revit;
using Autodesk.Revit.Elements;
using Autodesk.Revit.Events;
using Autodesk.Revit.Spaces;

using RvtMgdDbg.Snoop.Collectors;

namespace RvtMgdDbg.Snoop.CollectorExts {  

    public class CollectorExtMisc : CollectorExt
    {
        public
        CollectorExtMisc()
        {
        }

        protected override void
        CollectEvent(object sender, CollectorEventArgs e)
        {
                // cast the sender object to the SnoopCollector we are expecting
            Collector snoopCollector = sender as Collector;
            if (snoopCollector == null) {
                Debug.Assert(false);    // why did someone else send us the message?
                return;
            }

                // see if it is a type we are responsible for
            Autodesk.Revit.Color color = e.ObjToSnoop as Autodesk.Revit.Color;
            if (color != null) {
                Stream(snoopCollector.Data(), color);
                return;
            }

            LayoutRule layoutRule = e.ObjToSnoop as LayoutRule;
            if (layoutRule != null) {
                Stream(snoopCollector.Data(), layoutRule);
                return;
            }

            FormatOptions formatOptions = e.ObjToSnoop as FormatOptions;
            if (formatOptions != null) {
                Stream(snoopCollector.Data(), formatOptions);
                return;
            }

            CurtainGrid curtainGrid = e.ObjToSnoop as CurtainGrid;
            if (curtainGrid != null) {
                Stream(snoopCollector.Data(), curtainGrid);
                return;
            }

            CurtainCell curtainCell = e.ObjToSnoop as CurtainCell;
            if (curtainCell != null) {
                Stream(snoopCollector.Data(), curtainCell);
                return;
            }                        

            RebarHostData rebarHostData = e.ObjToSnoop as RebarHostData;
            if (rebarHostData != null) {
                Stream(snoopCollector.Data(), rebarHostData);
                return;
            }

            Leader leader = e.ObjToSnoop as Leader;
            if (leader != null) {
                Stream(snoopCollector.Data(), leader);
                return;
            }                        

            PostedError postErr = e.ObjToSnoop as PostedError;
            if (postErr != null) {
                Stream(snoopCollector.Data(), postErr);
                return;
            }

            VolumeCalculationSetting volCalc = e.ObjToSnoop as VolumeCalculationSetting;
            if (volCalc != null) {
                Stream(snoopCollector.Data(), volCalc);
                return;
            }

            VolumeCalculationOptions volCalcOpts = e.ObjToSnoop as VolumeCalculationOptions;
            if (volCalcOpts != null) {
                Stream(snoopCollector.Data(), volCalcOpts);
                return;
            }

            ViewSheetSetting viewSheetSetting = e.ObjToSnoop as ViewSheetSetting;
            if (viewSheetSetting != null) {
                Stream(snoopCollector.Data(), viewSheetSetting);
                return;
            }

            DialogBoxData dlgBoxData = e.ObjToSnoop as DialogBoxData;
            if (dlgBoxData != null) {
                Stream(snoopCollector.Data(), dlgBoxData);
                return;
            }           

            Construction construct = e.ObjToSnoop as Construction;
            if (construct != null) {
                Stream(snoopCollector.Data(), construct);
                return;
            }

            FamilyElementVisibility famElemVisib = e.ObjToSnoop as FamilyElementVisibility;
            if (famElemVisib != null) {
                Stream(snoopCollector.Data(), famElemVisib);
                return;
            }

            FamilyManager famManager = e.ObjToSnoop as FamilyManager;
            if (famManager != null) {
                Stream(snoopCollector.Data(), famManager);
                return;
            }

            FamilyParameter famParam = e.ObjToSnoop as FamilyParameter;
            if (famParam != null) {
                Stream(snoopCollector.Data(), famParam);
                return;
            }

            FamilyType famType = e.ObjToSnoop as FamilyType;
            if (famType != null) {
                Stream(snoopCollector.Data(), famType);
                return;
            }

            MEPSpaceConstruction mepSpaceConstuct = e.ObjToSnoop as MEPSpaceConstruction;
            if (mepSpaceConstuct != null) {
                Stream(snoopCollector.Data(), mepSpaceConstuct);
                return;
            }

            ParameterStorage paramStorage = e.ObjToSnoop as ParameterStorage;
            if (paramStorage != null) {
                Stream(snoopCollector.Data(), paramStorage);
                return;
            }

            BuildingSiteExportOptions bldSiteExpOptions = e.ObjToSnoop as BuildingSiteExportOptions;
            if (bldSiteExpOptions != null) {
                Stream(snoopCollector.Data(), bldSiteExpOptions);
                return;
            }

            DGNExportOptions dgnExpOptions = e.ObjToSnoop as DGNExportOptions;
            if (dgnExpOptions != null) {
                Stream(snoopCollector.Data(), dgnExpOptions);
                return;
            }

            DWFExportOptions dwfExpOptions = e.ObjToSnoop as DWFExportOptions;
            if (dwfExpOptions != null) {
                Stream(snoopCollector.Data(), dwfExpOptions);
                return;
            }

            DWGExportOptions dwgExpOptions = e.ObjToSnoop as DWGExportOptions;
            if (dwgExpOptions != null) {
                Stream(snoopCollector.Data(), dwgExpOptions);
                return;
            }

            DWGImportOptions dwgImpOptions = e.ObjToSnoop as DWGImportOptions;
            if (dwgImpOptions != null) {
                Stream(snoopCollector.Data(), dwgImpOptions);
                return;
            }

            FBXExportOptions fbxExpOptions = e.ObjToSnoop as FBXExportOptions;
            if (fbxExpOptions != null) {
                Stream(snoopCollector.Data(), fbxExpOptions);
                return;
            }

            TrussMemberInfo trussInfo = e.ObjToSnoop as TrussMemberInfo;
            if (trussInfo != null) {
                Stream(snoopCollector.Data(), trussInfo);
                return;
            }

            VertexIndexPair vertIndPair = e.ObjToSnoop as VertexIndexPair;
            if (vertIndPair != null) {
                Stream(snoopCollector.Data(), vertIndPair);
                return;
            }

            PointElementReference ptElemRef = e.ObjToSnoop as PointElementReference;
            if (ptElemRef != null) {
                Stream(snoopCollector.Data(), ptElemRef);
                return;
            }

            BoundarySegment boundSeg = e.ObjToSnoop as BoundarySegment;
            if (boundSeg != null) {
                Stream(snoopCollector.Data(), boundSeg);
                return;
            }

            SpacingRule spacingRule = e.ObjToSnoop as SpacingRule;
            if (spacingRule != null) {
                Stream(snoopCollector.Data(), spacingRule);
                return;
            }

            ButtonData buttonData = e.ObjToSnoop as ButtonData;
            if (buttonData != null) {
                Stream(snoopCollector.Data(), buttonData);
                return;
            }

            TilePatterns tilePatterns = e.ObjToSnoop as TilePatterns;
            if (tilePatterns != null) {
                Stream(snoopCollector.Data(), tilePatterns);
                return;
            }

            DividedSurfaceData divSurfData = e.ObjToSnoop as DividedSurfaceData;
            if (divSurfData != null) {
                Stream(snoopCollector.Data(), divSurfData);
                return;
            }

            Autodesk.Revit.Options.Application app = e.ObjToSnoop as Autodesk.Revit.Options.Application;
            if (app != null) {
                Stream(snoopCollector.Data(), app);
                return;
            }
        }

        private void
        Stream(ArrayList data, Autodesk.Revit.Color color)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(Autodesk.Revit.Color)));

            data.Add(new Snoop.Data.Int("Red", color.Red));
            data.Add(new Snoop.Data.Int("Green", color.Green));
            data.Add(new Snoop.Data.Int("Blue", color.Blue));
        }

        private void
        Stream(ArrayList data, LayoutRule layoutRule)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(LayoutRule)));

            // no data at this level

            LayoutRuleClearSpacing lrClearSpacing = layoutRule as LayoutRuleClearSpacing;
            if (lrClearSpacing != null) {
                Stream(data, lrClearSpacing);
                return;
            }

            LayoutRuleFixedDistance lrFixedDist = layoutRule as LayoutRuleFixedDistance;
            if (lrFixedDist != null) {
                Stream(data, lrFixedDist);
                return;
            }

            LayoutRuleFixedNumber lrFixedNum = layoutRule as LayoutRuleFixedNumber;
            if (lrFixedNum != null) {
                Stream(data, lrFixedNum);
                return;
            }

            LayoutRuleMaximumSpacing lrMaxSpacing = layoutRule as LayoutRuleMaximumSpacing;
            if (lrMaxSpacing != null) {
                Stream(data, lrMaxSpacing);
                return;
            }
        }

        private void
        Stream(ArrayList data, LayoutRuleClearSpacing layoutRule)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(LayoutRuleClearSpacing)));

            data.Add(new Snoop.Data.String("Justify type", layoutRule.JustifyType.ToString()));
            data.Add(new Snoop.Data.Double("Spacing", layoutRule.Spacing));
        }

        private void
        Stream(ArrayList data, LayoutRuleFixedDistance layoutRule)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(LayoutRuleFixedDistance)));

            data.Add(new Snoop.Data.String("Justify type", layoutRule.JustifyType.ToString()));
            data.Add(new Snoop.Data.Double("Spacing", layoutRule.Spacing));
        }

        private void
        Stream(ArrayList data, LayoutRuleFixedNumber layoutRule)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(LayoutRuleFixedNumber)));

            data.Add(new Snoop.Data.Int("Number of lines", layoutRule.NumberOfLines));
        }

        private void
        Stream(ArrayList data, LayoutRuleMaximumSpacing layoutRule)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(LayoutRuleMaximumSpacing)));

            data.Add(new Snoop.Data.Double("Spacing", layoutRule.Spacing));
        }

        private void
        Stream(ArrayList data, FormatOptions opt)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(FormatOptions)));

            data.Add(new Snoop.Data.String("Name", opt.Name));
            data.Add(new Snoop.Data.String("Unit symbol", opt.Unitsymbol.ToString()));
            data.Add(new Snoop.Data.String("Units", opt.Units.ToString()));
            data.Add(new Snoop.Data.Object("Group", opt.Group));     // TBD: strangely named property
            data.Add(new Snoop.Data.Double("Rounding", opt.Rounding));
        }

        private void
        Stream(ArrayList data, CurtainGrid grid)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(CurtainGrid)));

            data.Add(new Snoop.Data.Enumerable("Cells", grid.Cells));
            data.Add(new Snoop.Data.Enumerable("Mullions", grid.Mullions));
            data.Add(new Snoop.Data.Enumerable("Unlocked mullions", grid.UnlockedMullions));
            data.Add(new Snoop.Data.Angle("Grid 1 angle", grid.Grid1Angle));
            data.Add(new Snoop.Data.String("Grid 1 justification", grid.Grid1Justification.ToString()));
            data.Add(new Snoop.Data.Double("Grid 1 offset", grid.Grid1Offset));
            data.Add(new Snoop.Data.Angle("Grid 2 angle", grid.Grid2Angle));
            data.Add(new Snoop.Data.String("Grid 2 justification", grid.Grid2Justification.ToString()));
            data.Add(new Snoop.Data.Double("Grid 2 offset", grid.Grid2Offset));           
            data.Add(new Snoop.Data.Int("Number of panels", grid.NumPanels));
            data.Add(new Snoop.Data.Enumerable("Panels", grid.Panels));
            data.Add(new Snoop.Data.Enumerable("Unlocked panels", grid.UnlockedPanels));
            data.Add(new Snoop.Data.Int("Number of U lines", grid.NumULines));
            data.Add(new Snoop.Data.Enumerable("U grid lines", grid.UGridLines));
            data.Add(new Snoop.Data.Int("Number of V lines", grid.NumVLines));
            data.Add(new Snoop.Data.Enumerable("V grid lines", grid.VGridLines));             
        }

        private void
        Stream(ArrayList data, CurtainCell cell)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(CurtainCell)));

            data.Add(new Snoop.Data.Enumerable("Curve loops", cell.CurveLoops));
            data.Add(new Snoop.Data.Enumerable("Planarized curve loops", cell.PlanarizedCurveLoops));
        }

        private void
        Stream(ArrayList data, RebarHostData rebarHostData)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(RebarHostData)));

            data.Add(new Snoop.Data.Bool("Valid", rebarHostData.Valid));            
        }

        private void
        Stream(ArrayList data, Leader leader)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(Leader)));

            data.Add(new Snoop.Data.Xyz("Elbow", leader.Elbow));
            data.Add(new Snoop.Data.Xyz("End", leader.End));
        }                

        private void
        Stream(ArrayList data, PostedError postErr)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(PostedError)));
            
            data.Add(new Snoop.Data.String("Error type", postErr.ErrorType.ToString()));
            data.Add(new Snoop.Data.String("Message", postErr.Message));            
            
        }

        private void
        Stream(ArrayList data, VolumeCalculationSetting volCalc)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(VolumeCalculationSetting)));

            data.Add(new Snoop.Data.Object("Volume calculation options", volCalc.VolumeCalculationOptions));           
        }

        private void
        Stream(ArrayList data, VolumeCalculationOptions volCalcOpts)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(VolumeCalculationOptions)));

            data.Add(new Snoop.Data.String("Room area boundary location", volCalcOpts.RoomAreaBoundaryLocation.ToString()));
            data.Add(new Snoop.Data.Bool("Volume computation enable", volCalcOpts.VolumeComputationEnable));              
        }

        private void
        Stream(ArrayList data, ViewSheetSetting viewSheetSetting)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(ViewSheetSetting)));

            data.Add(new Snoop.Data.Object("Available views", viewSheetSetting.AvailableViews));
            data.Add(new Snoop.Data.Object("Current view sheet set", viewSheetSetting.CurrentViewSheetSet));
        }

        private void
        Stream(ArrayList data, DialogBoxData dlgBoxData)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(DialogBoxData)));

            data.Add(new Snoop.Data.Int("Help id", dlgBoxData.HelpId));           
        }

        private void
        Stream(ArrayList data, BuildingSiteExportOptions bldSiteOptions)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(BuildingSiteExportOptions)));

            data.Add(new Snoop.Data.Double("Area per person", bldSiteOptions.AreaPerPerson));
            data.Add(new Snoop.Data.Object("Property line", bldSiteOptions.PropertyLine));
            data.Add(new Snoop.Data.Double("Property line offset", bldSiteOptions.PropertyLineOffset));
            data.Add(new Snoop.Data.Double("Total gross area", bldSiteOptions.TotalGrossArea));
            data.Add(new Snoop.Data.Int("Total occupancy", bldSiteOptions.TotalOccupancy));
        }

        private void
        Stream(ArrayList data, Construction construct)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(Construction)));

            data.Add(new Snoop.Data.String("Id", construct.Id));
            data.Add(new Snoop.Data.String("Name", construct.Name));            
        }

        private void
        Stream(ArrayList data, FamilyElementVisibility famElemVisib)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(FamilyElementVisibility)));

            data.Add(new Snoop.Data.Bool("Is shown in Coarse", famElemVisib.IsShownInCoarse));
            data.Add(new Snoop.Data.Bool("Is shown in Fine", famElemVisib.IsShownInFine));
            data.Add(new Snoop.Data.Bool("Is shown in FrontBack", famElemVisib.IsShownInFrontBack));
            data.Add(new Snoop.Data.Bool("Is shown in LeftRight", famElemVisib.IsShownInLeftRight));
            data.Add(new Snoop.Data.Bool("Is shown in Medium", famElemVisib.IsShownInMedium));
            data.Add(new Snoop.Data.Bool("Is shown in PlanRCPCut", famElemVisib.IsShownInPlanRCPCut));
            data.Add(new Snoop.Data.Bool("Is shown in TopBottom", famElemVisib.IsShownInTopBottom));
            data.Add(new Snoop.Data.Bool("Is shown only when cut", famElemVisib.IsShownOnlyWhenCut));
            data.Add(new Snoop.Data.String("Visibility type", famElemVisib.VisibilityType.ToString()));
        }

        private void
        Stream(ArrayList data, FamilyManager famManager)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(FamilyManager)));

            data.Add(new Snoop.Data.Object("Current type", famManager.CurrentType));
            data.Add(new Snoop.Data.Enumerable("Parameters", famManager.Parameters));
            data.Add(new Snoop.Data.Enumerable("Types", famManager.Types));
        }

        private void
        Stream(ArrayList data, FamilyParameter famParam)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(FamilyParameter)));

            data.Add(new Snoop.Data.Enumerable("Associated parameters", famParam.AssociatedParameters));
            data.Add(new Snoop.Data.Bool("Can assign formula", famParam.CanAssignFormula));
            data.Add(new Snoop.Data.Object("Definition", famParam.Definition));

            try {
                data.Add(new Snoop.Data.String("Display unit type", famParam.DisplayUnitType.ToString()));
            }
            catch (SystemException ex) {
                data.Add(new Snoop.Data.Exception("Display unit type", ex));
            }

            data.Add(new Snoop.Data.String("Formula", famParam.Formula));
            data.Add(new Snoop.Data.Bool("Is determined by formula", famParam.IsDeterminedByFormula));
            data.Add(new Snoop.Data.Bool("Is instance", famParam.IsInstance));
            data.Add(new Snoop.Data.String("Storage type", famParam.StorageType.ToString()));
        }

        private void
        Stream(ArrayList data, FamilyType famType)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(FamilyType)));

            data.Add(new Snoop.Data.String("Name", famType.Name));            
        }

        private void
        Stream(ArrayList data, MEPSpaceConstruction mepSpaceConstr)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(MEPSpaceConstruction)));

            data.Add(new Snoop.Data.Object("Current construction", mepSpaceConstr.CurrentConstruction));
            data.Add(new Snoop.Data.Enumerable("Space constructions", mepSpaceConstr.SpaceConstructions));            
        }

        private void
        Stream(ArrayList data, ParameterStorage paramStorage)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(ParameterStorage)));

            data.Add(new Snoop.Data.String("Storage type", paramStorage.StorageType.ToString()));
        }        

        private void
        Stream(ArrayList data, DGNExportOptions dgnExportOptions)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(DGNExportOptions)));

            data.Add(new Snoop.Data.String("Export of solids", dgnExportOptions.ExportOfSolids.ToString()));
            data.Add(new Snoop.Data.String("Layer mapping", dgnExportOptions.LayerMapping));
            data.Add(new Snoop.Data.String("Template file", dgnExportOptions.TemplateFile));
        }

        private void
        Stream(ArrayList data, DWFExportOptions dwfExportOptions)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(DWFExportOptions)));

            data.Add(new Snoop.Data.Bool("Exporting areas", dwfExportOptions.ExportingAreas));
            data.Add(new Snoop.Data.Bool("Export object data", dwfExportOptions.ExportObjectData));
            data.Add(new Snoop.Data.Bool("Merged views", dwfExportOptions.MergedViews));
            data.Add(new Snoop.Data.Bool("Stop on error", dwfExportOptions.StopOnError));

            DWF2DExportOptions dwf2dExpOptions = dwfExportOptions as DWF2DExportOptions;
            if (dwf2dExpOptions != null) {
                Stream(data, dwf2dExpOptions);
                return;
            }

            DWF3DExportOptions dwf3dExpOptions = dwfExportOptions as DWF3DExportOptions;
            if (dwf3dExpOptions != null) {
                Stream(data, dwf3dExpOptions);
                return;
            }
        }

        private void
        Stream(ArrayList data, DWF2DExportOptions dwf2dExpOptions)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(DWF2DExportOptions)));

            data.Add(new Snoop.Data.String("Image format", dwf2dExpOptions.ImageFormat.ToString()));
            data.Add(new Snoop.Data.String("Image quality", dwf2dExpOptions.ImageQuality.ToString()));
            data.Add(new Snoop.Data.String("Paper format", dwf2dExpOptions.PaperFormat.ToString()));
            data.Add(new Snoop.Data.Bool("Portrait layout", dwf2dExpOptions.PortraitLayout));

            DWFX2DExportOptions dwfx2dExpOptions = dwf2dExpOptions as DWFX2DExportOptions;
            if (dwfx2dExpOptions != null) {
                Stream(data, dwfx2dExpOptions);
                return;
            }
        }

        private void
        Stream(ArrayList data, DWFX2DExportOptions dwfx2dExpOptions)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(DWFX2DExportOptions)));

            // No data at this level.
        }

        private void
        Stream(ArrayList data, DWF3DExportOptions dwf3dExpOptions)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(DWF3DExportOptions)));

            // No data at this level.

            DWFX3DExportOptions dwfx3dExpOptions = dwf3dExpOptions as DWFX3DExportOptions;
            if (dwfx3dExpOptions != null) {
                Stream(data, dwfx3dExpOptions);
                return;
            }
        }

        private void
        Stream(ArrayList data, DWFX3DExportOptions dwfx3dExpOptions)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(DWFX3DExportOptions)));

            //No data at this level.
        }

        private void
        Stream(ArrayList data, DWGExportOptions dwgExpOptions)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(DWGExportOptions)));            
            
            data.Add(new Snoop.Data.Bool("Exporting areas", dwgExpOptions.ExportingAreas));
            data.Add(new Snoop.Data.String("Export of solids", dwgExpOptions.ExportOfSolids.ToString()));
            data.Add(new Snoop.Data.String("File version", dwgExpOptions.FileVersion.ToString()));
            data.Add(new Snoop.Data.Bool("Forced HLR", dwgExpOptions.ForcedHLR));
            data.Add(new Snoop.Data.String("File version", dwgExpOptions.LayerMapping));
            data.Add(new Snoop.Data.String("Line scaling", dwgExpOptions.LineScaling.ToString()));
            data.Add(new Snoop.Data.String("Prop overrides", dwgExpOptions.PropOverrides.ToString()));
            data.Add(new Snoop.Data.Bool("Shared coords", dwgExpOptions.SharedCoords));
            data.Add(new Snoop.Data.String("Target unit", dwgExpOptions.TargetUnit.ToString()));
            data.Add(new Snoop.Data.Bool("Merged views", dwgExpOptions.MergedViews));
        }

        private void
        Stream(ArrayList data, DWGImportOptions dwgImpOptions)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(DWGImportOptions)));

            // No data at this level yet.
        }

        private void
        Stream(ArrayList data, FBXExportOptions fbxExpOptions)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(FBXExportOptions)));

            data.Add(new Snoop.Data.Bool("Stop on error", fbxExpOptions.StopOnError));
        }

        private void
        Stream(ArrayList data, TrussMemberInfo trussInfo)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(TrussMemberInfo)));

            data.Add(new Snoop.Data.Int("Host truss id", trussInfo.hostTrussId.Value));
            data.Add(new Snoop.Data.Bool("Locked to truss", trussInfo.lockedToTruss));
            data.Add(new Snoop.Data.String("Member type key", trussInfo.memberTypeKey.ToString()));
        }

        private void
        Stream(ArrayList data, VertexIndexPair vertIndPair)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(VertexIndexPair)));

            data.Add(new Snoop.Data.Int("Bottom", vertIndPair.Bottom));
            data.Add(new Snoop.Data.Int("Top", vertIndPair.Top));            
        }

        private void
        Stream(ArrayList data, PointElementReference ptElemRef)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(PointElementReference)));

            PointOnEdge ptOnEdge = ptElemRef as PointOnEdge;
            if (ptOnEdge != null) {
                Stream(data, ptOnEdge);
                return;
            }

            PointOnEdgeEdgeIntersection ptOnEdgeEdgeInt = ptElemRef as PointOnEdgeEdgeIntersection;
            if (ptOnEdgeEdgeInt != null) {
                Stream(data, ptOnEdgeEdgeInt);
                return;
            }

            PointOnEdgeFaceIntersection ptOnEdgeFaceInt = ptElemRef as PointOnEdgeFaceIntersection;
            if (ptOnEdgeFaceInt != null) {
                Stream(data, ptOnEdgeFaceInt);
                return;
            }

            PointOnFace ptOnFace = ptElemRef as PointOnFace;
            if (ptOnFace != null) {
                Stream(data, ptOnFace);
                return;
            }

            PointOnPlane ptOnPlane = ptElemRef as PointOnPlane;
            if (ptOnPlane != null) {
                Stream(data, ptOnPlane);
                return;
            }
        }

        private void
        Stream(ArrayList data, PointOnEdge ptOnEdge)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(PointOnEdge)));

            data.Add(new Snoop.Data.Double("Curve parameter", ptOnEdge.CurveParameter));
            data.Add(new Snoop.Data.Object("Edge reference", ptOnEdge.GetEdgeReference()));
        }

        private void
        Stream(ArrayList data, PointOnEdgeEdgeIntersection ptOnEdgeEdgeInt)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(PointOnEdgeEdgeIntersection)));

            data.Add(new Snoop.Data.Object("Edge reference 1", ptOnEdgeEdgeInt.GetEdgeReference1()));
            data.Add(new Snoop.Data.Object("Edge reference 2", ptOnEdgeEdgeInt.GetEdgeReference2()));
        }

        private void
        Stream(ArrayList data, PointOnEdgeFaceIntersection ptOnEdgeFaceInt)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(PointOnEdgeFaceIntersection)));

            data.Add(new Snoop.Data.Object("Edge reference", ptOnEdgeFaceInt.GetEdgeReference()));
            data.Add(new Snoop.Data.Object("Face reference", ptOnEdgeFaceInt.GetFaceReference()));
            data.Add(new Snoop.Data.Bool("Orient with edge", ptOnEdgeFaceInt.OrientWithEdge));
        }

        private void
        Stream(ArrayList data, PointOnFace ptOnFace)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(PointOnFace)));

            data.Add(new Snoop.Data.Object("Face reference", ptOnFace.GetFaceReference()));
            data.Add(new Snoop.Data.Uv("UV", ptOnFace.UV));
        }

        private void
        Stream(ArrayList data, PointOnPlane ptOnPlane)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(PointOnPlane)));

            data.Add(new Snoop.Data.Double("Offset", ptOnPlane.Offset));
            data.Add(new Snoop.Data.Uv("Position", ptOnPlane.Position));
            data.Add(new Snoop.Data.Uv("Xvec", ptOnPlane.XVec));

        } 

        private void
        Stream(ArrayList data, BoundarySegment boundSeg)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(BoundarySegment)));

            data.Add(new Snoop.Data.Object("Curve", boundSeg.Curve));
            data.Add(new Snoop.Data.Object("Document", boundSeg.Document));
            data.Add(new Snoop.Data.Object("Element", boundSeg.Element));
        }

        private void
        Stream(ArrayList data, SpacingRule spacingRule)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(SpacingRule)));

            data.Add(new Snoop.Data.Double("Belt measurement", spacingRule.BeltMeasurement));
            data.Add(new Snoop.Data.Double("Distance", spacingRule.Distance));
            data.Add(new Snoop.Data.Double("Gridlines rotation", spacingRule.GridlinesRotation));
            data.Add(new Snoop.Data.Bool("Has belt measurement", spacingRule.HasBeltMeasurement));
            data.Add(new Snoop.Data.String("Justification", spacingRule.Justification.ToString()));
            data.Add(new Snoop.Data.String("Layout", spacingRule.Layout.ToString()));
            data.Add(new Snoop.Data.Int("Number", spacingRule.Number));
            data.Add(new Snoop.Data.Double("Offset", spacingRule.Offset));
        }        

        private void
        Stream(ArrayList data, ButtonData buttonData)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(ButtonData)));

            data.Add(new Snoop.Data.Double("Image height", buttonData.Image.Height));
            data.Add(new Snoop.Data.Double("Image width", buttonData.Image.Width));
            data.Add(new Snoop.Data.String("Name", buttonData.Name));
            data.Add(new Snoop.Data.String("Text", buttonData.Text));
            data.Add(new Snoop.Data.String("ToolTip", buttonData.ToolTip));

            PushButtonData pushButtonData = buttonData as PushButtonData;
            if (pushButtonData != null) {
                Stream(data, pushButtonData);
                return;
            }

            PulldownButtonData pullDownButtonData = buttonData as PulldownButtonData;
            if (pullDownButtonData != null) {
                Stream(data, pullDownButtonData);
                return;
            }
        }

        private void
        Stream(ArrayList data, PushButtonData pushButtonData)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(PushButtonData)));

            data.Add(new Snoop.Data.String("Assembly name", pushButtonData.AssemblyName));
            data.Add(new Snoop.Data.String("Class name", pushButtonData.ClassName));           
        }

        private void
        Stream(ArrayList data, PulldownButtonData pullDownButtonData)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(PulldownButtonData)));

           // No data at this level yet!
        }

        private void
        Stream(ArrayList data, GenericImportOptions genImpOptions)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(GenericImportOptions)));

            data.Add(new Snoop.Data.Xyz("Ref point", genImpOptions.RefPoint));
            data.Add(new Snoop.Data.Object("View", genImpOptions.View));

            ImageImportOptions imgImpOptions = genImpOptions as ImageImportOptions;
            if (imgImpOptions != null) {
                Stream(data, imgImpOptions);
                return;
            }            
        }

        private void
        Stream(ArrayList data, ImageImportOptions imgImpOptions)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(ImageImportOptions)));

            data.Add(new Snoop.Data.String("Placement", imgImpOptions.Placement.ToString()));
            data.Add(new Snoop.Data.Int("Resolution", imgImpOptions.Resolution));           
        }

        private void
        Stream(ArrayList data, TilePatterns tilePatterns)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(TilePatterns)));

            // No data at this level yet!!   
        }

        private void
        Stream(ArrayList data, DividedSurfaceData divSurfData)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(DividedSurfaceData)));

            data.Add(new Snoop.Data.Enumerable("References with divided surfaces", divSurfData.GetReferencesWithDividedSurfaces())); 
        }

        private void
        Stream(ArrayList data, Autodesk.Revit.Options.Application app)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(Autodesk.Revit.Options.Application)));

            data.Add(new Snoop.Data.Enumerable("Library paths", app.LibraryPaths));
            data.Add(new Snoop.Data.String("Shared params file name", app.SharedParametersFilename));
        }
    }
}
