
//
// (C) Copyright 2005 by Autodesk, Inc. 
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted, 
// provided that the above copyright notice appears in all copies and 
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting 
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC. 
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to 
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//

using System;
using System.Collections;
using System.Diagnostics;

using Autodesk.Revit;
using Autodesk.Revit.Elements;
using Autodesk.Revit.Structural;

using RvtMgdDbg.Snoop.Collectors;

namespace RvtMgdDbg.Snoop.CollectorExts
{
	/// <summary>
	/// Provide Snoop.Data for any classes related to Structural.
	/// </summary>

	public class CollectorExtStructural : CollectorExt
	{
		public
		CollectorExtStructural()
		{
		}

        protected override void
        CollectEvent(object sender, CollectorEventArgs e)
        {
                // cast the sender object to the SnoopCollector we are expecting
            Collector snoopCollector = sender as Collector;
            if (snoopCollector == null) {
                Debug.Assert(false);    // why did someone else send us the message?
                return;
            }

                // see if it is a type we are responsible for
			AnalyticalModel aModel = e.ObjToSnoop as AnalyticalModel;
			if (aModel != null) {
				Stream(snoopCollector.Data(), aModel);
				return;
			}

			CompoundStructure compStruct = e.ObjToSnoop as CompoundStructure;
			if (compStruct != null) {
				Stream(snoopCollector.Data(), compStruct);
				return;
			}

			CompoundStructureLayer compStructLayer = e.ObjToSnoop as CompoundStructureLayer;
			if (compStructLayer != null) {
				Stream(snoopCollector.Data(), compStructLayer);
				return;
			}

            AnalyticalSupportData supportData = e.ObjToSnoop as AnalyticalSupportData;
            if (supportData != null) {
                Stream(snoopCollector.Data(), supportData);
                return;
            }

            AnalyticalSupportInfo supportInfo = e.ObjToSnoop as AnalyticalSupportInfo;
            if (supportInfo != null) {
                Stream(snoopCollector.Data(), supportInfo);
                return;
            }

            BarDescription barDesc = e.ObjToSnoop as BarDescription;
            if (barDesc != null) {
                Stream(snoopCollector.Data(), barDesc);
                return;
            }
        }
        
		private void
		Stream(ArrayList data, AnalyticalModel aModel)
		{
		    data.Add(new Snoop.Data.ClassSeparator(typeof(AnalyticalModel)));

				// Nothing at this level yet!

			AnalyticalModel3D aModel3D = aModel as AnalyticalModel3D;
			if (aModel3D != null) {
				Stream(data, aModel3D);
				return;
			}

			AnalyticalModelFloor aModelFloor = aModel as AnalyticalModelFloor;
			if (aModelFloor != null) {
				Stream(data, aModelFloor);
				return;
			}

			AnalyticalModelFrame aModelFrame = aModel as AnalyticalModelFrame;
			if (aModelFrame != null) {
				Stream(data, aModelFrame);
				return;
			}

			AnalyticalModelLocation aModelLoc = aModel as AnalyticalModelLocation;
			if (aModelLoc != null) {
				Stream(data, aModelLoc);
				return;
			}

			AnalyticalModelProfile aModelProf = aModel as AnalyticalModelProfile;
			if (aModelProf != null) {
				Stream(data, aModelProf);
				return;
			}

			AnalyticalModelWall aModelWall = aModel as AnalyticalModelWall;
			if (aModelWall != null) {
				Stream(data, aModelWall);
				return;
			}
        }

		private void
		Stream(ArrayList data, AnalyticalModel3D aModel3D)
		{
		    data.Add(new Snoop.Data.ClassSeparator(typeof(AnalyticalModel3D)));

            data.Add(new Snoop.Data.Enumerable("Curves", aModel3D.Curves));
            data.Add(new Snoop.Data.Object("Support data", aModel3D.SupportData));
        }

		private void
		Stream(ArrayList data, AnalyticalModelFloor aModelFloor)
		{
		    data.Add(new Snoop.Data.ClassSeparator(typeof(AnalyticalModelFloor)));

            data.Add(new Snoop.Data.Enumerable("Curves", aModelFloor.Curves));
            data.Add(new Snoop.Data.Object("Support data", aModelFloor.SupportData));
        }

		private void
		Stream(ArrayList data, AnalyticalModelFrame aModelFrame)
		{
		    data.Add(new Snoop.Data.ClassSeparator(typeof(AnalyticalModelFrame)));

            data.Add(new Snoop.Data.Object("Curve", aModelFrame.Curve));
            data.Add(new Snoop.Data.Enumerable("Curves", aModelFrame.Curves));
            data.Add(new Snoop.Data.Object("Profile", aModelFrame.Profile));
            data.Add(new Snoop.Data.Object("Support data", aModelFrame.SupportData));

            // TBD: get_RigidLink(int); ???
        }

		private void
		Stream(ArrayList data, AnalyticalModelLocation aModelLoc)
		{
		    data.Add(new Snoop.Data.ClassSeparator(typeof(AnalyticalModelLocation)));

            data.Add(new Snoop.Data.Enumerable("Curves", aModelLoc.Curves));
            data.Add(new Snoop.Data.Xyz("Point", aModelLoc.Point));
            data.Add(new Snoop.Data.Object("Support data", aModelLoc.SupportData));
        }

		private void
		Stream(ArrayList data, AnalyticalModelProfile aModelProf)
		{
		    data.Add(new Snoop.Data.ClassSeparator(typeof(AnalyticalModelProfile)));

            data.Add(new Snoop.Data.Object("Driving curve", aModelProf.DrivingCurve));
            data.Add(new Snoop.Data.Object("Swept profile", aModelProf.SweptProfile));
            data.Add(new Snoop.Data.Double("Start setback", aModelProf.StartSetBack));
            data.Add(new Snoop.Data.Double("End setback", aModelProf.EndSetBack));
        }

		private void
		Stream(ArrayList data, AnalyticalModelWall aModelWall)
		{
		    data.Add(new Snoop.Data.ClassSeparator(typeof(AnalyticalModelWall)));

            data.Add(new Snoop.Data.Enumerable("Curves", aModelWall.Curves));
            data.Add(new Snoop.Data.Object("Support data", aModelWall.SupportData));
        }

		private void
		Stream(ArrayList data, CompoundStructure compStruct)
		{
		    data.Add(new Snoop.Data.ClassSeparator(typeof(CompoundStructure)));

            data.Add(new Snoop.Data.Enumerable("Layers", compStruct.Layers));
        }

		private void
		Stream(ArrayList data, CompoundStructureLayer compStructLayer)
		{
		    data.Add(new Snoop.Data.ClassSeparator(typeof(CompoundStructureLayer)));

            data.Add(new Snoop.Data.String("Function", compStructLayer.Function.ToString()));
            data.Add(new Snoop.Data.Object("Material", compStructLayer.Material));
            data.Add(new Snoop.Data.Double("Thickness", compStructLayer.Thickness));
            data.Add(new Snoop.Data.Object("Deck profile", compStructLayer.DeckProfile));
            data.Add(new Snoop.Data.String("Deck usage", compStructLayer.DeckUsage.ToString()));
            data.Add(new Snoop.Data.Bool("Variable", compStructLayer.Variable));
        }

        private void
        Stream(ArrayList data, AnalyticalSupportData supportData)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(AnalyticalSupportData)));

            data.Add(new Snoop.Data.Enumerable("Info array", supportData.InfoArray));
            data.Add(new Snoop.Data.Bool("Supported", supportData.Supported));
        }

        private void
        Stream(ArrayList data, AnalyticalSupportInfo supportInfo)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(AnalyticalSupportInfo)));

            data.Add(new Snoop.Data.Object("Element", supportInfo.Element));
            data.Add(new Snoop.Data.String("Support type", supportInfo.SupportType.ToString()));
        }

        private void
        Stream(ArrayList data, BarDescription barDesc)
        {
            data.Add(new Snoop.Data.ClassSeparator(typeof(BarDescription)));

            data.Add(new Snoop.Data.Object("Bar type", barDesc.BarType));
            data.Add(new Snoop.Data.Bool("Hooks in same direction", barDesc.HooksInSameDirection));
            data.Add(new Snoop.Data.Int("Layer", barDesc.Layer));
            data.Add(new Snoop.Data.Double("Length", barDesc.Length));

            data.Add(new Snoop.Data.CategorySeparator("Hook Types"));
            data.Add(new Snoop.Data.Int("Count", barDesc.Count));
            for (int i=0; i<barDesc.Count; i++) {
                data.Add(new Snoop.Data.Object(string.Format("Hook type [{0:d}]", i), barDesc.get_HookType(i)));
            }
        }
	}
}
