//
// (C) Copyright 1994-2006 by Autodesk, Inc. All rights reserved.
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted
// provided that the above copyright notice appears in all copies and
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting
// documentation.

//
// AUTODESK PROVIDES THIS PROGRAM 'AS IS' AND WITH ALL ITS FAULTS.
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE. AUTODESK, INC.
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
 
using System;
using System.Data;
using System.Collections.Generic;
using System.Text;

using Autodesk.Revit;
using Autodesk.Revit.Elements;
using Autodesk.Revit.Structural;
using Autodesk.Revit.Symbols;

namespace RvtMgdDbg.Test.SDKSamples.AnalyticalSupportData {

    /// <summary>
    /// Class to cache info about Analytical Support Data
    /// </summary>
    
    public class Info {

        DataTable m_elementInformation = null;  // store all required information
        
        public
        Info(ElementSet selectedElements)
        {
            m_elementInformation = StoreInformationInDataTable(selectedElements);
        }

        /// <summary>
        /// property to get private member variable m_elementInformation.
        /// </summary>
        
        public DataTable
        ElementInformation
        {
            get { return m_elementInformation; }
        }

        /// <summary>
        /// get all the required information of selected elements and store them in a data table
        /// </summary>
        /// <param name="selectedElements">
        /// all selected elements in Revit main program
        /// </param>
        /// <returns>
        /// a data table which store all the required information
        /// </returns>
        
        private DataTable
        StoreInformationInDataTable(ElementSet selectedElements)
        {
            DataTable informationTable = CreateDataTable();
            informationTable.BeginLoadData();
            
            string typeName = "";
            string infoStr1 = "";
            string infoStr2 = "";
            Autodesk.Revit.Structural.AnalyticalSupportData supportData;               
            bool getInformationflag;
            
            foreach (Element element in selectedElements) {
                typeName = string.Empty;
                supportData = null;
                getInformationflag = false;

                if (element is ContFooting) {
                    ContFooting wallFoot  = element as ContFooting;  
                    Symbol wallFootSymbol = wallFoot.ObjectType;// get element Type
                    typeName              = wallFootSymbol.Category.Name + ": " + wallFootSymbol.Name;

                        // the tyoe of wall foundation's AnalyticalModel is AnalyticalModel3D 
                    AnalyticalModel3D foundationModel = wallFoot.AnalyticalModel as AnalyticalModel3D;
                    if (foundationModel != null) {
                        supportData        = foundationModel.SupportData;
                        getInformationflag = true;
                    }
                }
                else if (element is FamilyInstance) {
                    FamilyInstance familyInstance = element as FamilyInstance;                  
                    FamilySymbol symbol           = familyInstance.ObjectType as FamilySymbol;
                    typeName                      = symbol.Family.Name + ": " + symbol.Name;

                        // the type of FamilyInstance's AnalyticalModel is AnalyticalModelFrame(eg, beam).
                    AnalyticalModelFrame frameModel = familyInstance.AnalyticalModel as AnalyticalModelFrame;
                    if (frameModel != null) {
                        supportData        = frameModel.SupportData;
                        getInformationflag = true;
                    }
                }
                else if (element is Floor) {
                    Floor slab         = element as Floor;                    
                    FloorType slabType = slab.ObjectType as FloorType; // get element type
                    typeName           = slabType.Category.Name + ": " + slabType.Name;

                        // the type of Floor's AnalyticalModel is AnalyticalModelFloor
                    AnalyticalModelFloor slabModel = slab.AnalyticalModel as AnalyticalModelFloor;
                    if (slabModel != null) {
                        supportData        = slabModel.SupportData;
                        getInformationflag = true;
                    }
                }
                else if (element is Wall) {          
                    Wall wall         = element as Wall;  
                    WallType wallType = wall.ObjectType as WallType; // get element type
                    typeName          = wallType.Kind.ToString() + ": " + wallType.Name;

                        // the type of wall's AnalyticalModel is AnalyticalModelWall
                    AnalyticalModelWall wallModel = wall.AnalyticalModel as AnalyticalModelWall;
                    if (wallModel != null) {
                        supportData        = wallModel.SupportData;
                        getInformationflag = true;
                    }
                }
                
                    // get support information (if we found support data)
                if (getInformationflag) {
                    if (supportData != null) {
                        GetSupportInformation(supportData, ref infoStr1, ref infoStr2);
                    }
                    
                        // Add a new row with the information
                    DataRow newRow = informationTable.NewRow(); 
                    newRow["Id"] = element.Id.Value.ToString();
                    newRow["Element Type"] = typeName;
                    newRow["Support Type"] = infoStr1;
                    newRow["Remark"] = infoStr2;
                    informationTable.Rows.Add(newRow);
                }
            }

            informationTable.EndLoadData();
            return informationTable;
        }

        /// <summary>
        /// Create an empty dataTable
        /// </summary>
        /// <returns></returns>
        
        private DataTable
        CreateDataTable()
        {
            // Create a new DataTable.
			DataTable elementInformationTable = new DataTable("ElementInformationTable");

            // Create element id column and add to the DataTable.
            DataColumn idColumn = new DataColumn();
            idColumn.DataType   = typeof(System.String);
            idColumn.ColumnName = "Id";
            idColumn.Caption    = "Id";
            idColumn.ReadOnly   = true;
            elementInformationTable.Columns.Add(idColumn);

            // Create element type column and add to the DataTable.
            DataColumn typeColumn = new DataColumn();
            typeColumn.DataType   = typeof(System.String);
            typeColumn.ColumnName = "Element Type";
            typeColumn.Caption    = "Element Type";
            typeColumn.ReadOnly   = true;
            elementInformationTable.Columns.Add(typeColumn);

            // Create support column and add to the DataTable.
            DataColumn supportColumn = new DataColumn();
            supportColumn.DataType   = typeof(System.String);
            supportColumn.ColumnName = "Support Type";
            supportColumn.Caption    = "Support Type";
            supportColumn.ReadOnly   = true;
            elementInformationTable.Columns.Add(supportColumn);

            // Create a colum which can note others information
            DataColumn remarkColumn = new DataColumn();
            remarkColumn.DataType   = typeof(System.String);
            remarkColumn.ColumnName = "Remark";
            remarkColumn.Caption    = "Remark";
            remarkColumn.ReadOnly   = true;
            elementInformationTable.Columns.Add(remarkColumn);
          

            return elementInformationTable;
        }

        /// <summary>
        /// get element's support information
        /// </summary>
        /// <param name="supportData"> element's support data</param>
        /// <returns></returns>
        private void
        GetSupportInformation(Autodesk.Revit.Structural.AnalyticalSupportData supportData, ref string str1, ref string str2)
        {
            str1 = string.Empty;
            str2 = string.Empty;
            
                // judge if supported
            if (!supportData.Supported) {
                str1 = "not supported";
            }
            else {
                    // get support type
                AnalyticalSupportInfoArray infoArray = supportData.InfoArray;

                if (infoArray.IsEmpty) {
                    str2 = "supported but no more information";
                }
                else {
                    for (AnalyticalSupportInfoArrayIterator i = infoArray.ForwardIterator(); i.MoveNext(); ) {
                        AnalyticalSupportInfo supportInfo = i.Current as AnalyticalSupportInfo;
                        str1 = str1 + supportInfo.SupportType.ToString() + ", ";
                    }
                }
            }
        }
    }
}
