
//
// (C) Copyright 2005 by Autodesk, Inc. 
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted, 
// provided that the above copyright notice appears in all copies and 
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting 
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC. 
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to 
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.


using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;

using Autodesk.Revit;
using Autodesk.Revit.Elements;

namespace RvtMgdDbg.Test.SDKSamples {
   
    /// <summary>
    /// These are all the tests taken directly from or heavily inspired by the SDK examples.  The idea
    /// is to consolidate everything in one tool so people don't have to build 4000 different projects
    /// and edit the revit.ini file everytime they want to test one.
    /// </summary>

    public class SDKTestFuncs : RvtMgdDbgTestFuncs {
        public SDKTestFuncs(Autodesk.Revit.Application app)
            : base(app)
        {
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Generate Sheet", "Generate a new sheet with a set of selected views", "SDK Samples", new RvtMgdDbgTestFuncInfo.TestFunc(GenerateSheet), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Analytical Support Data", "Show Analytical data for selected objects (Must be using Revit Structure)", "SDK Samples", new RvtMgdDbgTestFuncInfo.TestFunc(AnalyticalSupportData), RvtMgdDbgTestFuncInfo.TestType.Query));

            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Create Shared Parameters", "Create new shared parameters", "SDK Samples", new RvtMgdDbgTestFuncInfo.TestFunc(CreateSharedParams), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Type Selector", "Change the Type of an Element", "SDK Samples", new RvtMgdDbgTestFuncInfo.TestFunc(TypeSelector), RvtMgdDbgTestFuncInfo.TestType.Modify));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Move Linear Bound Element", "Move a single Element to a new location", "SDK Samples", new RvtMgdDbgTestFuncInfo.TestFunc(MoveLinearBound), RvtMgdDbgTestFuncInfo.TestType.Modify));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Structural Sample", "Place a set of columns at equal intervals on wall's", "SDK Samples", new RvtMgdDbgTestFuncInfo.TestFunc(StructSample), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Level Properties", "Add, remove and modify levels", "SDK Samples", new RvtMgdDbgTestFuncInfo.TestFunc(LevelProps), RvtMgdDbgTestFuncInfo.TestType.Query));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Fire Rating - Create Shared Parameter", "Create the \"Fire Rating\" Parameter", "SDK Samples", new RvtMgdDbgTestFuncInfo.TestFunc(FireRatingSharedParam), RvtMgdDbgTestFuncInfo.TestType.Create));
 

                // SDK Samples not implemented:
            // DeleteObject             (See TestDocument.Delete)
            // DesignOptionsReader      (use Snoop)
            // RevitCommands            (use Snoop)
            // BrowseBindings           (use Snoop)
            // ParameterUtils           (use Snoop)
        }

        public void
        GenerateSheet()
        {
            Document doc = m_revitApp.ActiveDocument;
            CreateSheet.Views view = new CreateSheet.Views(doc);

            CreateSheet.AllViewsForm dlg = new CreateSheet.AllViewsForm(view);
            if (dlg.ShowDialog() == DialogResult.OK)
                view.GenerateSheet(doc);
        }

        public void
        AnalyticalSupportData()
        {
            ElementSet selectedElements = m_revitApp.ActiveDocument.Selection.Elements;
            
            if (selectedElements.IsEmpty) {
                MessageBox.Show("No Elements were selected.");
                return;
            }

            AnalyticalSupportData.Info analysisData = new AnalyticalSupportData.Info(selectedElements);
            if (analysisData.ElementInformation.Rows.Count == 0) {
                MessageBox.Show("No Elements with Analytical Support Data were selected.");
                return;
            }

            AnalyticalSupportData.InfoForm form = new AnalyticalSupportData.InfoForm(analysisData);
            form.ShowDialog();
        }
        
 

        public void
        CreateSharedParams()
        {
            SharedParams.Create creator = new SharedParams.Create(m_revitApp);
            creator.AddSharedParamsToFile();
            creator.AddSharedParamsToWalls();
        }

        public void
        TypeSelector()
        {
            ElementSet selectedElements = m_revitApp.ActiveDocument.Selection.Elements;

            if (selectedElements.IsEmpty) {
                MessageBox.Show("No elements were selected.");
                return;
            }

            TypeSelector.TypeSelectorForm form = new TypeSelector.TypeSelectorForm(m_revitApp, selectedElements);
            form.ShowDialog();
        }
        
        public void
        MoveLinearBound()
        {
                // TBD: this usually produces a bunch of errors unless you pick one simple element
                // by itself, like a single Wall in isolation.
            int moveCount = 0;
            int rejectedCount = 0;
            
			foreach (Element elem in m_revitApp.ActiveDocument.Selection.Elements) {
	            Location loc = elem.Location;
				LocationCurve lineLoc = loc as LocationCurve;
				if (lineLoc != null) {
				    Autodesk.Revit.Geometry.Line line;
				    Autodesk.Revit.Geometry.XYZ newStart;
				    Autodesk.Revit.Geometry.XYZ newEnd;
    	      
				    newStart = lineLoc.Curve.get_EndPoint(0);
				    newEnd = lineLoc.Curve.get_EndPoint(1);
    	      
				    newStart.X += 100;
				    newStart.Y += 100;
				    newEnd.X += 75;
				    newEnd.Y += 75;

                    line = m_revitApp.Create.NewLineBound(newStart, newEnd);
				    lineLoc.Curve = line;
				
				    moveCount++;
				}
                else {
                    rejectedCount++;
                }
			}
			
			string msgStr = string.Format("Moved {0} elements.  {1} were not Linear-bound.", moveCount, rejectedCount);
			MessageBox.Show(msgStr);
		}

        public void
        StructSample ()
        {
            // first filter out to only Wall elements
            ElementSet selectedElements = Utils.Selection.FilterToCategory(m_revitApp.ActiveDocument.Selection.Elements,
                                                          Autodesk.Revit.BuiltInCategory.OST_Walls, false, m_revitApp.ActiveDocument);

            if (selectedElements.IsEmpty)
            {
                MessageBox.Show("No wall elements are currently selected");
                return;
            }           

            StructuralSample.StructSample sample = new StructuralSample.StructSample(m_revitApp, selectedElements);
            sample.ExecuteStructSample();
        }

        public void
        LevelProps ()
        {
            LevelProperties.LevelsCommand levCommand = new RvtMgdDbg.Test.SDKSamples.LevelProperties.LevelsCommand(m_revitApp);
            levCommand.Execute();
        }
        
        public void
        FireRatingSharedParam()
        {
            FireRating.SharedParam creator = new FireRating.SharedParam(m_revitApp);
            creator.AddSharedParamToFile();
        }
        
 
    }
}
