//      Rveit API .NET Sample 
//
//      Copyright (c) 2005-2006 by Autodesk, Inc.
//
//      Permission to use, copy, modify, and distribute this software
//      for any purpose and without fee is hereby granted, provided
//      that the above copyright notice appears in all copies and
//      that both that copyright notice and the limited warranty and
//      restricted rights notice below appear in all supporting
//      documentation.
//
//      AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS.
//      AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
//      MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC.
//      DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
//      UNINTERRUPTED OR ERROR FREE.
//
//      Use, duplication, or disclosure by the U.S. Government is subject to
//      restrictions set forth in FAR 52.227-19 (Commercial Computer
//      Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
//      (Rights in Technical Data and Computer Software), as applicable.
//

using System;
using System.Windows.Forms;

using Autodesk;
using Autodesk.Revit;
using Autodesk.Revit.Elements;
using Autodesk.Revit.Geometry; 
using Autodesk.Revit.Symbols;

namespace RvtMgdDbg.Test.SDKSamples.StructuralSample
{
    /// <summary>
    ///  .Net sample: StructSample.
    /// 
    ///   This command places a set of coulmns in the selected wall.    /// 
    ///   Note that Revit uses Feet as an internal length unit.     
    /// 
    ///   (1) Draw some walls, and constrain their top and bottom to the levels in the properties dialog. 
    ///   (2) Run the test. 
    ///       It will place columns along each wall with the interval of 5 feet. (The interval is also hard coded.) 
    ///   
    /// </summary>

    public class StructSample
    {
        Autodesk.Revit.Application m_app = null;
        Autodesk.Revit.ElementSet  m_ss = null;


        public StructSample (Autodesk.Revit.Application app, Autodesk.Revit.ElementSet ss)
        {
            m_app = app;
            m_ss  = ss;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public Boolean
        ExecuteStructSample ()
        {           
            Autodesk.Revit.Document rvtDoc = m_app.ActiveDocument;
            FamilySymbol colType = null;


            Autodesk.Revit.ElementSet walls = m_app.Create.NewElementSet();

            //  iterate through a selection set, and collect walls which are constrained at the top and the bottom.
            //
            foreach (Autodesk.Revit.Element elem in m_ss)
            {
                if (elem.GetType() == typeof(Autodesk.Revit.Elements.Wall))
                {
                    if (elem.get_Parameter(Autodesk.Revit.Parameters.BuiltInParameter.WALL_HEIGHT_TYPE) != null && elem.get_Parameter(Autodesk.Revit.Parameters.BuiltInParameter.WALL_BASE_CONSTRAINT) != null)
                    {
                        walls.Insert(elem);
                    }
                }
            }

            //  how many did we get? 
            MessageBox.Show("Number of constrained walls in the selection set is " + walls.Size);

            if (walls.Size == 0)
            {
                DialogResult dlgRes = MessageBox.Show("You must select some walls that are constrained top or bottom", "Structural Sample", MessageBoxButtons.OK, MessageBoxIcon.Information);

                if (dlgRes == System.Windows.Forms.DialogResult.OK)
                {
                    return false;
                }
            }

            //  we have some # of walls now.            

            // Load the required family

            Autodesk.Revit.ElementIterator familySetIter = rvtDoc.get_Elements(typeof(Family)); 
            while (familySetIter.MoveNext())
            {
                Family fam = familySetIter.Current as Family;
                if (fam != null)
                {
                    if (fam.Name == "M_Wood Timber Column")
                    {
                        colType = Utils.FamilyUtil.GetFamilySymbol(fam, "191 x 292mm");
                    }
                }
            }         

           
            String fileName = "../Data/Platform/Metric/Library/Architectural/Columns/M_Wood Timber Column.rfa";
            Family columnFamily;

            if (colType == null)
            {
                m_app.ActiveDocument.LoadFamily(fileName, out columnFamily);
                colType = Utils.FamilyUtil.GetFamilySymbol(columnFamily, "191 x 292mm");
            }                                  

            if (colType == null)
            {
                MessageBox.Show("Please load the M_Wood Timber Column : 191 x 292mm family");
                Utils.UserInput.LoadFamily(null, m_app.ActiveDocument);
            }                       

            //
            //  place columns.
            // 
            double spacing = 5;  //  Spacing in feet hard coded. Note: Revit's internal length unit is feet. 

            foreach (Autodesk.Revit.Elements.Wall wall in walls)
            {
                FrameWall(m_app, wall, spacing, colType);
            }

            return true;
        }             
    
        /// <summary>
        /// Frame a wall.
        /// </summary>
        /// <param name="rvtApp"></param>
        /// <param name="wall"></param>
        /// <param name="spacing"></param>
        /// <param name="columnType"></param>
        private void
        FrameWall (Autodesk.Revit.Application rvtApp, Autodesk.Revit.Elements.Wall wall, double spacing, Autodesk.Revit.Symbols.FamilySymbol columnType)
        {
            Autodesk.Revit.Document rvtDoc = wall.Document;

            Autodesk.Revit.LocationCurve loc = (Autodesk.Revit.LocationCurve)wall.Location;
            Autodesk.Revit.Geometry.XYZ startPt = loc.Curve.get_EndPoint(0);
            Autodesk.Revit.Geometry.XYZ endPt = loc.Curve.get_EndPoint(1);

            Autodesk.Revit.Geometry.UV wallVec = new Autodesk.Revit.Geometry.UV();
            wallVec.U = endPt.X - startPt.X;
            wallVec.V = endPt.Y - startPt.Y;

            Autodesk.Revit.Geometry.UV axis = new Autodesk.Revit.Geometry.UV();
            axis.U = 1.0;
            axis.V = 0.0;

            Autodesk.Revit.ElementId baseLevelId = wall.get_Parameter(Autodesk.Revit.Parameters.BuiltInParameter.WALL_BASE_CONSTRAINT).AsElementId();
            Autodesk.Revit.ElementId topLevelId = wall.get_Parameter(Autodesk.Revit.Parameters.BuiltInParameter.WALL_HEIGHT_TYPE).AsElementId();

            double wallLength = VecLength(wallVec);
            wallVec = VecNormalise(wallVec);

            int nmax = (int)(wallLength / spacing);

            MessageBox.Show("Wall Length = " + wallLength + "\nSpacing = " + spacing + "\nMax Number = " + nmax, "Structural Sample", MessageBoxButtons.OK, MessageBoxIcon.Information);

            double angle = VecAngle(wallVec, axis);

            Autodesk.Revit.Geometry.XYZ loc2 = startPt;

            double dx = wallVec.U * spacing;
            double dy = wallVec.V * spacing;

            for (int i = 0; i < nmax; i++)
            {
                PlaceColumn(rvtApp, rvtDoc, loc2, angle, columnType, baseLevelId, topLevelId);
                loc2.X += dx;
                loc2.Y += dy;
            }

            PlaceColumn(rvtApp, rvtDoc, endPt, angle, columnType, baseLevelId, topLevelId);
        }

        /// <summary>
        /// Create a column instance and place it on the wall line. 
        /// </summary>
        /// <param name="rvtApp"></param>
        /// <param name="rvtDoc"></param>
        /// <param name="point2"></param>
        /// <param name="angle"></param>
        /// <param name="columnType"></param>
        /// <param name="baseLevelId"></param>
        /// <param name="topLevelId"></param>
        private void
        PlaceColumn (Autodesk.Revit.Application rvtApp, Autodesk.Revit.Document rvtDoc, Autodesk.Revit.Geometry.XYZ point2, double angle, Autodesk.Revit.Symbols.FamilySymbol columnType, Autodesk.Revit.ElementId baseLevelId, Autodesk.Revit.ElementId topLevelId)
        {
            Autodesk.Revit.Geometry.XYZ point = point2;
            // Note: Must use level-hosted NewFamilyInstance!
            Level instLevel = (Level)rvtDoc.get_Element(ref baseLevelId);
            Autodesk.Revit.Elements.FamilyInstance column = rvtDoc.Create.NewFamilyInstance(point, columnType, instLevel, Autodesk.Revit.Structural.Enums.StructuralType.Column);

            if (column == null)
            {
                MessageBox.Show("failed to create an instance of a column.");
                return;
            }

            Autodesk.Revit.Geometry.XYZ zVec = new Autodesk.Revit.Geometry.XYZ();
            zVec.X = 0;
            zVec.Y = 0;
            zVec.Z = 1;

            Autodesk.Revit.Geometry.Line axis = rvtApp.Create.NewLineUnbound(point, zVec);

            column.Location.Rotate(axis, angle);

            // Set the level Ids
            Autodesk.Revit.Parameter baseLevelParameter = column.get_Parameter(Autodesk.Revit.Parameters.BuiltInParameter.FAMILY_BASE_LEVEL_PARAM);
            Autodesk.Revit.Parameter topLevelParameter = column.get_Parameter(Autodesk.Revit.Parameters.BuiltInParameter.FAMILY_TOP_LEVEL_PARAM); ;
            baseLevelParameter.Set(ref baseLevelId);
            topLevelParameter.Set(ref topLevelId);

        }

        //
        // helper functions. 
        //

        /// <summary>
        /// 
        /// </summary>
        /// <param name="v"></param>
        /// <returns></returns>
        private double
        VecLength (Autodesk.Revit.Geometry.UV v)
        {
            double len = v.U * v.U + v.V * v.V;
            len = Math.Sqrt(len);
            return (len);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="v"></param>
        /// <returns></returns>
        private Autodesk.Revit.Geometry.UV
        VecNormalise (Autodesk.Revit.Geometry.UV v)
        {
            double len = VecLength(v);
            v.U = v.U / len;
            v.V = v.V / len;
            return (v);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="va"></param>
        /// <param name="vb"></param>
        /// <returns></returns>
        private double
        VecDot (Autodesk.Revit.Geometry.UV va, Autodesk.Revit.Geometry.UV vb)
        {
            return ((va.U * vb.U) + (va.V * vb.V));
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="d"></param>
        /// <returns></returns>
        private double
        ACos (double d)
        {
            //  we probably need to consider tolerance here.
            if (Math.Abs(d) != 1.0)
            {
                return (1.5707963267949 - Math.Atan(d / Math.Sqrt(1.0 - d * d)));
            }
            if (d == -1.0)
            {
                return 3.14159265358979;
            }
            return 0.0;

        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="va"></param>
        /// <param name="vb"></param>
        /// <returns></returns>
        private double
        VecAngle (Autodesk.Revit.Geometry.UV va, Autodesk.Revit.Geometry.UV vb)
        {
            Autodesk.Revit.Geometry.UV vecA = VecNormalise(va);
            Autodesk.Revit.Geometry.UV vecB = VecNormalise(vb);

            return (ACos(VecDot(vecA, vecB)));
        }              
    }
}