
//
// (C) Copyright 2006 by Autodesk, Inc. 
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted, 
// provided that the above copyright notice appears in all copies and 
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting 
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC. 
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to 
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//

using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;

using Revit = Autodesk.Revit;
using Autodesk.Revit.Geometry;
using Autodesk.Revit.Parameters;
using Autodesk.Revit.Creation;

namespace RvtMgdDbg.Test {

    class TestDocument : RvtMgdDbgTestFuncs {

        public
        TestDocument(Autodesk.Revit.Application app)
            : base(app)
        {
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Delete SelSet", "Delete the current SelSet", typeof(Revit.Document), new RvtMgdDbgTestFuncInfo.TestFunc(Delete), RvtMgdDbgTestFuncInfo.TestType.Modify));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Move SelSet --> (10', 10', 0')", "Move the current SelSet", typeof(Revit.Document), new RvtMgdDbgTestFuncInfo.TestFunc(Move), RvtMgdDbgTestFuncInfo.TestType.Modify));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Rotate SelSet by 45 degrees", "Rotate the current SelSet", typeof(Revit.Document), new RvtMgdDbgTestFuncInfo.TestFunc(Rotate), RvtMgdDbgTestFuncInfo.TestType.Modify));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Load Family", "Load a .rfa file", typeof(Revit.Document), new RvtMgdDbgTestFuncInfo.TestFunc(LoadFamily), RvtMgdDbgTestFuncInfo.TestType.Modify));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Linear Array SelSet (Number = 4)", "Linearly array the current SelSet", typeof(Revit.Document), new RvtMgdDbgTestFuncInfo.TestFunc(LinearArray), RvtMgdDbgTestFuncInfo.TestType.Modify));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Radial Array SelSet by 30 degrees (Number = 4)", "Radially array the current SelSet", typeof(Revit.Document), new RvtMgdDbgTestFuncInfo.TestFunc(RadialArray), RvtMgdDbgTestFuncInfo.TestType.Modify));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Linear Array SelSet without associate", "Linearly array the current SelSet without associate", typeof(Revit.Document), new RvtMgdDbgTestFuncInfo.TestFunc(ArrayWithoutAssociate), RvtMgdDbgTestFuncInfo.TestType.Modify));
            //m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Mirror", "Mirror current SelSet", typeof(Revit.Document), new RvtMgdDbgTestFuncInfo.TestFunc(Mirror), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Boundary Lines", "Draw lines to sketch the Room Boundary", typeof(Revit.Document), new RvtMgdDbgTestFuncInfo.TestFunc(SketchBoundary), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Room Area", "Insert Area Value textnotes for all available rooms", typeof(Revit.Document), new RvtMgdDbgTestFuncInfo.TestFunc(RoomArea), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Filter Element types", "Filter for selected element types", typeof(Revit.Document), new RvtMgdDbgTestFuncInfo.TestFunc(FilterElementTypes), RvtMgdDbgTestFuncInfo.TestType.Query));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Add family types and parameters", "Use family manager for adddition of types/params", typeof(Revit.Document), new RvtMgdDbgTestFuncInfo.TestFunc(AddFamilyParameterAndType), RvtMgdDbgTestFuncInfo.TestType.Create));
        }

        public void
        Delete()
        {
            m_revitApp.ActiveDocument.Delete(m_revitApp.ActiveDocument.Selection.Elements);
        }

        public void
        Move()
        {
            XYZ vec = new XYZ(10.0, 10.0, 0.0);
            m_revitApp.ActiveDocument.Move(m_revitApp.ActiveDocument.Selection.Elements, vec);
        }

        public void
        Rotate()
        {
            Line zAxis = Line.get_Unbound(GeomUtils.kOrigin, GeomUtils.kZAxis);

            m_revitApp.ActiveDocument.Rotate(m_revitApp.ActiveDocument.Selection.Elements, zAxis, GeomUtils.kRad45);
        }

        public void
        LoadFamily()
        {
            Utils.UserInput.LoadFamily(null, m_revitApp.ActiveDocument);
        }

        public void
        LinearArray()
        {
            if (!m_revitApp.ActiveDocument.Selection.Elements.IsEmpty) {
                m_revitApp.ActiveDocument.Array(m_revitApp.ActiveDocument.Selection.Elements, 4, false, GeomUtils.kYAxis);
            }
        }

        public void
        RadialArray()
        {
            if (!m_revitApp.ActiveDocument.Selection.Elements.IsEmpty) {
                m_revitApp.ActiveDocument.Array(m_revitApp.ActiveDocument.Selection.Elements, 4, true, 30.0);
            }
        }

        public void
        ArrayWithoutAssociate()
        {
            if (!m_revitApp.ActiveDocument.Selection.Elements.IsEmpty) {
                m_revitApp.ActiveDocument.ArrayWithoutAssociate(m_revitApp.ActiveDocument.Selection.Elements, 4, false, GeomUtils.kYAxis);
            }
        }

        public void
        Mirror()
        {   // TBD: 

            //XYZ startPt1 = new XYZ(0.0, 0.0, 0.0);
            //XYZ endPt1 = new XYZ(56.5, 0.0, 0.0);

            //Line line1 = m_revitApp.Create.NewLine(ref startPt1, ref endPt1, true);

            ////// Note: There seems to be an issue with the Reference returned by curves. It always remains null.
            ////// arj 1/12/07           
            //m_revitApp.ActiveDocument.Mirror(m_revitApp.ActiveDocument.Selection.Elements, line1.Reference);
        }
       

        /// <summary>
        /// Draw lines to sketch the boundaries of available rooms in the Active Document.
        /// </summary>
        public void
        SketchBoundary ()
        {
            Revit.Creation.Document doc = m_revitApp.ActiveDocument.Create;
            Revit.Elements.SketchPlane sketchPlane = Utils.Geometry.GetWorldPlane(m_revitApp);

            RvtMgdDbg.Test.Forms.Levels lev = new RvtMgdDbg.Test.Forms.Levels(m_revitApp);
            if (lev.ShowDialog() != DialogResult.OK)
                return;

            Revit.Elements.Level curLevel = lev.LevelSelected;
            if (curLevel == null) {
                MessageBox.Show("No Level was selected.");
                return;
            }

                // Get the plan topology of the active doc first
            Revit.PlanTopology planTopo = m_revitApp.ActiveDocument.get_PlanTopology(curLevel);
            Revit.ElementSet roomSet = planTopo.Rooms;

            if (roomSet.Size > 0) {
                Revit.ElementSetIterator setIter = roomSet.ForwardIterator();
                while (setIter.MoveNext()) {
                    Revit.Elements.Room room = setIter.Current as Revit.Elements.Room;

                    if (null != room) {
                        Revit.Rooms.BoundarySegmentArrayArray boundSegArrayArray = room.Boundary;

                        Revit.Rooms.BoundarySegmentArrayArrayIterator iter = boundSegArrayArray.ForwardIterator();
                        while (iter.MoveNext()) {
                            Revit.Rooms.BoundarySegmentArray boundSegArray = iter.Current as Revit.Rooms.BoundarySegmentArray;

                            if (null != boundSegArray) {
                                Revit.Rooms.BoundarySegmentArrayIterator arrayIter = boundSegArray.ForwardIterator();
                                while (arrayIter.MoveNext()) {
                                    Revit.Rooms.BoundarySegment boundSeg = arrayIter.Current as Revit.Rooms.BoundarySegment;

                                    if (null != boundSeg) {
                                        // once you get to the Boundary Segment which represent one of the sides of the room boundary, draw a Model Curve to 
                                        // represent the outline.
                                        Revit.Elements.ModelCurve modCurve = m_revitApp.ActiveDocument.Create.NewModelCurve(boundSeg.Curve, sketchPlane);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            else {
                MessageBox.Show("No rooms found in the Active Document", "RvtMgdDbg", MessageBoxButtons.OK, MessageBoxIcon.Information);
            }

        }


        /// <summary>
        /// Calulate the area of all the available rooms and specify them using TextNotes
        /// </summary>
        public void
        RoomArea ()
        {
            Revit.Creation.Document doc = m_revitApp.ActiveDocument.Create;
            Revit.Elements.SketchPlane sketchPlane = Utils.Geometry.GetWorldPlane(m_revitApp);

            RvtMgdDbg.Test.Forms.Levels lev = new RvtMgdDbg.Test.Forms.Levels(m_revitApp);
            if (lev.ShowDialog() != DialogResult.OK)
                return;

            Revit.Elements.Level curLevel = lev.LevelSelected;
            if (curLevel == null) {
                MessageBox.Show("No Level was selected.");
                return;
            }

                // Get the plan topology of the active doc first
            Revit.PlanTopology planTopo = m_revitApp.ActiveDocument.get_PlanTopology(curLevel);
            Revit.ElementSet roomSet = planTopo.Rooms;

            if (roomSet.Size > 0) {
                Revit.ElementSetIterator setIter = roomSet.ForwardIterator();
                while (setIter.MoveNext()) {
                    Revit.Elements.Room room = setIter.Current as Revit.Elements.Room;

                    if (null != room) {
                        Revit.Elements.View view = m_revitApp.ActiveDocument.ActiveView;
                        Revit.LocationPoint locationPoint = room.Location as Revit.LocationPoint;

                        Double area = room.get_Parameter(Revit.Parameters.BuiltInParameter.ROOM_AREA).AsDouble();
                        
                        Double roundedArea = Math.Round(area, 2);

                        // providing an offset so that the Room Tag and the Area Tag dont overlap. Overlapping leads to an 
                        // alignment related assert.

                        XYZ offset = new XYZ(5.0, 0, 0);

                        /// align text middle and center
                        Revit.Enums.TextAlignFlags align = Revit.Enums.TextAlignFlags.TEF_ALIGN_MIDDLE ^ Revit.Enums.TextAlignFlags.TEF_ALIGN_CENTER;
                        Revit.Elements.TextNote txtNote = m_revitApp.ActiveDocument.Create.NewTextNote(view, offset + locationPoint.Point, GeomUtils.kXAxis,
                                                view.ViewDirection, .25,
                                                align, roundedArea.ToString());                        
                    }
                }
            }
            else {
                MessageBox.Show("No rooms found in the Active Document", "RvtMgdDbg", MessageBoxButtons.OK, MessageBoxIcon.Information);
            }


            // TBD: Tried to play around with PlanCircuits and there seems to be a problem with the IsRoomLocated property. 
            // arj 1/23/07

            //Revit.PlanCircuitSet circSet = planTopo.Circuits;
            //Revit.PlanCircuitSetIterator setIters = circSet.ForwardIterator();

            //while (setIters.MoveNext())
            //{
            //    Revit.PlanCircuit planCircuit = setIters.Current as Revit.PlanCircuit;

            //    if (null != planCircuit)
            //    {
            //
            //        if (planCircuit.IsRoomLocated) // throws an exception always "Attempted to read or write protected memory.
                                                     // This is often an indication that other memory is corrupt."
            //        {
            //        }
            //    }
            //}
        }

        /// <summary>
        /// Have the user select a type of Element and then filter the document for all instances of that type.
        /// </summary>
        public void
        FilterElementTypes()
        {
            Test.Forms.Elements elems = new Test.Forms.Elements(m_revitApp);
            if (elems.ShowDialog() != DialogResult.OK)
                return;

            Revit.ElementSet elemSet = new Revit.ElementSet();
            Revit.ElementIterator elemFilterIter = m_revitApp.ActiveDocument.get_Elements(elems.ElemTypeSelected);

            while (elemFilterIter.MoveNext()) {
                elemSet.Insert(elemFilterIter.Current as Revit.Element);
            }

            Snoop.Forms.Objects objs = new Snoop.Forms.Objects(elemSet);
            objs.ShowDialog();
        }

        /// <summary>
        /// Ask the user to open a revit family template and then add FamilyTypes and FamilyParameters
        /// to it. Say the user opens a Door template, he can then save the family as a new door family and load
        /// it into a new project for use.
        /// </summary>
        public void AddFamilyParameterAndType()
        {
            Revit.Document doc;           

            OpenFileDialog openFileDialog = new OpenFileDialog();
            openFileDialog.Title = "Select family document";
            openFileDialog.Filter = "RFT Files (*.rft)|*.rft";

            if (openFileDialog.ShowDialog() == DialogResult.OK) {
                doc = m_revitApp.NewFamilyDocument(openFileDialog.FileName);
            }
            else
                return;                       

            doc.BeginTransaction();
            if (doc.IsFamilyDocument) { // has to be a family document to be able to use the Family Manager.
               
                Autodesk.Revit.FamilyManager famMgr = doc.FamilyManager;                

                //Add a family param. 
                Autodesk.Revit.FamilyParameter famParam = famMgr.AddParameter("RvtMgdDbg_Param", BuiltInParameterGroup.PG_TITLE, ParameterType.Text, false);
                famMgr.Set(famParam, "Default text.");                

                //Create a couple of new family types. Note that we can set different values for the param
                //in different family types.                
               
                Revit.FamilyType newFamType = famMgr.NewType("RvtMgdDbg_Type1");
                famMgr.CurrentType = newFamType;

                if (newFamType.HasValue(famParam)) {
                    famMgr.Set(famParam, "Text1.");
                }

                Revit.FamilyType newFamType1 = famMgr.NewType("RvtMgdDbg_Type2");
                famMgr.CurrentType = newFamType;

                if (newFamType.HasValue(famParam)) {
                    famMgr.Set(famParam, "Text2.");
                }                

                famMgr.MakeType(famParam);

                if ((famParam != null) && (newFamType != null)) {
                    MessageBox.Show("New family types/params added successfully.", "RvtMgdDbg", MessageBoxButtons.OK, MessageBoxIcon.Information);
                }
                else
                    MessageBox.Show("Family types/params addition failed.", "RvtMgdDbg", MessageBoxButtons.OK, MessageBoxIcon.Error);                
            }
            doc.EndTransaction();
        }
    }
}