//
// (C) Copyright 2005 by Autodesk, Inc. 
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted, 
// provided that the above copyright notice appears in all copies and 
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting 
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC. 
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to 
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//

using System;
using System.Windows.Forms;
using System.Collections;
using System.Diagnostics;
using System.Collections.Generic;

using Revit = Autodesk.Revit;
using Autodesk.Revit.Geometry;
using Autodesk.Revit.Elements;
using Autodesk.Revit.Symbols;
using Autodesk.Revit.Creation;
using Autodesk.Revit.Parameters;

namespace RvtMgdDbg.Test {
    /// <summary>
    /// These are all the tests concerning Elements.  If there become alot of them, then we can break this up into
    /// mulitple RvtMgdDbgTestFuncs objects.
    /// </summary>

    public class TestElements : RvtMgdDbgTestFuncs
    {

        private static Revit.ElementIdSet m_shedElements = new Autodesk.Revit.ElementIdSet();

        public TestElements (Revit.Application app)
            : base(app)
        {
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Windows at 18\" sills", "Change all Windows to have an 18\" sill height", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(WindowSill_18), RvtMgdDbgTestFuncInfo.TestType.Modify));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Center Windows Vertically", "Change all Windows to be centered in the wall", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(WindowCenterVertical), RvtMgdDbgTestFuncInfo.TestType.Modify));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Parameter Enum Mapping", "Show enum -> param name mapping", "Enum Mappings", new RvtMgdDbgTestFuncInfo.TestFunc(ParameterEnums), RvtMgdDbgTestFuncInfo.TestType.Other));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Parameter Enum Mapping (no duplicates)", "Show enum -> param name mapping", "Enum Mappings", new RvtMgdDbgTestFuncInfo.TestFunc(ParameterEnumsNoDups), RvtMgdDbgTestFuncInfo.TestType.Other));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Category Enum Mapping", "Show enum -> category name mapping", "Enum Mappings", new RvtMgdDbgTestFuncInfo.TestFunc(CategoryEnums), RvtMgdDbgTestFuncInfo.TestType.Other));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Simple Wall", "Create a hardwired wall", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(SimpleWall), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Simple Floor", "Add a floor for the selected walls", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(SimpleFloor), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Hardwired Shed", "Create some hardwired walls, floors, windows and doors", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(SimpleShed), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Swap Family", "Swap any door with a double door", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(SimpleSwap), RvtMgdDbgTestFuncInfo.TestType.Modify));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Level Iteration", "Iterate over levels and change floor to floor height", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(SimpleLevelIteration), RvtMgdDbgTestFuncInfo.TestType.Modify));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Hardwired Classroom", "Create curtain walls, door and furniture", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(ClassRoom), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Flip element(s)", "Rotates element(s) by 180 degrees", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(Flip), RvtMgdDbgTestFuncInfo.TestType.Modify));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Mirror", "Mirrors selected elements along X axis (Draw above X Axis)", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(Mirror), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Annotation Symbol", "Creates a new annotation symbol", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(AnnoSymbol), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Beam System", "Creates a hardwired beam system", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(BeamSystemHardWired), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Detail Curves", "Creates hardwired detail curves", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(DetailCurveHardWired), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Dimension", "Creates hardwired lines and a dimension for their distance", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(DimensionHardWired), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Foundation Slab", "Creates a hardwired foundation slab", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(FoundationSlabHardWired), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Text Note", "Creates a hardwired text note", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(TextNoteHardWired), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Simple Slab", "Creates a hardwired slab", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(SimpleSlab), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("View Section", "Creates a view section", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(SimpleViewSection), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("View Plan", "Creates a floor view plan", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(FloorViewPlan), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("View and Sheet Addition", "Adds Floor Plan (Level 1) view to new sheet", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(ViewToNewSheetHardwired), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Simple Tag", "Add a tag to the selected elements", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(SimpleTag), RvtMgdDbgTestFuncInfo.TestType.Create));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Update Wall Width", "Update the WallType layer thickness to change the width of the wall.", typeof(Revit.Element), new RvtMgdDbgTestFuncInfo.TestFunc(ModifyWallWidth), RvtMgdDbgTestFuncInfo.TestType.Modify));                       
        }


        public void
        WindowSill_18 ()
        {
            // first filter out to only Window elements
            Revit.ElementSet windowSet = Utils.Selection.FilterToCategory(m_revitApp.ActiveDocument.Selection.Elements,
                                                    Revit.BuiltInCategory.OST_Windows, false, m_revitApp.ActiveDocument);

            if (windowSet.IsEmpty) {
                MessageBox.Show("No Window elements are currently selected");
                return;
            }

            // go through each window and set the Sill parameter
            foreach (Revit.Element elem in windowSet) {
                double sillHt = Utils.Convert.InchesToFeet(18.0);   // Revit wants things in Feet and Fractional Inches

                Revit.Parameter param = elem.get_Parameter(BuiltInParameter.INSTANCE_SILL_HEIGHT_PARAM);
                param.Set(sillHt);
            }
        }

        public void
        WindowCenterVertical ()
        {
            // first filter out to only Window elements
            Revit.ElementSet windowSet = Utils.Selection.FilterToCategory(m_revitApp.ActiveDocument.Selection.Elements,
                                                    Revit.BuiltInCategory.OST_Windows, false, m_revitApp.ActiveDocument);

            if (windowSet.IsEmpty) {
                MessageBox.Show("No Window elements are currently selected");
                return;
            }

            // go through each window and set the Sill parameter
            foreach (Revit.Element elem in windowSet) {
                try {
                    FamilyInstance window = (FamilyInstance)elem;
                    HostObject host = (HostObject)window.Host;

                    double wallHt = GetWallHeight(host);
                    double windowHt = GetWindowHeight(window);
                    double sillHt = (wallHt / 2.0) - (windowHt / 2.0);

                    Revit.Parameter param = elem.get_Parameter(BuiltInParameter.INSTANCE_SILL_HEIGHT_PARAM);
                    param.Set(sillHt);
                }
                catch (System.Exception e) {	// we want to catch it so we can see the problem, otherwise it just silently bails out
                    MessageBox.Show(e.Message);
                    throw e;
                }
            }
        }


        private double
        GetWallHeight (HostObject hostObj)
        {
            return hostObj.get_Parameter(BuiltInParameter.WALL_USER_HEIGHT_PARAM).AsDouble();
        }

        private double
        GetWindowHeight (FamilyInstance window)
        {
            return window.ObjectType.get_Parameter(BuiltInParameter.WINDOW_HEIGHT).AsDouble();
        }

        private void
        ParameterEnums ()
        {
            ArrayList labelStrs = new ArrayList();
            ArrayList valueStrs = new ArrayList();

            foreach (BuiltInParameter paramEnum in System.Enum.GetValues(typeof(BuiltInParameter))) {
                labelStrs.Add(paramEnum.ToString());
                valueStrs.Add(string.Format("{0:d}", (int)paramEnum));
            }

            RvtMgdDbg.Snoop.Forms.ParamEnum dbox = new RvtMgdDbg.Snoop.Forms.ParamEnum(labelStrs, valueStrs);
            dbox.ShowDialog();
        }

        private void
        ParameterEnumsNoDups ()
        {
            ArrayList labelStrs = new ArrayList();
            ArrayList valueStrs = new ArrayList();

            string[] strs = System.Enum.GetNames(typeof(BuiltInParameter));
            foreach (string str in strs) {
                BuiltInParameter paramEnum = (BuiltInParameter)System.Enum.Parse(typeof(BuiltInParameter), str);

                labelStrs.Add(str);
                valueStrs.Add(string.Format("{0:d}", (int)paramEnum));
            }

            RvtMgdDbg.Snoop.Forms.ParamEnum dbox = new RvtMgdDbg.Snoop.Forms.ParamEnum(labelStrs, valueStrs);
            dbox.ShowDialog();
        }

        private void
        CategoryEnums ()
        {
            Revit.Categories categories = m_revitApp.ActiveDocument.Settings.Categories;
            ArrayList labelStrs = new ArrayList();
            ArrayList valueStrs = new ArrayList();

            foreach (Revit.BuiltInCategory catEnum in System.Enum.GetValues(typeof(Revit.BuiltInCategory))) {
                labelStrs.Add(catEnum.ToString());

                Revit.Category cat = categories.get_Item(catEnum);
                if (cat == null)
                    valueStrs.Add("<null>");
                else
                    valueStrs.Add(cat.Name);
            }

            RvtMgdDbg.Snoop.Forms.ParamEnum dbox = new RvtMgdDbg.Snoop.Forms.ParamEnum(labelStrs, valueStrs);
            dbox.ShowDialog();
        }                  

        /// <summary>
        /// 
        /// </summary>
        public void ClassRoom ()
        {
            Revit.Document doc = m_revitApp.ActiveDocument;            

            // get the symbols in the beginning itself , so that 
            // if one is missing you can load it
            Revit.Symbols.FamilySymbol columnSymbol = null;
            Revit.Symbols.FamilySymbol chairSymbol = null;
            Revit.Symbols.FamilySymbol deskSymbol = null;
            Revit.Symbols.FamilySymbol chairMainSymbol = null;
            //Revit.Symbols.FamilySymbol chairSym = null;
            Revit.Symbols.FamilySymbol doorSymbol = null;
            Revit.Elements.Family      columnfamily;
            Revit.Elements.Family      deskfamily;
            Revit.Elements.Family      doorfamily;

            Revit.ElementIterator familySetIter = doc.get_Elements(typeof(Family));           
            while (familySetIter.MoveNext())
            {
                Family fam = familySetIter.Current as Family;
                if (fam != null) {
                    if (fam.Name == "Rectangular Column") {
                        columnSymbol = Utils.FamilyUtil.GetFamilySymbol(fam, "24\" x 24\"");
                    }

                    if (fam.Name == "Desk") {
                        deskSymbol = Utils.FamilyUtil.GetFamilySymbol(fam, "60\" x 30\"");
                    }

                    if (fam.Name == "Single-Flush") {
                        doorSymbol = Utils.FamilyUtil.GetFamilySymbol(fam, "36\" x 84\"");
                    }

                    if (fam.Name == "Chair-Tablet Arm") {
                        chairSymbol = Utils.FamilyUtil.GetFamilySymbol(fam, "Chair-Tablet");
                    }

                    if (fam.Name == "Chair-Breuer") {
                        chairMainSymbol = Utils.FamilyUtil.GetFamilySymbol(fam, "Chair-Breuer");
                    }
                }
            }

            String familyNameToLoad = "";               

            // check for required families
            if (columnSymbol == null) {
                MessageBox.Show("Please load Rectangular Column 24\" x 24\" into project");

                String famName = Utils.UserInput.GetFamilyNameFromUser(null, ref familyNameToLoad);

                m_revitApp.ActiveDocument.LoadFamily(famName, out columnfamily);
                if (columnfamily != null) {
                    columnSymbol = Utils.FamilyUtil.GetFamilySymbol(columnfamily, "24\" x 24\"");
                }
            }

            if (deskSymbol == null) {
                MessageBox.Show("Please load Desk 60\" x 30\" into project");

                String famName = Utils.UserInput.GetFamilyNameFromUser(null, ref familyNameToLoad);

                m_revitApp.ActiveDocument.LoadFamily(famName, out deskfamily);
                if (deskfamily != null) {
                    deskSymbol = Utils.FamilyUtil.GetFamilySymbol(deskfamily, "36\" x 84\"");
                }
            }

            if (doorSymbol == null) {
                MessageBox.Show("Please load door Single Flush 36\" x 84\" into project");

                String famName = Utils.UserInput.GetFamilyNameFromUser(null, ref familyNameToLoad);

                m_revitApp.ActiveDocument.LoadFamily(famName, out doorfamily);
                if (doorfamily != null) {
                    doorSymbol = Utils.FamilyUtil.GetFamilySymbol(doorfamily, "36\" x 84\"");
                }
            }

            Revit.Collections.StringStringMapIterator iter = m_revitApp.Options.LibraryPaths.ForwardIterator();
            string pathName = string.Empty;
            if (iter.MoveNext()) {
                pathName = iter.Current.ToString();
            }

            string fileName = string.Empty;
            bool success = false;

            fileName = pathName + @"\Furniture\Chair-Tablet Arm.rfa";

            //Boolean load = m_revitApp.ActiveDocument.LoadFamilySymbol(fileName, "Chair-Tablet Arm", ref chairSym);

            Revit.Elements.Family chairFamily;
            // if not found in doc try to load automatically
            if (chairSymbol == null) {
                success = m_revitApp.ActiveDocument.LoadFamily(fileName, out chairFamily);
                chairSymbol = Utils.FamilyUtil.GetFamilySymbol(chairFamily, "Chair-Tablet Arm");
            }
            // last ditch effort by trying to load manually
            if (chairSymbol == null) {
                MessageBox.Show("Please load Chair-Tablet Arm into project");
                Utils.UserInput.LoadFamily(null, m_revitApp.ActiveDocument);
            }

            fileName = pathName + @"\Furniture\Chair-Breuer.rfa";
            Revit.Elements.Family chairMainFamily;
            // if not found in doc try to load automatically
            if (chairMainSymbol == null) {
                success = m_revitApp.ActiveDocument.LoadFamily(fileName, out chairMainFamily);
                chairMainSymbol = Utils.FamilyUtil.GetFamilySymbol(chairMainFamily, "Chair-Breuer");
            }
            // last ditch effort by trying to load manually
            if (chairMainSymbol == null) {
                MessageBox.Show("Please load Chair-Breuer into project");
                Utils.UserInput.LoadFamily(null, m_revitApp.ActiveDocument);
            }

            // get the level on which we want to build
            Level level1 = null;
            Revit.ElementIterator levelSetIter = doc.get_Elements(typeof(Revit.Elements.Level));                       
            while (levelSetIter.MoveNext()) {
                Revit.Elements.Level levelTemp = levelSetIter.Current as Revit.Elements.Level;
                if (levelTemp != null) {
                    if (levelTemp.Name == "Level 1") {
                        level1 = levelTemp;
                        break;
                    }
                }
            }

            // draw 4 lines
            Revit.Geometry.XYZ startPt1 = new XYZ(0.0, 0.0, 0.0);
            Revit.Geometry.XYZ endPt1 = new XYZ(56.5, 0.0, 0.0);
            Line line1 = m_revitApp.Create.NewLine(startPt1, endPt1, true);

            Revit.Geometry.XYZ startPt2 = endPt1;
            Revit.Geometry.XYZ endPt2 = new XYZ(56.5, 61.5, 0.0);
            Line line2 = m_revitApp.Create.NewLine(startPt2, endPt2, true);

            Revit.Geometry.XYZ startPt3 = endPt2;
            Revit.Geometry.XYZ endPt3 = new XYZ(0.0, 61.5, 0.0);
            Line line3 = m_revitApp.Create.NewLine(startPt3, endPt3, true);

            Revit.Geometry.XYZ startPt4 = endPt3;
            Revit.Geometry.XYZ endPt4 = startPt1;
            Line line4 = m_revitApp.Create.NewLine(startPt4, endPt4, true);

            // get the wall types we want
            Revit.Symbols.WallType curtainWallType = null;
            Revit.Symbols.WallType outerWallType = null;

            Revit.ElementIterator wallTypesSetIter = doc.get_Elements(typeof(Revit.Symbols.WallType));            
            while (wallTypesSetIter.MoveNext()) {
                Revit.Symbols.WallType wallType = wallTypesSetIter.Current as Revit.Symbols.WallType;
                if (wallType != null) {
                    if (wallType.Name == "Curtain Wall 1") {
                        curtainWallType = wallType;
                    }
                    if (wallType.Name == "Generic - 6\" Masonry") {
                        outerWallType = wallType;
                    }
                }
            }

            // draw 4 walls
            Wall wall1 = m_revitApp.ActiveDocument.Create.NewWall(line1, outerWallType, level1, 7.0, 0.0, false, false);
            Wall wall2 = m_revitApp.ActiveDocument.Create.NewWall(line2, curtainWallType, level1, 7.0, 0.0, false, false);
            Wall wall3 = m_revitApp.ActiveDocument.Create.NewWall(line3, curtainWallType, level1, 7.0, 0.0, false, false);
            Wall wall4 = m_revitApp.ActiveDocument.Create.NewWall(line4, curtainWallType, level1, 7.0, 0.0, false, false);


            FamilyInstance StructColumn1 = m_revitApp.ActiveDocument.Create.NewFamilyInstance(startPt1, columnSymbol, wall1, level1, Autodesk.Revit.Structural.Enums.StructuralType.Column);
            FamilyInstance StructColumn2 = m_revitApp.ActiveDocument.Create.NewFamilyInstance(startPt2, columnSymbol, wall2, level1, Autodesk.Revit.Structural.Enums.StructuralType.Column);
            FamilyInstance StructColumn3 = m_revitApp.ActiveDocument.Create.NewFamilyInstance(startPt3, columnSymbol, wall3, level1, Autodesk.Revit.Structural.Enums.StructuralType.Column);
            FamilyInstance StructColumn4 = m_revitApp.ActiveDocument.Create.NewFamilyInstance(startPt4, columnSymbol, wall4, level1, Autodesk.Revit.Structural.Enums.StructuralType.Column);

            int row = 5;
            int col = 8;

            double x = 5.0;
            double y = 11.0;
            double z = 0.0;

            XYZ refDirection = new XYZ(1, 10, 0);

            // place all the chairs
            for (int i = 0; i < row; i++) {
                y = y + 8;
                for (int j = 0; j < col; j++) {
                    x = x + 5;
                    XYZ location = new XYZ(x, y, z);
                    FamilyInstance famInst = m_revitApp.ActiveDocument.Create.NewFamilyInstance(location, chairSymbol, Autodesk.Revit.Structural.Enums.StructuralType.UnknownFraming);

                    // flip the chairs 180 degrees
                    XYZ pt = location;
                    XYZ dir = GeomUtils.kZAxis;
                    Line zAxis = m_revitApp.Create.NewLineUnbound(pt, dir);
                    famInst.Location.Rotate(zAxis, GeomUtils.kRad180);
                }
                x = x - (col * 5);
            }

            // place the desk
            XYZ deskPosition = new XYZ(15.0, 8.0, 0.0);
            m_revitApp.ActiveDocument.Create.NewFamilyInstance(deskPosition, deskSymbol, Autodesk.Revit.Structural.Enums.StructuralType.UnknownFraming);

            // place the chair
            XYZ chairPosition = new XYZ(18, 7.0, 0.0);
            m_revitApp.ActiveDocument.Create.NewFamilyInstance(chairPosition, chairMainSymbol, Autodesk.Revit.Structural.Enums.StructuralType.UnknownFraming);

            // place the door
            XYZ doorPosition = new XYZ(46.0, 0.0, 0.0);
            m_revitApp.ActiveDocument.Create.NewFamilyInstance(doorPosition, doorSymbol, wall1, Autodesk.Revit.Structural.Enums.StructuralType.Brace);
        }

        /// <summary>
        /// Create a single wall at the specified location and level
        /// </summary>
        public void
        SimpleWall ()
        {
            XYZ pt1 = new XYZ(10.0, 10.0, 0.0);
            XYZ pt2 = new XYZ(30.0, 10.0, 0.0);

            Line line = Line.get_Bound(pt1, pt2);

            Revit.Creation.Document doc = m_revitApp.ActiveDocument.Create;

            // Just for kicks we will place it at a newly created level, but on checking the 
            // properties of the wall we can note that it still exists on Level 1.
            doc.NewWall(line, doc.NewLevel(0.0), true);
        }


        /// <summary>
        /// Create a floor for the selected set of walls.
        /// </summary>
        public void
        SimpleFloor ()
        {
            Revit.Creation.Document doc = m_revitApp.ActiveDocument.Create;
            Revit.Creation.Application applic = m_revitApp.Create;

            // first filter out to only Wall elements
            Revit.ElementSet wallSet = Utils.Selection.FilterToCategory(m_revitApp.ActiveDocument.Selection.Elements,
                                                    Revit.BuiltInCategory.OST_Walls, false, m_revitApp.ActiveDocument);

            if (wallSet.IsEmpty) {
                MessageBox.Show("No wall elements are currently selected");
                return;
            }

            // Get the wall profile needed for the floor
            CurveArray profile = applic.NewCurveArray();

            foreach (Wall w in wallSet) {
                Revit.LocationCurve curve = w.Location as Revit.LocationCurve;
                profile.Append(curve.Curve);
            }


            // Obtain the required floor type
            FloorType floorType = null;

            try {
                Revit.ElementIterator elements = m_revitApp.ActiveDocument.Elements;
                elements.Reset();

                while (elements.MoveNext()) {
                    Revit.Symbols.FloorType f = elements.Current as Revit.Symbols.FloorType;

                    if (null == f) {
                        continue;
                    }

                    if (f.Name == "Generic - 12\"") {
                        floorType = f;
                    }
                }
            }
            catch (Exception e) {
                throw e;
            }

            // Set the stuctural value
            bool structural = true;

            // Create the floor instance
            try {
                Floor f = doc.NewFloor(profile, floorType, doc.NewLevel(0.0), structural);
            }
            catch (Exception e) {
                throw e;
            }
        }



        /// <summary>
        /// Creates a simple shed consisting of a door, window, a few floors and walls
        /// </summary>
        public void
        SimpleShed ()
        {
            Revit.Symbols.FamilySymbol doorSymbol    = null;
            Revit.Symbols.FamilySymbol windowSymbol  = null;            
            Revit.Elements.Family      doorFamily;
            Revit.Elements.Family      windowFamily;            
            Revit.Creation.Document    doc           = m_revitApp.ActiveDocument.Create;
            Revit.Document             docu          = m_revitApp.ActiveDocument;
            Revit.Creation.Application applic        = m_revitApp.Create;


            //The levels for the floors and the fake roof
            Level floorLevel = null;
            Level midLevel = null;

            Revit.ElementSet wallSet = new Autodesk.Revit.ElementSet();

            // Create a new CurveArray to provide the profile of the walls to the floor
            CurveArray curArray = applic.NewCurveArray();


            // iterate through all available levels...               
            Revit.ElementIterator elemIterator = m_revitApp.ActiveDocument.Elements;
            elemIterator.Reset();

            while (elemIterator.MoveNext()) {
                Level sysLevel = elemIterator.Current as Level;
                if (sysLevel != null) {
                    String name = sysLevel.Name;

                    if (name == "Level 1") {
                        floorLevel = sysLevel;
                    }

                    if (name == "Level 2") {
                        midLevel = sysLevel;
                    }
                }
            }           
            
            // first create 4 walls            

            // wall1
            XYZ pt1 = new XYZ(10.0, 10.0, 0.0);
            XYZ pt2 = new XYZ(30.0, 10.0, 0.0);

            Line line = applic.NewLine(pt1, pt2, true);

            Revit.Elements.Wall windowHost = doc.NewWall(line, floorLevel, false);

            curArray.Append(line);
            wallSet.Insert(windowHost);

            Revit.ElementId windowHostId = windowHost.Id;
            m_shedElements.Insert(ref windowHostId);

            // wall2
            XYZ pt3 = new XYZ(10.0, 10.0, 0.0);
            XYZ pt4 = new XYZ(10.0, 30.0, 0.0);

            Line line1 = applic.NewLine(pt3, pt4, true);

            Revit.Elements.Wall wall2 = doc.NewWall(line1, floorLevel, false);

            curArray.Append(line1);
            wallSet.Insert(wall2);

            Revit.ElementId wall2Id = wall2.Id;
            m_shedElements.Insert(ref wall2Id);

            // wall3
            XYZ pt5 = new XYZ(10.0, 30.0, 0.0);
            XYZ pt6 = new XYZ(30.0, 30.0, 0.0);

            Line line2 = applic.NewLine(pt5, pt6, true);

            Revit.Elements.Wall doorHost = doc.NewWall(line2, floorLevel, false);

            curArray.Append(line2);
            wallSet.Insert(doorHost);

            Revit.ElementId doorHostId = doorHost.Id;
            m_shedElements.Insert(ref doorHostId);

            // wall4
            XYZ pt7 = new XYZ(30.0, 30.0, 0.0);
            XYZ pt8 = new XYZ(30.0, 10.0, 0.0);

            Line line3 = applic.NewLine(pt7, pt8, true);

            Revit.Elements.Wall wall4 = doc.NewWall(line3, floorLevel, false);

            curArray.Append(line3);
            wallSet.Insert(wall4);

            Revit.ElementId wall4Id = wall4.Id;
            m_shedElements.Insert(ref wall4Id);

            if (curArray != null) {
                // add two floors to the walls created above
                Revit.ElementId floorId = SimpleFloor(curArray, floorLevel);
                SimpleFloor(curArray, midLevel);

                // set the top/bottom constraint and room bounding  params for the walls

                // Note: The WALL_TOP/BOTTOM_IS_ATTACHED parameter's value when set, does not seem to stick. This needs
                // to be looked into as well. arj 7/25/06

                // The WALL_TOP/BOTTOM_IS_ATTACHED parameters are Read Only and cannot be set. arj 01/17/07
                foreach (Wall wall in wallSet){                   
                    Revit.Parameter wallRoomBound = wall.get_Parameter(Revit.Parameters.BuiltInParameter.WALL_ATTR_ROOM_BOUNDING);
                    wallRoomBound.Set(1);
                    
                    Revit.Parameter wallBaseConstr = wall.get_Parameter(Revit.Parameters.BuiltInParameter.WALL_BASE_CONSTRAINT);
                    wallBaseConstr.Set(ref floorId);
                }
            }            

            // set the location for elements such as doors, windows

            XYZ doorLocation = new XYZ(20.0, 30.0, 0.0);
            XYZ windowLocation = new XYZ(20.0, 10.0, 10.0);

            // check for required symbols     
            Revit.ElementIterator familySetIter = docu.get_Elements(typeof(Family));            
            while (familySetIter.MoveNext()) {
                Family fam = familySetIter.Current as Family;
                if (fam != null) {
                    if (fam.Name == "Single-Flush") {
                        doorSymbol = Utils.FamilyUtil.GetFamilySymbol(fam, "36\" x 84\"");
                    }

                    if (fam.Name == "Fixed") {
                        windowSymbol = Utils.FamilyUtil.GetFamilySymbol(fam, "24\" x 48\"");
                    }
                }
            }

            // check for required symbols

            String familyNameToLoad = "";

            if (doorSymbol == null) {
                MessageBox.Show("Please load Single-Flush 34\" x 84\" into project");

                String famName = Utils.UserInput.GetFamilyNameFromUser(null, ref familyNameToLoad);

                m_revitApp.ActiveDocument.LoadFamily(famName, out doorFamily);
                if (doorFamily != null) {
                    doorSymbol = Utils.FamilyUtil.GetFamilySymbol(doorFamily, "36\" x 84\"");
                }
            }

            if (windowSymbol == null) {
                MessageBox.Show("Please load Fixed 24\" x 48\" into project");

                String famName = Utils.UserInput.GetFamilyNameFromUser(null, ref familyNameToLoad);

                m_revitApp.ActiveDocument.LoadFamily(famName, out windowFamily);               
                if (windowFamily != null) {
                    windowSymbol = Utils.FamilyUtil.GetFamilySymbol(windowFamily, "24\" x 48\"");
                }
            }

            // Create the door instance            

            floorLevel.Elevation = 0.0;
            FamilyInstance doorInst = doc.NewFamilyInstance(doorLocation, doorSymbol, doorHost, floorLevel, Revit.Structural.Enums.StructuralType.Column);

            Revit.ElementId doorInstId = doorInst.Id;
            m_shedElements.Insert(ref  doorInstId);


            // Create the window instance                     

            floorLevel.Elevation = 0.0;
            FamilyInstance windowInst = doc.NewFamilyInstance(windowLocation, windowSymbol, windowHost, floorLevel, Revit.Structural.Enums.StructuralType.Column);

            // The sill height needs to be set to 4.0 or less for the window to be visible in Level 1 view.
            Revit.Parameter sillHeightParam = windowInst.get_Parameter(Revit.Parameters.BuiltInParameter.INSTANCE_SILL_HEIGHT_PARAM);
            sillHeightParam.Set(4.0);

            Revit.ElementId windowInstId = windowInst.Id;
            m_shedElements.Insert(ref  windowInstId);
        }



        /// <summary>
        ///  Used by the SimpleShed to create its floors and the fake roof
        /// </summary>
        /// <param name="profile"></param>
        public Revit.ElementId
        SimpleFloor (CurveArray profile, Level level)
        {
            Revit.Creation.Document doc = m_revitApp.ActiveDocument.Create;
            Revit.Creation.Application applic = m_revitApp.Create;

            // Obtain the required floor type
            FloorType floorType = null;

            try {
                Revit.ElementIterator elements = m_revitApp.ActiveDocument.Elements;

                elements.Reset();
                while (elements.MoveNext()) {
                    Revit.Symbols.FloorType fType = elements.Current as Revit.Symbols.FloorType;

                    if (fType == null) {
                        continue;
                    }

                    if (fType.Name == "Generic - 12\"") {
                        floorType = fType;
                    }
                }
            }
            catch (Exception e) {
                throw e;
            }

            // Set the stuctural value
            bool structural = true;

            Revit.ElementId elemId = new Autodesk.Revit.ElementId();

            // Create the floor instance
            try {
                if (level.Name == "Level 2") {
                    level.Elevation = 10.0;

                    Floor f = doc.NewFloor(profile, floorType, level, structural);

                    Revit.ElementId fId = f.Id;
                    m_shedElements.Insert(ref  fId);

                    // This param need to be set for any level above Level 1 for the floor to move to the correct level
                    Revit.Parameter midFloorparam = f.get_Parameter(Revit.Parameters.BuiltInParameter.FLOOR_HEIGHTABOVELEVEL_PARAM);
                    midFloorparam.Set(0.0);

                    return f.Level.Id;
                }

                if (level.Name == "Level 1") {

                    Floor f = doc.NewFloor(profile, floorType, level, structural);

                    Revit.ElementId fId = f.Id;
                    m_shedElements.Insert(ref  fId);

                    return f.Level.Id;
                }

                // if none of the types match 
                return elemId;
            }
            catch (Exception e) {
                throw e;
            }
        }

        /// <summary>
        /// Swap any selected door with a hardwired double door.
        /// </summary>
        private void
        SimpleSwap ()
        {
            Revit.Symbols.FamilySymbol doubleDoorSymbol = null;

            Revit.Creation.Document    doc                  = m_revitApp.ActiveDocument.Create;
            Revit.Document             docu                 = m_revitApp.ActiveDocument;
            Revit.Selection            sel                  = m_revitApp.ActiveDocument.Selection;
            Revit.Creation.Application applic               = m_revitApp.Create;

            // first filter out to only Door elements
            Revit.ElementSet doorSet = Utils.Selection.FilterToCategory(m_revitApp.ActiveDocument.Selection.Elements,
                                                    Revit.BuiltInCategory.OST_Doors, false, m_revitApp.ActiveDocument);

            if (doorSet.IsEmpty) {
                MessageBox.Show("No door element is currently selected");
                return;
            }

            // Load the concerned door family        

            Revit.ElementIterator familySetIter = docu.get_Elements(typeof(Family));            
            while (familySetIter.MoveNext()) {
                Family fam = familySetIter.Current as Family;
                if (fam != null) {
                    if (fam.Name == "Double-Glass 1") {
                        doubleDoorSymbol = Utils.FamilyUtil.GetFamilySymbol(fam, "72\" x 78\"");
                    }
                }
            }

            string fileName = string.Empty;
            bool success = false;

            // Load the required family

            fileName = "../Data/Platform/Imperial/Library/Architectural/Doors/Double-Glass 1.rfa";
            Revit.Elements.Family doubleDoorFamily;

            if (doubleDoorSymbol == null) {
                success = m_revitApp.ActiveDocument.LoadFamily(fileName, out doubleDoorFamily);
                doubleDoorSymbol = Utils.FamilyUtil.GetFamilySymbol(doubleDoorFamily, "72\" x 78\"");
            }           

            if (doubleDoorSymbol == null) {
                MessageBox.Show("Please load Double-Glass 1 into project");
                Utils.UserInput.LoadFamily(null, m_revitApp.ActiveDocument);
            }

            // Perform the swap.
            Revit.ElementSet elemSet = sel.Elements;
            System.Collections.IEnumerator iters = elemSet.GetEnumerator();

            while (iters.MoveNext()) {
                FamilyInstance famInst = (FamilyInstance)iters.Current;
                famInst.Symbol = doubleDoorSymbol;
            }
        }


        /// <summary>
        /// Demonstrates the increase of floor to floor height by iterating over a set of available
        /// levels.
        /// </summary>
        public void
        SimpleLevelIteration ()
        {
            // Clear all related elements if a shed has already been created to prevent overlapping elements.
            m_revitApp.ActiveDocument.Delete(m_shedElements);

            // Use the simple shed sample to demonstrate the increase in floor to floor height.
            SimpleShed();

            DialogResult dlgRes = MessageBox.Show("This will increase the floor to floor height of the Simple Shed by 10 Ft \n \t\t Hit Cancel to see original shed.", "Test Framework", MessageBoxButtons.OKCancel, MessageBoxIcon.Information);

            if (dlgRes == System.Windows.Forms.DialogResult.OK) {

                ArrayList levels = new ArrayList();
                ArrayList elevations = new ArrayList();
                Hashtable hash = new Hashtable();

                Revit.ElementIterator systemElementIterator = m_revitApp.ActiveDocument.Elements;
                systemElementIterator.Reset();

                while (systemElementIterator.MoveNext()) {
                    Level systemLevel = systemElementIterator.Current as Level;
                    if (null != systemLevel) {
                        // collect all the levels
                        levels.Add(systemLevel);
                    }
                }


                // Polulate a hashtable with the [key. value] pair represented as [Elevation, Level]
                // Also populate a list of elevations.

                for (Int32 i = 0; i < levels.Count; i++) {

                    Level lev = (Level)levels[i];

                    if (!hash.ContainsKey(lev.Elevation) && !hash.ContainsValue(lev)) {
                        hash.Add(lev.Elevation, lev);
                    }

                    elevations.Add(lev.Elevation);
                }

                // Sort the elevations
                elevations.Sort();

                Double elevation = (Double)elevations[0];

                // Now that we have the elevations sorted, get the levels corresponding to them and bump 
                // them up by a desired value.

                for (Int32 z = 1; z < elevations.Count; z++) {

                    Double elev = (Double)elevations[z];

                    if (elev != 0.0) {
                        Level level = (Level)hash[elev];

                        elevation = elevation + 20.0;
                        level.Elevation = elevation;
                    }
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        public void
        Flip ()
        {
            Revit.ElementSet elemSet = m_revitApp.ActiveDocument.Selection.Elements;
            if (elemSet.IsEmpty) {
                MessageBox.Show("Please select element(s) to flip");
                return;
            }

            XYZ zDir = GeomUtils.kZAxis;
            XYZ flipPoint;
            Line zAxis = null;
            Revit.LocationCurve locCurve;
            Revit.LocationPoint locPt;
            bool success;

            Revit.ElementSetIterator elemSetIter = elemSet.ForwardIterator();
            while (elemSetIter.MoveNext()) {
                success = false;
                Revit.Element elem = elemSetIter.Current as Revit.Element;                
                Revit.Location location = elem.Location;

                // for elements like walls
                locCurve = location as Revit.LocationCurve;
                if (locCurve != null) {
                    XYZ startPt = locCurve.Curve.get_EndPoint(0);
                    XYZ endPt = locCurve.Curve.get_EndPoint(1);
                    double x = (endPt.X + startPt.X) / 2;
                    double y = (endPt.Y + startPt.Y) / 2;
                    flipPoint = new XYZ(x, y, 0);
                    zAxis = m_revitApp.Create.NewLineUnbound(flipPoint, zDir);
                }

                // for familyinstances
                locPt = location as Revit.LocationPoint;
                if (locPt != null) {
                    flipPoint = locPt.Point;
                    zAxis = m_revitApp.Create.NewLineUnbound(flipPoint, zDir);
                }

                success = elem.Location.Rotate(zAxis, GeomUtils.kRad180);
            }
        }

        /// <summary>
        /// Mirrors a selected modscope
        /// </summary>
        public void 
        Mirror ()
        {
            Revit.ElementSet elemSet = m_revitApp.ActiveDocument.Selection.Elements;

            if (elemSet.IsEmpty) {
                MessageBox.Show("Please select elements to be mirrored");
                return;
            }

            /// a map to hold original elementids and their corresponding
            /// clone elements
            Hashtable origAndCopy = new Hashtable();

            Revit.Elements.HostObject hostObj;
            Revit.LocationCurve locationCurve = null;
            Revit.ElementSetIterator elemSetIter = elemSet.ForwardIterator();

            /// first pass will create only host elements, because without them
            /// any hosted elements cannot be created 
            while (elemSetIter.MoveNext()) {
                Revit.Element elem = elemSetIter.Current as Revit.Element;

                /// there are only 4 host objects, of which only wall 
                /// and floor can be created for now
                hostObj = elem as Revit.Elements.HostObject;
                if (hostObj != null) {

                    Floor floor = elem as Revit.Elements.Floor;
                    if (floor != null) {
                        // get geometry to figure out location of floor
                        Options options = m_revitApp.Create.NewGeometryOptions();
                        options.DetailLevel = Options.DetailLevels.Coarse;
                        Revit.Geometry.Element geomElem = floor.get_Geometry(options);
                        GeometryObjectArray geomObjArray = geomElem.Objects;
                        Solid solid = geomObjArray.get_Item(0) as Solid;
                        double absoluteElev = floor.Level.Elevation + floor.get_Parameter(Revit.Parameters.BuiltInParameter.FLOOR_HEIGHTABOVELEVEL_PARAM).AsDouble();
                        CurveArray curveArray = Utils.Geometry.GetProfile(solid, absoluteElev, m_revitApp);

                        // mirror the location
                        CurveArray mirroredCurveArray = m_revitApp.Create.NewCurveArray();
                        for (int i = 0; i < curveArray.Size; i++) {
                            mirroredCurveArray.Append(Utils.Geometry.Mirror(curveArray.get_Item(i), GeomUtils.kXAxis, m_revitApp));
                        }

                        // create the mirrored floor
                        Floor floorClone = m_revitApp.ActiveDocument.Create.NewFloor(mirroredCurveArray, floor.FloorType, floor.Level, false);

                        // give mirrored floor same param values as original floor
                        Utils.ParamUtil.SetParameters(floorClone.Parameters, floor.Parameters);

                        // save the original to mirrored floor mapping
                        origAndCopy.Add(floor.Id.Value, floorClone);
                        continue;
                    }

                    Revit.Element elemClone = Utils.Elements.CloneElement(m_revitApp, elem);
                    Revit.Location elemLocation = elemClone.Location;

                    if (elemLocation != null) {
                        locationCurve = elemLocation as Revit.LocationCurve;
                        if (locationCurve != null) {
                            locationCurve.Curve = Utils.Geometry.Mirror(locationCurve.Curve, GeomUtils.kXAxis, m_revitApp);
                        }
                    }

                    // save the mapping
                    origAndCopy.Add(elem.Id.Value, elemClone);
                }
            }

            elemSetIter.Reset();


            Revit.LocationPoint locationPoint = null;

            /// second pass will put in the family instances. If it is hosted on an element
            /// extract that info from the map and use it
            while (elemSetIter.MoveNext()) {
                Revit.Element elem = elemSetIter.Current as Revit.Element;

                hostObj = elem as Revit.Elements.HostObject;
                if (hostObj == null) {

                    Revit.Elements.FamilyInstance famInst = elem as Revit.Elements.FamilyInstance;
                    if (famInst != null) {

                        XYZ pt = new XYZ();

                        // get location of familyinstance
                        locationPoint = famInst.Location as Revit.LocationPoint;
                        if (locationPoint != null) {
                            // mirror the location
                            pt = Utils.Geometry.Mirror(locationPoint.Point, GeomUtils.kXAxis);
                        }
                        locationCurve = famInst.Location as Revit.LocationCurve;
                        if (locationCurve != null) {
                            // mirror the location
                            pt = Utils.Geometry.Mirror(locationCurve.Curve.get_EndPoint(0), GeomUtils.kXAxis);
                        }

                        // if family instance is hosted get it from the map
                        Revit.Element elemHost = null;
                        if (famInst.Host != null) {
                            elemHost = origAndCopy[famInst.Host.Id.Value] as Revit.Element;
                        }

                        // create mirrored family instance
                        FamilyInstance famInstClone = m_revitApp.ActiveDocument.Create.NewFamilyInstance(pt, famInst.Symbol, elemHost, famInst.Level, famInst.StructuralType);

                        // give mirrored family instance same param values as original family instance
                        Utils.ParamUtil.SetParameters(famInstClone.Parameters, famInst.Parameters);
                        continue;
                    }

                    Revit.Elements.Grid grid = elem as Revit.Elements.Grid;
                    if (grid != null) {
                        Curve curve = grid.Curve;

                        curve = Utils.Geometry.Mirror(curve, GeomUtils.kXAxis, m_revitApp);

                        Line line = curve as Line;
                        if (line != null) {
                            m_revitApp.ActiveDocument.Create.NewGrid(line);
                        }

                        Arc arc = curve as Arc;
                        if (arc != null) {
                            m_revitApp.ActiveDocument.Create.NewGrid(arc);
                        }
                        continue;
                    }

                    Revit.Elements.Opening opening = elem as Revit.Elements.Opening;
                    if (opening != null) {
                        // get host from the map
                        Revit.Element elemHost = null;
                        if (opening.Host != null) {
                            elemHost = origAndCopy[opening.Host.Id.Value] as Revit.Element;
                        }

                        CurveArray curveArray = opening.BoundaryCurves;
                        // mirror the location
                        CurveArray mirroredCurveArray = m_revitApp.Create.NewCurveArray();
                        for (int i = 0; i < curveArray.Size; i++) {
                            mirroredCurveArray.Append(Utils.Geometry.Mirror(curveArray.get_Item(i), GeomUtils.kXAxis, m_revitApp));
                        }

                        m_revitApp.ActiveDocument.Create.NewOpening(elemHost, mirroredCurveArray, true);
                        continue;
                    }

                    // TBD: Dimension.Curve always throws an exception
                    //Revit.Elements.Dimension dimension = elem as Revit.Elements.Dimension;
                    //if (dimension != null) {
                    //    Curve curve = dimension.Curve;
                    //    curve = Utils.Geometry.Mirror(curve, GeomUtils.kXAxis, m_revitApp);

                    //    Line line = curve as Line;
                    //    if (line != null) {
                    //        m_revitApp.ActiveDocument.Create.NewDimension(dimension.View, line, dimension.References, dimension.DimensionType);
                    //    }
                    //    continue;
                    //}


                    Revit.Element elementClone = Utils.Elements.CloneElement(m_revitApp, elem);
                    if (elementClone != null) {

                        Revit.Elements.BeamSystem beamSystem = elementClone as Revit.Elements.BeamSystem;
                        if (beamSystem != null) {
                            CurveArray curveArray = beamSystem.Profile;
                            // mirror the location
                            CurveArray mirroredCurveArray = m_revitApp.Create.NewCurveArray();
                            for (int i = 0; i < curveArray.Size; i++) {
                                mirroredCurveArray.Append(Utils.Geometry.Mirror(curveArray.get_Item(i), GeomUtils.kXAxis, m_revitApp));
                            }
                            beamSystem.Profile = mirroredCurveArray;
                        }

                        Revit.Elements.ModelCurve modelCurve = elementClone as ModelCurve;
                        if (modelCurve != null) {
                            Curve curve = modelCurve.GeometryCurve;
                            curve = Utils.Geometry.Mirror(curve, GeomUtils.kXAxis, m_revitApp);
                            modelCurve.GeometryCurve = curve;
                        }

                        Revit.Location elemLocation = elementClone.Location;
                        if (elemLocation != null) {
                            locationCurve = elemLocation as Revit.LocationCurve;
                            if (locationCurve != null) {
                                locationCurve.Curve = Utils.Geometry.Mirror(locationCurve.Curve, GeomUtils.kXAxis, m_revitApp);
                            }
                            locationPoint = elemLocation as Revit.LocationPoint;
                            if (locationPoint != null) {
                                locationPoint.Point = Utils.Geometry.Mirror(locationPoint.Point, GeomUtils.kXAxis);
                            }
                        }

                        continue;
                    }
                }
            }
        }

        /// <summary>
        /// Creates a new annotation symbol
        /// </summary>
        public void
        AnnoSymbol ()
        {
            Revit.Document doc = m_revitApp.ActiveDocument;

            FamilyItemFactory famFact = doc.FamilyCreate;            

            Revit.Symbols.AnnotationSymbolTypeSet annoSymTypeSet = doc.AnnotationSymbolTypes;  
            // TBD: why is the size only 2
            Int32 size = annoSymTypeSet.Size;

            Revit.Symbols.AnnotationSymbolTypeSetIterator annoSymTypeSetIter = annoSymTypeSet.ForwardIterator();

            Revit.Symbols.AnnotationSymbolType annoSymType = null;        

            while (annoSymTypeSetIter.MoveNext())
            {
                Revit.Symbols.AnnotationSymbolType tempAnnoSymType = annoSymTypeSetIter.Current as Revit.Symbols.AnnotationSymbolType;

                if (null != tempAnnoSymType)
                {
                    if (tempAnnoSymType.Name == "North Arrow 2")
                    {
                        annoSymType = tempAnnoSymType;
                    }                    
                }
            }

            if (annoSymType == null)
                return;

            XYZ location = GeomUtils.kOrigin;
            Revit.Elements.View view = m_revitApp.ActiveDocument.ActiveView;
            
            doc.Create.NewAnnotationSymbol(location, annoSymType, view);            
        }        

        /// <summary>
        /// Creates a new Beam System
        /// </summary>
        public void
        BeamSystemHardWired ()
        {
            CurveArray curveArray = m_revitApp.Create.NewCurveArray();

            XYZ location1 = GeomUtils.kOrigin;
            XYZ location2 = new XYZ(0.0, 10.0, 0.0);
            XYZ location3 = new XYZ(10.0, 10.0, 0.0);
            XYZ location4 = new XYZ(10.0, 0.0, 0.0);

            curveArray.Append(m_revitApp.Create.NewLine(location1, location2, true));
            curveArray.Append(m_revitApp.Create.NewLine(location2, location3, true));
            curveArray.Append(m_revitApp.Create.NewLine(location3, location4, true));
            curveArray.Append(m_revitApp.Create.NewLine(location4, location1, true));

            XYZ normal = GeomUtils.kZAxis;
            XYZ origin = GeomUtils.kOrigin;

            Plane plane = m_revitApp.Create.NewPlane(normal, origin);
            
            SketchPlane sketchPlane = m_revitApp.ActiveDocument.Create.NewSketchPlane(plane);
            m_revitApp.ActiveDocument.Create.NewBeamSystem(curveArray, sketchPlane, new XYZ(1.0, 1.0, 0.0), false);           
        }

        /// <summary>
        /// Creates new Detail Curves
        /// </summary>
        public void
        DetailCurveHardWired ()
        {
            CurveArray curveArray = m_revitApp.Create.NewCurveArray();

            //"J"
            XYZ location1 = GeomUtils.kOrigin;
            XYZ location2 = new XYZ(0.0, -20.0, 0.0);
            XYZ location3 = new XYZ(-10.0, -20.0, 0.0);
            XYZ ptOnCurve = new XYZ(-5.0, -25.0, 0.0);
            curveArray.Append(m_revitApp.Create.NewLine(location1, location2, true));
            curveArray.Append(m_revitApp.Create.NewArc(location2, location3, ptOnCurve));

            // "a"
            XYZ center = new XYZ(10.0, -17.5, 0.0);
            XYZ xVec = GeomUtils.kXAxis;
            XYZ yVec = GeomUtils.kYAxis;
            Ellipse ellipse = m_revitApp.Create.NewEllipse(center, 5.0, 8.0, xVec, yVec, 0.0, (GeomUtils.kRevitPi * 2.0));
            // the curve should be bound otherwise NewDetailCurveArray will reject it
            ellipse.MakeBound(0.0, GeomUtils.kRevitPi * 2.0);
            curveArray.Append(ellipse);

            XYZ end0 = new XYZ(15.0, -17.5, 0.0);
            XYZ end1 = new XYZ(18.0, -25.5, 0.0);
            ptOnCurve = new XYZ(15.5, -23.0, 0.0);
            curveArray.Append(m_revitApp.Create.NewArc(end0, end1, ptOnCurve));

            // "i"
            XYZ start = new XYZ(22.0, -12.5, 0.0);
            XYZ end = new XYZ(22.0, -25.0, 0.0);
            curveArray.Append(m_revitApp.Create.NewLine(start, end, true));

            XYZ normal = GeomUtils.kZAxis;
            XYZ origin = GeomUtils.kOrigin;

            Plane plane = m_revitApp.Create.NewPlane(normal, origin);
            SketchPlane sketchPlane = m_revitApp.ActiveDocument.Create.NewSketchPlane(plane);

            m_revitApp.ActiveDocument.Create.NewDetailCurveArray(m_revitApp.ActiveDocument.ActiveView,
                                                                curveArray);
        }

        /// <summary>
        /// Creates 2 hardwired lines and a dimension to measure 
        /// the distance between them
        /// </summary>
        public void
        DimensionHardWired ()
        {
            XYZ location1 = GeomUtils.kOrigin;
            XYZ location2 = new XYZ(20.0, 0.0, 0.0);
            XYZ location3 = new XYZ(0.0, 20.0, 0.0);
            XYZ location4 = new XYZ(20.0, 20.0, 0.0);

            Curve curve1 = m_revitApp.Create.NewLine(location1, location2, true);
            Curve curve2 = m_revitApp.Create.NewLine(location3, location4, true);

            DetailCurve dCurve1 = null;
            DetailCurve dCurve2 = null;

            if (!m_revitApp.ActiveDocument.IsFamilyDocument) {
                dCurve1 = m_revitApp.ActiveDocument.Create.NewDetailCurve(m_revitApp.ActiveDocument.ActiveView,
                                                                curve1);
                dCurve2 = m_revitApp.ActiveDocument.Create.NewDetailCurve(m_revitApp.ActiveDocument.ActiveView,
                                                                curve2);
            }
            else {
                if (null != m_revitApp.ActiveDocument.OwnerFamily && null != m_revitApp.ActiveDocument.OwnerFamily.FamilyCategory) {
                    if (!m_revitApp.ActiveDocument.OwnerFamily.FamilyCategory.Name.Contains("Detail")) {
                        MessageBox.Show("Please make sure you open a detail based family template.", "RvtMgdDbg", MessageBoxButtons.OK, MessageBoxIcon.Information);
                        return;
                    }
                }
                dCurve1 = m_revitApp.ActiveDocument.FamilyCreate.NewDetailCurve(m_revitApp.ActiveDocument.ActiveView,
                                                                curve1);
                dCurve2 = m_revitApp.ActiveDocument.FamilyCreate.NewDetailCurve(m_revitApp.ActiveDocument.ActiveView,
                                                                curve2);
            }

            Line line = m_revitApp.Create.NewLine(location2, location4, true);

            ReferenceArray refArray = new ReferenceArray();
            /// TBD: all curves return a null Reference, what am I missing?
            /// 01/12/07
            refArray.Append(dCurve1.GeometryCurve.Reference);
            refArray.Append(dCurve2.GeometryCurve.Reference);

            if (!m_revitApp.ActiveDocument.IsFamilyDocument) {
                m_revitApp.ActiveDocument.Create.NewDimension(m_revitApp.ActiveDocument.ActiveView,
                    line, refArray);
            }
            else {
                m_revitApp.ActiveDocument.FamilyCreate.NewDimension(m_revitApp.ActiveDocument.ActiveView,
                  line, refArray);
            }
        }

        /// <summary>
        /// Creates hardwired foundation slabs
        /// </summary>
        public void
        FoundationSlabHardWired ()
        {
            CurveArray curveArray = m_revitApp.Create.NewCurveArray();

            XYZ location1 = GeomUtils.kOrigin;
            XYZ location2 = new XYZ(0.0, 20.0, 0.0);
            XYZ location3 = new XYZ(20.0, 20.0, 0.0);
            XYZ location4 = new XYZ(20.0, 0.0, 0.0);

            curveArray.Append(m_revitApp.Create.NewLine(location1, location2, true));
            curveArray.Append(m_revitApp.Create.NewLine(location2, location3, true));
            curveArray.Append(m_revitApp.Create.NewLine(location3, location4, true));
            curveArray.Append(m_revitApp.Create.NewLine(location4, location1, true));

            FloorTypeSet floorTypeSet = m_revitApp.ActiveDocument.FloorTypes;
            FloorTypeSetIterator floorTypeSetIter = floorTypeSet.ForwardIterator();
            FloorType floorType = null;

            while (floorTypeSetIter.MoveNext()) {
                FloorType floorTypeTemp = floorTypeSetIter.Current as FloorType;
                if (floorTypeTemp.Name == "6\" Foundation Slab") {
                    floorType = floorTypeTemp;
                    break;
                }

            }

            Level level = m_revitApp.ActiveDocument.ActiveView.GenLevel;
            XYZ normal = GeomUtils.kZAxis;

            /// create a slab
            m_revitApp.ActiveDocument.Create.NewFoundationSlab(curveArray, floorType, level, false, normal);

            /// floor slab is below all levels and is not visible in floor plan view!
            if (m_revitApp.ActiveDocument.ActiveView.ViewType != Revit.Enums.ViewType.ThreeD
                && m_revitApp.ActiveDocument.ActiveView.ViewType != Revit.Enums.ViewType.Elevation)
                MessageBox.Show("Foundation slab created. Go to 3D or Elevation views to view");
        }

        /// <summary>
        /// Creates a hardwired text note with a straight leader
        /// </summary>
        public void
        TextNoteHardWired ()
        {
            Autodesk.Revit.Elements.View view = m_revitApp.ActiveDocument.ActiveView;

            /// create a wall to point to
            XYZ location1 = new XYZ(21.0, 10.0, 0.0);
            XYZ location2 = new XYZ(21.0, -10.0, 0.0);

            Line line = m_revitApp.Create.NewLine(location1, location2, true);
            m_revitApp.ActiveDocument.Create.NewWall(line, m_revitApp.ActiveDocument.ActiveView.GenLevel, true);

            /// align text middle and center
            Revit.Enums.TextAlignFlags align = Revit.Enums.TextAlignFlags.TEF_ALIGN_MIDDLE ^ Revit.Enums.TextAlignFlags.TEF_ALIGN_CENTER;
            TextNote txtNote = m_revitApp.ActiveDocument.Create.NewTextNote(view, GeomUtils.kOrigin, GeomUtils.kXAxis,
                                                                            view.ViewDirection, .25,
                                                                            align, "Simple wall");
            /// add a straight leader
            txtNote.AddLeader(Revit.Enums.TextNoteLeaderTypes.TNLT_STRAIGHT_R);
        }


        /// <summary>
        /// Creates a hardwired slab.
        /// </summary>
        public void
        SimpleSlab ()
        {
            XYZ startPt = new XYZ(20.0, 0.0, 0.0);            
            XYZ endPt = new XYZ(20.0, 20.0, 0.0);

            CurveArray curveArray = m_revitApp.Create.NewCurveArray();

            XYZ location1 = GeomUtils.kOrigin;
            XYZ location2 = new XYZ(0.0, 10.0, 0.0);
            XYZ location3 = new XYZ(10.0, 10.0, 0.0);
            XYZ location4 = new XYZ(10.0, 0.0, 0.0);

            curveArray.Append(m_revitApp.Create.NewLine(location1, location2, true));
            curveArray.Append(m_revitApp.Create.NewLine(location2, location3, true));
            curveArray.Append(m_revitApp.Create.NewLine(location3, location4, true));
            curveArray.Append(m_revitApp.Create.NewLine(location4, location1, true));            

            Level level = m_revitApp.ActiveDocument.ActiveView.GenLevel;
            Line line = m_revitApp.Create.NewLine(startPt, endPt, true);
            
            m_revitApp.ActiveDocument.Create.NewSlab(curveArray, level, line, 45.0, false);
        }


        /// <summary>
        /// 
        /// </summary>
        public void
        SimpleViewSection ()
        {                        
            Revit.Document doc = m_revitApp.ActiveDocument;

            Random rnd = new Random();
            String viewName = "RvtMgdDbg_ViewSection" + rnd.Next().ToString();

            ViewSection viewSec = m_revitApp.ActiveDocument.Create.NewViewSection(doc.ActiveView.CropBox);
            viewSec.ViewName = viewName;                      
        }

        /// <summary>
        ///  
        /// </summary>
        public void
        FloorViewPlan ()
        {
            Revit.Creation.Document doc = m_revitApp.ActiveDocument.Create;

            // Just appending a random number for uniqueness. An exception will be thrown if the view name is not unique.
            Random rnd = new Random();
            String viewName = "RvtMgdDbg_Plan" + rnd.Next().ToString();

            m_revitApp.ActiveDocument.Create.NewViewPlan(viewName, doc.NewLevel(10.0), Autodesk.Revit.Enums.ViewType.FloorPlan);
        }


        /// <summary>
        /// Add the Level 1 (type = Floor Plan) view to a newly created sheet.
        /// </summary>
        public void
        ViewToNewSheetHardwired ()
        {
            Revit.Document doc = m_revitApp.ActiveDocument;

            FamilySymbol titleBlock = null;

            // Collect all available views
            ViewSet allViews = Utils.View.GetAllViews(doc);

            // Collection of views that have been added to existing sheets.
            ViewSet usedViews = new ViewSet();

            // Collect all the existing sheets. It would be nice to have some API to do this.
            Revit.ElementSet allSheets = Utils.View.GetAllSheets(doc);
            
            foreach (FamilySymbol f in doc.TitleBlocks)
            {
                if (null == titleBlock)
                {
                    titleBlock = f;
                }
            }            

            // Create a new sheet
            ViewSheet sheet = doc.Create.NewViewSheet(titleBlock);           

            Random rnd = new Random();
            String name = "RvtMgdDbg_Sheet" + rnd.Next().ToString();
            sheet.Name = name;

            foreach (ViewSheet viewSheet in allSheets)
            {
                foreach (Autodesk.Revit.Elements.View usedView in viewSheet.Views)
                {
                    usedViews.Insert(usedView);
                }
            }

            // Note: The application does not allow the addition of a view to a new sheet if that view has already been added to
            // an exixting sheet. I realized this when a programmatic attempt to do so resulted in an exception being thrown.
            // So running this hardwired test a second time would result in an exception (As "Level 1 (FloorPlan) would have already been added)". 
            // The workaround that has been used here is that first all the available sheets and their added views are gathered.
            // Before adding a view it is compared with the used views. arj - 1/17/07
            //            
            foreach (Autodesk.Revit.Elements.View view in allViews)
            {
                // Has been hardwired for the following view.
                if ((view.ViewName == "Level 1") && (view.ViewType == Autodesk.Revit.Enums.ViewType.FloorPlan))
                {                    
                    UV point = view.Outline.Max;
                   
                    // Display notification message if view is already being used.
                    foreach (Autodesk.Revit.Elements.View usedView in usedViews)
                    {
                        if ((usedView.ViewName == view.ViewName) && (usedView.ViewType == view.ViewType))
                        {
                            MessageBox.Show("View already added to an existing sheet", "RvtMgdDbg Test", MessageBoxButtons.OK, MessageBoxIcon.Information);                           
                            return;
                        }                        
                    }
                    sheet.AddView(view, point);                                                                 
                }
            }         
        }     
        

        /// <summary>
        /// Add a simple tag to selected elements.
        /// </summary>
        public void
        SimpleTag ()
        {          
            Revit.Document docu = m_revitApp.ActiveDocument;
            Revit.SelElementSet elemSet = m_revitApp.ActiveDocument.Selection.Elements;

            Revit.Elements.View view = m_revitApp.ActiveDocument.ActiveView;           

            XYZ tagLocation = new XYZ();
            
            Revit.ElementSetIterator elemIter = elemSet.ForwardIterator();
            while (elemIter.MoveNext())
            {
                Revit.Element elem = elemIter.Current as Revit.Element;
                Revit.Location location = elem.Location;

                // for elements like walls
                Revit.LocationCurve locCurve = location as Revit.LocationCurve;
                if (null != locCurve)
                {
                    XYZ startPt = locCurve.Curve.get_EndPoint(0);
                    XYZ endPt = locCurve.Curve.get_EndPoint(1);
                    double x = (endPt.X + startPt.X) / 2;
                    double y = (endPt.Y + startPt.Y) / 2;
                    tagLocation = new XYZ(x, y, 0);                   
                }

                // for familyinstances
                Revit.LocationPoint locPt = location as Revit.LocationPoint;
                if (null != locPt)
                {
                    tagLocation = locPt.Point;                    
                }                

                // Add the tag.
                if (null != elem)
                {
                    m_revitApp.ActiveDocument.Create.NewTag(view, elem, true, Autodesk.Revit.Enums.TagMode.TM_ADDBY_CATEGORY, Autodesk.Revit.Enums.TagOrientation.TAG_HORIZONTAL, tagLocation);            
                }
            }           
        }

        /// <summary>
        /// Modify the width of a wall by changing the layer thickness.
        /// </summary>
        public void
        ModifyWallWidth ()
        {
            Revit.Document         doc           = m_revitApp.ActiveDocument;
            Revit.Symbols.WallType outerWallType = null;

            Revit.ElementIterator wallTypesSetIter = doc.get_Elements(typeof(Revit.Symbols.WallType));

            while (wallTypesSetIter.MoveNext()) {
                Revit.Symbols.WallType wallType = wallTypesSetIter.Current as Revit.Symbols.WallType;

                if (wallType != null) {
                    // hardwired to affect a Generic - 8\" Wall Type 
                    if (wallType.Name == "Generic - 8\"") {
                        outerWallType = wallType;
                    }
                }
            }
            Boolean success = UpdateHostCompoudStructures(outerWallType);
        }


        /// <summary>
        /// 
        /// </summary>
        /// <param name="symbol"></param>
        /// <returns></returns>
        private Boolean
        UpdateHostCompoudStructures (Revit.Symbol symbol)
        {
            HostObjAttributes host = symbol as HostObjAttributes;

            if (host == null) return false;          

            foreach (Revit.Structural.CompoundStructureLayer layer in host.CompoundStructure.Layers) {        
                layer.Thickness = 15.0; // this should change the width of the wall.
            }
            return true;
        }
    }
}
