
//
// (C) Copyright 2006 by Autodesk, Inc. 
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted, 
// provided that the above copyright notice appears in all copies and 
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting 
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC. 
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to 
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//

using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Reflection;
using System.Diagnostics;
using System.Windows.Forms;

using Autodesk.Revit;
using Autodesk.Revit.Geometry;
using Autodesk.Revit.Elements;

using Revit = Autodesk.Revit;

 

namespace RvtMgdDbg.Test {
    class TestImportExport : RvtMgdDbgTestFuncs {
    
        public
        TestImportExport(Autodesk.Revit.Application app)
            : base(app)
        {
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Export to DWG", "Write out a set of Views to DWG format", "Import/Export", new RvtMgdDbgTestFuncInfo.TestFunc(ExportToDwg), RvtMgdDbgTestFuncInfo.TestType.Other));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Export to 2D DWF", "Write out a set of Views to 2D DWF format", "Import/Export", new RvtMgdDbgTestFuncInfo.TestFunc(ExportTo2dDwf), RvtMgdDbgTestFuncInfo.TestType.Other));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Export to 3D DWF", "Write out a set of Views to 3D DWF format", "Import/Export", new RvtMgdDbgTestFuncInfo.TestFunc(ExportTo3dDwf), RvtMgdDbgTestFuncInfo.TestType.Other));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Import DWG", "Import a file of type DWG", "Import/Export", new RvtMgdDbgTestFuncInfo.TestFunc(ImportDwg), RvtMgdDbgTestFuncInfo.TestType.Other));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Import Image", "Import an Image file", "Import/Export", new RvtMgdDbgTestFuncInfo.TestFunc(ImportImage), RvtMgdDbgTestFuncInfo.TestType.Other));
 
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Publish XML", "Write out an XML file with project data", "Import/Export", new RvtMgdDbgTestFuncInfo.TestFunc(PublishToXML), RvtMgdDbgTestFuncInfo.TestType.Other));
            m_testFuncs.Add(new RvtMgdDbgTestFuncInfo("Snoop XML DOM Document", "Exercise XML DOM API", "Import/Export", new RvtMgdDbgTestFuncInfo.TestFunc(SnoopXML), RvtMgdDbgTestFuncInfo.TestType.Query));
 
        }        

        public void
        ExportToDwg()
		{
				// get output dir to save DWGs to
			System.Windows.Forms.FolderBrowserDialog dbox = new System.Windows.Forms.FolderBrowserDialog();
			dbox.Description = "Folder to save exported DWG files to";
            dbox.ShowNewFolderButton = true;

            if (dbox.ShowDialog() == DialogResult.OK) {
                ViewSet viewsToExport = Utils.View.GetAvailableViewsToExport(m_revitApp.ActiveDocument);

                DWGExportOptions opts = new DWGExportOptions();

                // The ACIS option has supposedly been fixed, will have to verify this with the latest code 
                // at a later time.
                opts.ExportOfSolids = Autodesk.Revit.Enums.SolidGeometry.ACIS;
                opts.TargetUnit = Autodesk.Revit.Enums.ExportUnit.Millimeter;

                m_revitApp.ActiveDocument.Export(dbox.SelectedPath, "", viewsToExport, opts);
            }
		}

        public void
        ExportTo2dDwf ()
        {
            /// get output dir to save DWFs to
            FolderBrowserDialog dbox = new FolderBrowserDialog();
            dbox.Description = "Folder to save exported DWF files to";
            dbox.ShowNewFolderButton = true;

            try {
                if (dbox.ShowDialog() == DialogResult.OK) {

                    ViewSet viewsToExport = Utils.View.GetAvailableViewsToExport(m_revitApp.ActiveDocument);

                    DWF2DExportOptions opts = new DWF2DExportOptions();
                    opts.MergedViews = true;

                    /// export now
                    m_revitApp.ActiveDocument.Export(dbox.SelectedPath, "", viewsToExport, opts);

                    /// feedback to user
                    MessageBox.Show("Done exporting to 2d Dwf!!");
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.Message);
            }
        }

        public void
        ExportTo3dDwf ()
        {
            // get output dir to save DWFs to
            System.Windows.Forms.FolderBrowserDialog dbox = new System.Windows.Forms.FolderBrowserDialog();
            dbox.Description = "Folder to save exported DWF files to";
            dbox.ShowNewFolderButton = true;

            try {
                if (dbox.ShowDialog() == DialogResult.OK) {

                    ViewSet viewsToExport = Utils.View.GetAvailableViewsToExport(m_revitApp.ActiveDocument);

                    /// filter out only 3d views
                    ViewSet views3dToExport = new ViewSet();
                    foreach (Revit.Elements.View view in viewsToExport) {
                        if (view.ViewType == Revit.Enums.ViewType.ThreeD) {
                            views3dToExport.Insert(view);
                        }
                    }

                    if (views3dToExport.Size == 0)
                        throw new Exception("No files exported. Make sure you have atleast one 3d view.");

                    DWF3DExportOptions opts = new DWF3DExportOptions();
                    /// export now
                    m_revitApp.ActiveDocument.Export(dbox.SelectedPath, "", views3dToExport, opts);

                    /// feedback to user
                    MessageBox.Show("Done exporting to 3d Dwf!!");
                }
            }
            catch (Exception e) {
                MessageBox.Show(e.Message);
            }
        }

        public void
        ImportDwg()
		{
				// get input file of type DWG
			System.Windows.Forms.OpenFileDialog dbox = new System.Windows.Forms.OpenFileDialog();
			dbox.CheckFileExists = true;
			dbox.Multiselect = false;
			dbox.AddExtension = true;
			dbox.DefaultExt = "dwg";
			dbox.Filter = "DWG Files (*.dwg)|*.dwg";
			dbox.Title = "DWG file to import";
            
			if (dbox.ShowDialog() == DialogResult.OK) {
                DWGImportOptions opts = new DWGImportOptions();
                opts.Placement = Autodesk.Revit.Enums.ImportPlacement.Origin;

                Revit.Element newElement;
                m_revitApp.ActiveDocument.Import(dbox.FileName, opts, out newElement);
			}
		}

        public void
        ImportImage()
		{
				// get input file of type DWG
			System.Windows.Forms.OpenFileDialog dbox = new System.Windows.Forms.OpenFileDialog();
			dbox.CheckFileExists = true;
			dbox.Multiselect = false;
			dbox.AddExtension = true;
			dbox.DefaultExt = "jpg";
			dbox.Filter = "Image Files (*.jpg;*.png;*.bmp)|*.jpg;*.png;*.bmp";
			dbox.Title = "Image file to import";
            
			if (dbox.ShowDialog() == DialogResult.OK) {
                ImageImportOptions opts = new ImageImportOptions();
                opts.Placement = Autodesk.Revit.Enums.BoxPlacement.BottomLeft;

                Revit.Element newElement;
                m_revitApp.ActiveDocument.Import(dbox.FileName, opts, out newElement);
			}
		}

 
        
        public void
        PublishToXML()
		{
				// get output file to save XML report to
			System.Windows.Forms.SaveFileDialog dbox = new System.Windows.Forms.SaveFileDialog();
			dbox.CreatePrompt = false;
			dbox.OverwritePrompt = true;
			dbox.AddExtension = true;
			dbox.DefaultExt = "xml";
			dbox.Filter = "XML Files (*.xml)|*.xml";
			dbox.Title = "XML file to save report as";
            
			if (dbox.ShowDialog() == DialogResult.OK) {
				ModelStats.Report statReport = new ModelStats.Report();
                statReport.XmlReport(dbox.FileName, m_revitApp.ActiveDocument);
			}
		}
		
		public void
		SnoopXML()
		{ 
			System.Windows.Forms.OpenFileDialog dbox = new System.Windows.Forms.OpenFileDialog();
			dbox.CheckFileExists = true;
			dbox.AddExtension = true;
			dbox.DefaultExt = "xml";
			dbox.Filter = "XML Files (*.xml)|*.xml";
			dbox.Multiselect = false;
			dbox.Title = "Select an XML file";
	        
			if (dbox.ShowDialog() == DialogResult.OK) {
				try {
					System.Xml.XmlDocument xmlDoc = new System.Xml.XmlDocument();
					xmlDoc.Load(dbox.FileName);
	            
					Xml.Forms.Dom form = new Xml.Forms.Dom(xmlDoc);
					form.ShowDialog();
				}
				catch (System.Xml.XmlException e) {
					MessageBox.Show(e.Message, "XML Exception");
				}
			}
		}

 
    }
}
