
//
// (C) Copyright 2005 by Autodesk, Inc. 
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted, 
// provided that the above copyright notice appears in all copies and 
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting 
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC. 
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to 
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//

using System;
using System.Windows.Forms;
using System.Collections;
using System.Data;
//using System.IO;
//using System.Reflection;
//using System.Diagnostics;

using Autodesk.Revit;
using RvtMgdDbg.Test;

// Each command is implemented as a class that provides the IExternalCommand Interface
//

namespace RvtMgdDbg
{

	/// <summary>
	/// SnoopDB command:  Browse all Elements in the current Document
	/// </summary>
 
	public class CmdSnoopDb : IExternalCommand
	{
		public IExternalCommand.Result
        Execute (Autodesk.Revit.ExternalCommandData cmdData, ref string msg, ElementSet elems)
		{
		    IExternalCommand.Result result;

			try {
				Snoop.CollectorExts.CollectorExt.m_app = cmdData.Application;	// TBD: see note in CollectorExt.cs
				
				    // iterate over the collection and put them in an ArrayList so we can pass on
				    // to our Form
			    ElementIterator iter = cmdData.Application.ActiveDocument.Elements;
			    if (iter != null) {
			        ArrayList objs = new ArrayList();
				    while (iter.MoveNext())
					    objs.Add(iter.Current);

                    Snoop.Forms.Objects form = new Snoop.Forms.Objects(objs);
				    form.ShowDialog();
			    }
                result = IExternalCommand.Result.Succeeded;
			}
            catch (System.Exception e) {
                msg = e.Message;
                result = IExternalCommand.Result.Failed;
            }

			return result;
		}
	}

	/// <summary>
	/// Snoop ModScope command:  Browse all Elements in the current selection set
	/// </summary>
	
	public class CmdSnoopModScope : IExternalCommand
	{
		public IExternalCommand.Result
        Execute (Autodesk.Revit.ExternalCommandData cmdData, ref string msg, ElementSet elems)
		{
            IExternalCommand.Result result;

			try {
				Snoop.CollectorExts.CollectorExt.m_app = cmdData.Application;	// TBD: see note in CollectorExt.cs

                Snoop.Forms.Objects form = new Snoop.Forms.Objects(cmdData.Application.ActiveDocument.Selection.Elements);
				form.ShowDialog();
				
                result = IExternalCommand.Result.Succeeded;
			}
            catch (System.Exception e) {
                msg = e.Message;
                result = IExternalCommand.Result.Failed;
            }

			return result;
		}
	}

	/// <summary>
	/// Snoop App command:  Browse all objects that are part of the Application object
	/// </summary>
	
	public class CmdSnoopApp : IExternalCommand
	{
		public IExternalCommand.Result
        Execute (Autodesk.Revit.ExternalCommandData cmdData, ref string msg, ElementSet elems)
		{
            IExternalCommand.Result result;

			try {
                Snoop.CollectorExts.CollectorExt.m_app = cmdData.Application;	// TBD: see note in CollectorExt.cs

                Snoop.Forms.Objects form = new Snoop.Forms.Objects(cmdData.Application);
				form.ShowDialog();
				
                result = IExternalCommand.Result.Succeeded;
			}
            catch (System.Exception e) {
                msg = e.Message;
                result = IExternalCommand.Result.Failed;
            }

			return result;
		}
	}

	/// <summary>
	/// TestShell command:  The TestShell is a framework for adding small tests.  Each test is 
	/// plugged into the TestShell UI so that you don't have to write a new external command for
    /// each test and occupy additional menu items.  Create a Test by adding new RvtMgdDbgTestFuncs objects.
	/// </summary>
	
	public class CmdTestShell : IExternalCommand
	{
        ArrayList m_tests = new ArrayList();
        Autodesk.Revit.Application m_app = null;

		public IExternalCommand.Result
        Execute (Autodesk.Revit.ExternalCommandData cmdData, ref string msg, ElementSet elems)
		{
            m_app = cmdData.Application;
            IExternalCommand.Result result;

            try {
                // Since we dont have an "App" as such, there is no 
                // app-wide data - so just create tests for the duration
                // of the cmd and then detsroy them
                CreateAndAddTests();

                Test.TestForm form = new Test.TestForm(RvtMgdDbgTestFuncs.RegisteredTestFuncs());
                if (form.ShowDialog() == DialogResult.OK)
                    form.DoTest();
                
                result = IExternalCommand.Result.Succeeded;
            }
            catch (System.Exception e) {	// we want to catch it so we can see the problem, otherwise it just silently bails out

                /*System.Exception innerException = e.InnerException;

                while (innerException != null) {
                    innerException = innerException.InnerException;
                }

                msg = innerException.Message;*/
                msg = e.Message;
                MessageBox.Show(msg);


                result = IExternalCommand.Result.Failed;
            }

            finally {
                // if an exception happens anywhere, clean up before quitting
                RemoveAndFreeTestFuncs();
            }

            return result;
		}

        /// <summary>
        /// 
        /// </summary>
        private void
        CreateAndAddTests()
        {
            m_tests.Add(new Test.TestElements(m_app));
            m_tests.Add(new Test.TestDocument(m_app));
            m_tests.Add(new Test.TestGeometry(m_app));
            m_tests.Add(new Test.TestGraphicsStream(m_app));
            m_tests.Add(new Test.TestImportExport(m_app));
            m_tests.Add(new Test.TestUi(m_app));
            m_tests.Add(new Test.SDKSamples.SDKTestFuncs(m_app));
            m_tests.Add(new Test.TestApplication(m_app));

            foreach (RvtMgdDbgTestFuncs testFunc in m_tests) {
                RvtMgdDbgTestFuncs.AddTestFuncsToFramework(testFunc);
            }
        }

        /// <summary>
        /// 
        /// </summary>
        private void
        RemoveAndFreeTestFuncs()
        {
            foreach (RvtMgdDbgTestFuncs testFunc in m_tests) {
                RvtMgdDbgTestFuncs.RemoveTestFuncsFromFramework(testFunc);
            }
        }
	}

    /// <summary>
    /// Snoop ModScope command:  Browse all Elements in the current selection set
    /// </summary>
	
    public class CmdSampleMenuItemCallback : IExternalCommand
    {
	    public IExternalCommand.Result
        Execute (Autodesk.Revit.ExternalCommandData cmdData, ref string msg, ElementSet elems)
	    {
            IExternalCommand.Result result;

		    try {
                MessageBox.Show("Called back into RvtMgdDbg by picking toolbar or menu item");				
                result = IExternalCommand.Result.Succeeded;
		    }
            catch (System.Exception e) {
                msg = e.Message;
                result = IExternalCommand.Result.Failed;
            }

		    return result;
	    }
    }

    public class CmdEvents : IExternalCommand
    {       
        public IExternalCommand.Result
        Execute (Autodesk.Revit.ExternalCommandData cmdData, ref string msg, ElementSet elems)
        {           
            EventTrack.Events.ApplicationEvents.m_app = cmdData.Application;
            EventTrack.Events.DocEvents.m_docSet = cmdData.Application.Documents;

            IExternalCommand.Result result;

            try
            {
                RvtMgdDbg.EventTrack.Forms.EventsForm dbox = new RvtMgdDbg.EventTrack.Forms.EventsForm();
                dbox.ShowDialog();                               

                result = IExternalCommand.Result.Succeeded;
            }
            catch (System.Exception e)
            {
                msg = e.Message;
                result = IExternalCommand.Result.Failed;
            }

            return result;
        }
    }

}

