using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

using Revit = Autodesk.Revit;
using Autodesk.Revit.Geometry;
using Autodesk.Revit.Elements;
using Autodesk.Revit.Symbols;

namespace RvtMgdDbg.Utils {
    /// <summary>
    /// 
    /// </summary>
    public class Elements {
        /// <summary>
        /// 
        /// </summary>
        /// <param name="app"></param>
        /// <param name="element"></param>
        /// <returns></returns>
        public static Revit.Element CloneElement(Revit.Application app, Revit.Element element)
        {
            Opening opening = element as Opening;
            if (opening != null) {
                return CloneElement(app, opening);
            }

            BoundaryConditions boundaryConditions = element as BoundaryConditions;
            if (boundaryConditions != null) {
                return CloneElement(app, boundaryConditions);
            }

            AreaLoad areaLoad = element as AreaLoad;
            if (areaLoad != null) {
                return CloneElement(app, areaLoad);
            }

            AreaReinforcement areaReinforcement = element as AreaReinforcement;
            if (areaReinforcement != null) {
                return CloneElement(app, areaReinforcement);
            }

            BeamSystem beamSystem = element as BeamSystem;
            if (beamSystem != null) {
                return CloneElement(app, beamSystem);
            }

            Dimension dimension = element as Dimension;
            if (dimension != null) {
                return CloneElement(app, dimension);
            }

            FamilyInstance familyInstance = element as FamilyInstance;
            if (familyInstance != null) {
                return CloneElement(app, familyInstance);
            }

            Floor floor = element as Floor;
            if (floor != null) {
                return CloneElement(app, floor);
            }

            Grid grid = element as Grid;
            if (grid != null) {
                return CloneElement(app, grid);
            }

            Group group = element as Group;
            if (group != null) {
                return CloneElement(app, group);
            }

            Level level = element as Level;
            if (floor != null) {
                return CloneElement(app, floor);
            }

            LineLoad lineLoad = element as LineLoad;
            if (lineLoad != null) {
                return CloneElement(app, lineLoad);
            }

            LoadCase loadCase = element as LoadCase;
            if (loadCase != null) {
                return CloneElement(app, loadCase);
            }

            LoadCombination loadCombination = element as LoadCombination;
            if (loadCombination != null) {
                return CloneElement(app, loadCombination);
            }

            LoadNature loadNature = element as LoadNature;
            if (loadNature != null) {
                return CloneElement(app, loadNature);
            }

            LoadUsage loadUsage = element as LoadUsage;
            if (loadUsage != null) {
                return CloneElement(app, loadUsage);
            }

            ModelCurve modelCurve = element as ModelCurve;
            if (modelCurve != null) {
                return CloneElement(app, modelCurve);
            }

            PointLoad pointLoad = element as PointLoad;
            if (pointLoad != null) {
                return CloneElement(app, pointLoad);
            }

            Rebar rebar = element as Rebar;
            if (rebar != null) {
                return CloneElement(app, rebar);
            }

            ReferencePlane referencePlane = element as ReferencePlane;
            if (referencePlane != null) {
                return CloneElement(app, referencePlane);
            }

            Room room = element as Room;
            if (room != null) {
                return CloneElement(app, room);
            }

            RoomTag roomTag = element as RoomTag;
            if (roomTag != null) {
                return CloneElement(app, roomTag);
            }

            SketchPlane sketchPlane = element as SketchPlane;
            if (sketchPlane != null) {
                return CloneElement(app, sketchPlane);
            }

            View3D view3D = element as View3D;
            if (view3D != null) {
                return CloneElement(app, view3D);
            }

            ViewDrafting viewDrafting = element as ViewDrafting;
            if (viewDrafting != null) {
                return CloneElement(app, viewDrafting);
            }

            ViewSection viewSection = element as ViewSection;
            if (viewSection != null) {
                return CloneElement(app, viewSection);
            }

            ViewSheet viewSheet = element as ViewSheet;
            if (viewSheet != null) {
                return CloneElement(app, viewSheet);
            }

            Wall wall = element as Wall;
            if (wall != null) {
                return CloneElement(app, wall);
            }

            // this element has not yet been exposed in the Creation Document class
            //Debug.Assert(false);

            return null;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="app"></param>
        /// <param name="opening"></param>
        /// <returns></returns>
        private static Revit.Element CloneElement(Revit.Application app, Opening opening)
        {
            Opening openingClone = app.ActiveDocument.Create.NewOpening(opening.Host, opening.BoundaryCurves, true);
            Utils.ParamUtil.SetParameters(openingClone.Parameters, opening.Parameters);
            return openingClone;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="app"></param>
        /// <param name="beamSystem"></param>
        /// <returns></returns>
        private static Revit.Element CloneElement(Revit.Application app, BeamSystem beamSystem)
        {
            BeamSystem beamSystemClone = app.ActiveDocument.Create.NewBeamSystem(beamSystem.Profile);
            Utils.ParamUtil.SetParameters(beamSystemClone.Parameters, beamSystem.Parameters);
            return beamSystemClone;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="app"></param>
        /// <param name="wall"></param>
        /// <returns></returns>
        private static Revit.Element CloneElement(Revit.Application app, Wall wall)
        {
            Revit.LocationCurve locCurve = wall.Location as Revit.LocationCurve;
            bool isStructural = (wall.StructuralUsage == Autodesk.Revit.Structural.Enums.WallUsage.NonBearing) ? false : true;
            Wall wallClone = app.ActiveDocument.Create.NewWall(locCurve.Curve, wall.Level, isStructural);
            Utils.ParamUtil.SetParameters(wallClone.Parameters, wall.Parameters);
            return wallClone;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="app"></param>
        /// <param name="familyInstance"></param>
        /// <returns></returns>
        private static Revit.Element CloneElement(Revit.Application app, FamilyInstance familyInstance)
        {
            Revit.Geometry.XYZ location = new Autodesk.Revit.Geometry.XYZ();

            // special case for something like a beam system which has a curve
            Revit.LocationCurve locationCurve = familyInstance.Location as Revit.LocationCurve;
            if (locationCurve != null) {
                location = locationCurve.Curve.get_EndPoint(0);
            }
            Revit.LocationPoint locationPoint = familyInstance.Location as Revit.LocationPoint;
            if (locationPoint != null) {
                location = locationPoint.Point;
            }

            FamilyInstance familyInstanceClone = app.ActiveDocument.Create.NewFamilyInstance(location, familyInstance.Symbol, familyInstance.StructuralType);
            Utils.ParamUtil.SetParameters(familyInstanceClone.Parameters, familyInstance.Parameters);
            return familyInstanceClone;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="app"></param>
        /// <param name="floor"></param>
        /// <returns></returns>
        private static Revit.Element CloneElement(Revit.Application app, Floor floor)
        {
            //get geometry to figure out location of floor
            Options options = app.Create.NewGeometryOptions();
            options.DetailLevel = Options.DetailLevels.Coarse;
            Element geomElem = floor.get_Geometry(options);
            GeometryObjectArray geomObjArray = geomElem.Objects;
            Solid solid = geomObjArray.get_Item(0) as Solid;
            double absoluteElev = floor.Level.Elevation + floor.get_Parameter(Revit.Parameters.BuiltInParameter.FLOOR_HEIGHTABOVELEVEL_PARAM).AsDouble();
            CurveArray curveArray = Utils.Geometry.GetProfile(solid, absoluteElev, app);

            Floor floorClone = app.ActiveDocument.Create.NewFloor(curveArray, floor.FloorType, floor.Level, false);
            Utils.ParamUtil.SetParameters(floorClone.Parameters, floor.Parameters);
            return floorClone;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="app"></param>
        /// <param name="dimension"></param>
        /// <returns></returns>
        private static Revit.Element CloneElement(Revit.Application app, Dimension dimension)
        {
            XYZ startPt = dimension.Curve.get_EndPoint(0);
            XYZ endPt = dimension.Curve.get_EndPoint(1);
            Line line = app.Create.NewLine(startPt, endPt, true);
            Dimension dimensionClone = app.ActiveDocument.Create.NewDimension(dimension.View, line, dimension.References);
            Utils.ParamUtil.SetParameters(dimensionClone.Parameters, dimension.Parameters);
            return dimensionClone;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="app"></param>
        /// <param name="grid"></param>
        /// <returns></returns>
        private static Revit.Element CloneElement(Revit.Application app, Grid grid)
        {
            Grid gridClone = null;
            Curve curve = grid.Curve;

            Line line = grid.Curve as Line;
            if (line != null) {
                gridClone = app.ActiveDocument.Create.NewGrid(line);
            }

            Arc arc = grid.Curve as Arc;
            if (arc != null) {
                gridClone = app.ActiveDocument.Create.NewGrid(arc);
            }
            //Utils.ParamUtil.SetParameters(gridClone.Parameters, grid.Parameters);
            return gridClone;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="app"></param>
        /// <param name="group"></param>
        /// <returns></returns>
        private static Revit.Element CloneElement(Revit.Application app, Group group)
        {
            Revit.ElementSet elemSet = app.Create.NewElementSet();
            Revit.ElementArray elemArray = group.Members;
            Revit.ElementArrayIterator elemArrayIter = elemArray.ForwardIterator();
            while (elemArrayIter.MoveNext()) {
                Revit.Element elem = elemArrayIter.Current as Revit.Element;
                elemSet.Insert(elem);
            }
            Group groupClone = app.ActiveDocument.Create.NewGroup(elemSet);
            Utils.ParamUtil.SetParameters(groupClone.Parameters, group.Parameters);
            return groupClone;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="app"></param>
        /// <param name="level"></param>
        /// <returns></returns>
        private static Revit.Element CloneElement(Revit.Application app, Level level)
        {
            Level levelClone = app.ActiveDocument.Create.NewLevel(level.Elevation);
            Utils.ParamUtil.SetParameters(levelClone.Parameters, level.Parameters);
            return levelClone;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="app"></param>
        /// <param name="modelCurve"></param>
        /// <returns></returns>
        private static Revit.Element CloneElement(Revit.Application app, ModelCurve modelCurve)
        {
            ModelCurve modelCurveClone = app.ActiveDocument.Create.NewModelCurve(modelCurve.GeometryCurve, modelCurve.SketchPlane);
            Utils.ParamUtil.SetParameters(modelCurveClone.Parameters, modelCurve.Parameters);
            return modelCurveClone;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="app"></param>
        /// <param name="referencePlane"></param>
        /// <returns></returns>
        private static Revit.Element CloneElement(Revit.Application app, ReferencePlane referencePlane)
        {
            ReferencePlane referencePlaneClone = app.ActiveDocument.Create.NewReferencePlane(referencePlane.BubbleEnd, referencePlane.FreeEnd, referencePlane.Normal, app.ActiveDocument.ActiveView);   // TBD: ReferencePlane.View dissappeared (jma - 12/05/06)
            Utils.ParamUtil.SetParameters(referencePlaneClone.Parameters, referencePlane.Parameters);
            return referencePlaneClone;
        }


        /// <summary>
        /// 
        /// </summary>
        /// <param name="app"></param>
        /// <param name="room"></param>
        /// <returns></returns>
        private static Revit.Element CloneElement(Revit.Application app, Room room)
        {
            Revit.LocationPoint locationPoint = room.Location as Revit.LocationPoint;
            UV point = new UV();
            point.U = locationPoint.Point.X;
            point.V = locationPoint.Point.Y;
            Room roomClone = app.ActiveDocument.Create.NewRoom(room.Level, point);
            Utils.ParamUtil.SetParameters(roomClone.Parameters, room.Parameters);
            return roomClone;
        }


        /// <summary>
        /// 
        /// </summary>
        /// <param name="app"></param>
        /// <param name="roomTag"></param>
        /// <returns></returns>
        private static Revit.Element CloneElement(Revit.Application app, RoomTag roomTag)
        {
            Revit.LocationPoint locationPoint = roomTag.Location as Revit.LocationPoint;
            UV point = new UV();
            point.U = locationPoint.Point.X;
            point.V = locationPoint.Point.Y;
            RoomTag roomTagClone = null;//app.ActiveDocument.Create.NewRoomTag(roomTag.Room, ref point, );
            Utils.ParamUtil.SetParameters(roomTagClone.Parameters, roomTag.Parameters);
            return roomTagClone;
        }


        /// <summary>
        /// 
        /// </summary>
        /// <param name="app"></param>
        /// <param name="sketchPlane"></param>
        /// <returns></returns>
        private static Revit.Element CloneElement(Revit.Application app, SketchPlane sketchPlane)
        {
            SketchPlane sketchPlaneClone = app.ActiveDocument.Create.NewSketchPlane(sketchPlane.Plane);
            Utils.ParamUtil.SetParameters(sketchPlaneClone.Parameters, sketchPlane.Parameters);
            return sketchPlaneClone;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="app"></param>
        /// <param name="view3D"></param>
        /// <returns></returns>
        private static Revit.Element CloneElement(Revit.Application app, View3D view3D)
        {
            XYZ viewDir = view3D.ViewDirection;
            View3D view3DClone = app.ActiveDocument.Create.NewView3D(viewDir);
            Utils.ParamUtil.SetParameters(view3DClone.Parameters, view3D.Parameters);
            return view3DClone;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="app"></param>
        /// <param name="viewDrafting"></param>
        /// <returns></returns>
        private static Revit.Element CloneElement(Revit.Application app, ViewDrafting viewDrafting)
        {
            ViewDrafting viewDraftingClone = app.ActiveDocument.Create.NewViewDrafting();
            Utils.ParamUtil.SetParameters(viewDraftingClone.Parameters, viewDrafting.Parameters);
            return viewDraftingClone;
        }
    }
}
