#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
using System.Windows.Forms;
using System.Drawing;
using Autodesk.Windows;
#endregion

namespace ScheduleData
{
  [Transaction( TransactionMode.ReadOnly )]
  public class Command : IExternalCommand
  {
    /// <summary>
    /// Define the schedule export folder.
    /// All existing files will be overwritten.
    /// </summary>
    const string _export_folder_name 
      = "C:\\tmp\\schedule_data";

    /// <summary>
    /// Schedule export filename extension.
    /// </summary>
    const string _ext = ".txt";

    const string _caption_prefix
      = "Schedule Data - ";

    /// <summary>
    /// Select a text file in the gioven folder.
    /// </summary>
    /// <param name="folder">Initial folder.</param>
    /// <param name="filename">Selected filename on success.</param>
    /// <returns>Return true if a file was successfully selected.</returns>
    static bool FileSelect( 
      string folder, 
      out string filename )
    {
      OpenFileDialog dlg = new OpenFileDialog();
      dlg.Title = "Select Schedule Data File";
      dlg.CheckFileExists = true;
      dlg.CheckPathExists = true;
      //dlg.RestoreDirectory = true;
      dlg.InitialDirectory = folder;
      dlg.Filter = ".txt Files (*.txt)|*.txt";
      bool rc = (DialogResult.OK == dlg.ShowDialog());
      filename = dlg.FileName;
      return rc;
    }

    void DisplayScheduleData( 
      string filename, 
      IWin32Window owner )
    {
      ScheduleDataParser parser 
        = new ScheduleDataParser( filename );

      System.Windows.Forms.Form form 
        = new System.Windows.Forms.Form();
      form.Size = new Size( 400, 400 );
      form.Text = _caption_prefix + parser.Name;

      DataGridView dg = new DataGridView();
      dg.AllowUserToAddRows = false;
      dg.AllowUserToDeleteRows = false;
      dg.AllowUserToOrderColumns = true;
      dg.Dock = System.Windows.Forms.DockStyle.Fill;
      dg.Location = new System.Drawing.Point( 0, 0 );
      dg.ReadOnly = true;
      dg.TabIndex = 0;
      dg.DataSource = parser.Table;
      dg.Parent = form;

      form.ShowDialog( owner );
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      IWin32Window revit_window 
        = new JtWindowHandle( 
          ComponentManager.ApplicationWindow );

      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Document doc = uidoc.Document;

      FilteredElementCollector col
        = new FilteredElementCollector( doc )
          .OfClass( typeof( ViewSchedule ) );

      ViewScheduleExportOptions opt 
        = new ViewScheduleExportOptions();

      foreach( ViewSchedule vs in col )
      {
        Directory.CreateDirectory( 
          _export_folder_name );

        vs.Export( _export_folder_name, 
          vs.Name + _ext, opt );
      }

      string filename;

      while( FileSelect( _export_folder_name, 
        out filename ) )
      {
        DisplayScheduleData( filename, 
          revit_window );
      }
      return Result.Succeeded;
    }
  }
}
