﻿using System;
using System.Collections.Generic;
using System.Linq;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.Architecture;
using Autodesk.Revit.UI;

namespace SelectionChangedExample
{
  [Transaction( TransactionMode.Manual )]
  [Regeneration( RegenerationOption.Manual )]
  [Journaling( JournalingMode.NoCommandData )]

  public class SelectionChangedExampleApp
    : IExternalApplication
  {
    private SelectionChangedWatcher
      _selectionChangedWatcher;

    private TextBox _tb;

    public Result OnShutdown(
      UIControlledApplication a )
    {
      return Result.Succeeded;
    }

    public Result OnStartup(
      UIControlledApplication a )
    {
      _selectionChangedWatcher
        = new SelectionChangedWatcher( a );

      _selectionChangedWatcher.SelectionChanged
        += new EventHandler(
          OnSelectionChanged );

      // it does not seem to be possible to add items 
      // to context-sensitive ribbon panels.
      // however the user can detach this panel from 
      // the main ribbon so that it is not hidden by 
      // context-sensitive panels.

      RibbonPanel rpSelectionWatcher
        = a.CreateRibbonPanel( "Selection Watcher" );

      var t1 = new TextBoxData( "txtInfo" );

      _tb = rpSelectionWatcher.AddItem( t1 ) as TextBox;

      return Result.Succeeded;
    }

    void OnSelectionChanged(
      object sender,
      EventArgs e )
    {
      if( null == _selectionChangedWatcher.Selection )
      {
        ShowInfo( "No selection" );
        return;
      }

      // this example just reports the name of the 
      // room that is selected. Obviously any other
      // kind of element can be handled instead.

      List<Room> rooms = new List<Room>(
        _selectionChangedWatcher.Selection.OfType<Room>() );

      if( 0 == rooms.Count )
      {
        ShowInfo( "No rooms selected" );
      }
      else if( 1 == rooms.Count )
      {
        ShowInfo( "Room " + rooms[0].Number );
      }
      else
      {
        ShowInfo( "Multiple rooms selected" );
      }
    }

    private void ShowInfo( string p )
    {
      _tb.PromptText = p;
    }
  }
}
