﻿#Region "Header"
'
' Util.cs - The Building Coder Revit API utility methods
'
' Copyright (C) 2008 by Jeremy Tammik,
' Autodesk Inc. All rights reserved.
'
#End Region

#Region "Namespaces"
Imports System.Collections.Generic
Imports System.Diagnostics
Imports WinForms = System.Windows.Forms
Imports Autodesk.Revit
Imports Autodesk.Revit.DB


'Imports Autodesk.Revit.DB.Element
'Imports Autodesk.Revit.Geometry
Imports Autodesk.Revit.DB.Element
Imports Autodesk.Revit.UI.Selection
Imports RvtElement = Autodesk.Revit.DB.Element

#End Region

Namespace BuildingCoder



    Class Util



        Public Shared Function Max(ByVal a As Double()) As Double
            Debug.Assert(1 = a.Rank, "expected one-dimensional array")
            Debug.Assert(0 = a.GetLowerBound(0), "expected zero-based array")
            Debug.Assert(0 < a.GetUpperBound(0), "expected non-empty array")
            Dim maxa As Double = a(0)
            For i As Integer = 1 To a.GetUpperBound(0)
                If maxa < a(i) Then
                    maxa = a(i)
                End If
            Next
            Return maxa
        End Function

#Region "Geometrical Comparison"
        Const _eps As Double = 0.000000001

        Public Shared ReadOnly Property Eps() As Double
            Get
                Return _eps
            End Get
        End Property

        Public Shared ReadOnly Property MinLineLength() As Double
            Get
                Return _eps
            End Get
        End Property

        Public Shared ReadOnly Property TolPointOnPlane() As Double
            Get
                Return _eps
            End Get
        End Property

        Public Shared Function IsZero(ByVal a As Double) As Boolean
            Return _eps > Math.Abs(a)
        End Function

        Public Shared Function IsEqual(ByVal a As Double, ByVal b As Double) As Boolean
            Return IsZero(b - a)
        End Function

        Public Shared Function Compare(ByVal a As Double, ByVal b As Double) As Integer
            Return If(IsEqual(a, b), 0, (If(a < b, -1, 1)))
        End Function

        Public Shared Function Compare(ByVal p As DB.XYZ, ByVal q As DB.XYZ) As Integer
            Dim diff As Integer = Compare(p.X, q.X)
            If 0 = diff Then
                diff = Compare(p.Y, q.Y)
                If 0 = diff Then
                    diff = Compare(p.Z, q.Z)
                End If
            End If
            Return diff
        End Function

        Public Shared Function IsHorizontal(ByVal v As DB.XYZ) As Boolean
            Return IsZero(v.Z)
        End Function

        Public Shared Function IsVertical(ByVal v As DB.XYZ) As Boolean
            Return IsZero(v.X) AndAlso IsZero(v.Y)
        End Function

        Public Shared Function IsHorizontal(ByVal e As DB.Edge) As Boolean
            Dim p As DB.XYZ = e.Evaluate(0)
            Dim q As DB.XYZ = e.Evaluate(1)
            Return IsHorizontal(q - p)
        End Function

        Public Shared Function IsHorizontal(ByVal f As PlanarFace) As Boolean
            Return IsVertical(f.Normal)
        End Function

        Public Shared Function IsVertical(ByVal f As PlanarFace) As Boolean
            Return IsHorizontal(f.Normal)
        End Function

        Public Shared Function IsVertical(ByVal f As CylindricalFace) As Boolean
            Return IsVertical(f.Axis)
        End Function
#End Region

#Region "Unit Handling"
        Const _convertFootToMm As Double = 12 * 25.4

        Const _convertFootToMeter As Double = _convertFootToMm * 0.001

        Const _convertCubicFootToCubicMeter As Double = _convertFootToMeter * _convertFootToMeter * _convertFootToMeter

        ''' <summary>
        ''' Convert a given length in feet to millimetres.
        ''' </summary>
        Public Shared Function FootToMm(ByVal length As Double) As Double
            Return length * _convertFootToMm
        End Function

        ''' <summary>
        ''' Convert a given length in millimetres to feet.
        ''' </summary>
        Public Shared Function MmToFoot(ByVal length As Double) As Double
            Return length / _convertFootToMm
        End Function

        ''' <summary>
        ''' Convert a given volume in feet to cubic meters.
        ''' </summary>
        Public Shared Function CubicFootToCubicMeter(ByVal volume As Double) As Double
            Return volume * _convertCubicFootToCubicMeter
        End Function
#End Region

#Region "Formatting"
        ''' <summary>
        ''' Return an English plural suffix 's' or
        ''' nothing for the given number of items.
        ''' </summary>
        Public Shared Function PluralSuffix(ByVal n As Integer) As String
            Return If(1 = n, "", "s")
        End Function

        ''' <summary>
        ''' Return an English plural suffix 'ies' or
        ''' 'y' for the given number of items.
        ''' </summary>
        Public Shared Function PluralSuffixY(ByVal n As Integer) As String
            Return If(1 = n, "y", "ies")
        End Function

        Public Shared Function DotOrColon(ByVal n As Integer) As String
            Return If(0 < n, ":", ".")
        End Function

        Public Shared Function RealString(ByVal a As Double) As String
            Return a.ToString("0.##")
        End Function

        Public Shared Function AngleString(ByVal angle As Double) As String
            Return RealString(angle * 180 / Math.PI) & " degrees"
        End Function

        Public Shared Function MmString(ByVal length As Double) As String
            Return RealString(FootToMm(length)) & " mm"
        End Function

        Public Shared Function PointString(ByVal p As DB.XYZ) As String
            Return String.Format("({0},{1},{2})", RealString(p.X), RealString(p.Y), RealString(p.Z))
        End Function

        Public Shared Function PlaneString(ByVal p As DB.Plane) As String
            Return String.Format("plane origin {0}, plane normal {1}", PointString(p.Origin), PointString(p.Normal))
        End Function

        Public Shared Function TransformString(ByVal t As Transform) As String
            Return String.Format("({0},{1},{2},{3})", PointString(t.Origin), PointString(t.BasisX), PointString(t.BasisY), PointString(t.BasisZ))
        End Function

        ' ''Public Shared Function PointArrayString(ByVal pts As XYZArray) As String
        ' ''    Dim s As String = String.Empty
        ' ''    For Each p As DB.XYZ In pts
        ' ''        If 0 < s.Length Then
        ' ''            s += ", "
        ' ''        End If
        ' ''        s += PointString(p)
        ' ''    Next
        ' ''    Return s
        ' ''End Function

        ' ''Public Shared Function CurveString(ByVal curve As Curve) As String
        ' ''    Return "curve tesselation " & PointArrayString(curve.Tessellate())
        ' ''End Function
#End Region

        Const _caption As String = "The Building Coder"

        Public Shared Sub InfoMsg(ByVal msg As String)
            Debug.WriteLine(msg)
            WinForms.MessageBox.Show(msg, _caption, WinForms.MessageBoxButtons.OK, WinForms.MessageBoxIcon.Information)
        End Sub

        Public Shared Sub ErrorMsg(ByVal msg As String)
            Debug.WriteLine(msg)
            WinForms.MessageBox.Show(msg, _caption, WinForms.MessageBoxButtons.OK, WinForms.MessageBoxIcon.[Error])
        End Sub

        ' ''Public Shared Function ElementDescription(ByVal e As RvtElement) As String
        ' ''    ' for a wall, the element name equals the
        ' ''    ' wall type name, which is equivalent to the
        ' ''    ' family name ...
        ' ''    Dim fi As FamilyInstance = TryCast(e, FamilyInstance)
        ' ''    Dim fn As String = If((fi Is Nothing), String.Empty, fi.Symbol.Family.Name + " ")

        ' ''    Dim cn As String = If((e.Category Is Nothing), e.[GetType]().Name, e.Category.Name)

        ' ''    Return String.Format("{0} {1}<{2} {3}>", cn, fn, e.Id.Value, e.Name)
        ' ''End Function

#Region "Element Selection"
        ' ''Public Shared Function SelectSingleElement(ByVal doc As Document, ByVal description As String) As RvtElement
        ' ''    Dim sel As Selection = doc.Selection
        ' ''    Dim e As RvtElement = Nothing
        ' ''    sel.Elements.Clear()
        ' ''    sel.StatusbarTip = "Please select " & description
        ' ''    If sel.PickOne() Then
        ' ''        Dim elemSetItr As ElementSetIterator = sel.Elements.ForwardIterator()
        ' ''        elemSetItr.MoveNext()
        ' ''        e = TryCast(elemSetItr.Current, RvtElement)
        ' ''    End If
        ' ''    Return e
        ' ''End Function

        ' ''Public Shared Function GetSingleSelectedElement(ByVal doc As Document) As RvtElement
        ' ''    Dim e As RvtElement = Nothing
        ' ''    Dim [set] As SelElementSet = doc.Selection.Elements

        ' ''    If 1 = [set].Size Then
        ' ''        For Each e2 As RvtElement In [set]
        ' ''            e = e2
        ' ''        Next
        ' ''    End If
        ' ''    Return e
        ' ''End Function

        ' ''Public Shared Function SelectSingleElementOfType(ByVal doc As Document, ByVal t As Type, ByVal description As String) As RvtElement
        ' ''    Dim e As RvtElement = GetSingleSelectedElement(doc)

        ' ''    If e Is Nothing OrElse Not e.[GetType]().Equals(t) Then
        ' ''        ' IsSubclassOf( t )
        ' ''        e = Util.SelectSingleElement(doc, description)
        ' ''    End If
        ' ''    Return If((e Is Nothing) OrElse e.[GetType]().Equals(t), e, Nothing)
        ' ''End Function

      
        ''Public Shared Function GetSelectedElementsOrAll(ByVal a As List(Of RvtElement), ByVal doc As Document, ByVal t As Type) As Boolean
        ''    Dim sel As Selection = doc.Selection
        ''    If 0 < sel.Elements.Size Then
        ''        For Each e As RvtElement In sel.Elements
        ''            If t.IsInstanceOfType(e) Then
        ''                a.Add(e)
        ''            End If
        ''        Next
        ''    Else
        ''        doc.get_Elements(t, a)
        ''    End If
        ''    Return 0 < a.Count
        ''End Function
#End Region
    End Class
End Namespace