﻿//#define DEBUG

#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
//using System.Web.UI; // provides a non-generic Pair class
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
#endregion

namespace TopoSurfacePointClassify
{
  /*
  [Serializable]
	public class Pair<TFirst, TSecond>
	{
		public Pair()
		{
		}
 
		public Pair(TFirst first, TSecond second)
		{
			First = first;
			Second = second;
		}
 
		public TFirst First	{ get; set; }
		public TSecond Second { get; set; }
 
		public override bool Equals(object obj)
		{
			if (this == obj) return true;
			Pair<TFirst, TSecond> pair = obj as Pair<TFirst, TSecond>;
			if( null == pair ) return false;
			return Equals(First, pair.First) && Equals(Second, pair.Second);
		}
 
		public override int GetHashCode()
		{
			return (null == First ? 0 : First.GetHashCode()) 
        + 29 * (null == Second ? 0 : Second.GetHashCode());
		}
 
		public static List<Pair<TFirst, TSecond>> Dictionary2List(IDictionary<TFirst, TSecond> dictionary)
		{
			List<Pair<TFirst, TSecond>> result = new List<Pair<TFirst, TSecond>>();
			if (dictionary != null)
			{
				foreach (KeyValuePair<TFirst, TSecond> pair in dictionary)
				{
					result.Add(new Pair<TFirst, TSecond>(pair.Key, pair.Value));
				}
			}
			return result;
		}
 
		public static Dictionary<TFirst, TSecond> List2Dictionary(IList<Pair<TFirst, TSecond>> list)
		{
			Dictionary<TFirst, TSecond> result = new Dictionary<TFirst, TSecond>();
			if (list != null)
			{
				foreach (Pair<TFirst, TSecond> pair in list)
				{
					result.Add(pair.First, pair.Second);
				}
			}
			return result;
		}
 
		public static Dictionary<TFirst, List<TSecond>> List2TwoDimentionalDictionary(IList<Pair<TFirst, TSecond>> list)
		{
			Dictionary<TFirst, List<TSecond>> result = new Dictionary<TFirst, List<TSecond>>();
			if (list != null)
			{
				foreach (Pair<TFirst, TSecond> pair in list)
				{
					List<TSecond> dimentionList;
					if (!result.TryGetValue(pair.First, out dimentionList))
					{
						dimentionList = new List<TSecond>();
						result[pair.First] = dimentionList;
					}
 
					dimentionList.Add(pair.Second);
				}
			}
			return result;
		}
	}
  */

  [Transaction( TransactionMode.Manual )]
  [Regeneration( RegenerationOption.Manual )]
  public class Command : IExternalCommand
  {
    #region Formatting
    public static string PluralSuffix( int n )
    {
      return 1 == n ? "" : "s";
    }

    public static string RealString( double a )
    {
      return a.ToString( "0.##" );
    }

    public static string PointString( XYZ p )
    {
      return string.Format( "({0},{1},{2})",
        RealString( p.X ), 
        RealString( p.Y ),
        RealString( p.Z ) );
    }
    #endregion // Formatting

    #region Geometrical Comparison
    const double _eps = 1.0e-9;

    public static bool IsZero( double a, double tolerance )
    {
      return tolerance > Math.Abs( a );
    }

    public static bool IsZero( double a )
    {
      return IsZero( a, _eps );
    }

    public static bool IsEqual( double a, double b )
    {
      return IsZero( b - a );
    }

    /// <summary>
    /// A comparison method for real values.
    /// </summary>
    /// <param name="a"></param>
    /// <param name="b"></param>
    /// <returns>-1 if a is smaller than b, 1 if larger, and 0 if equal.</returns>
    public static int Compare( double a, double b )
    {
      return IsEqual( a, b ) ? 0 : ( a < b ? -1 : 1 );
    }

    /// <summary>
    /// A comparison method for XYZ instances.
    /// </summary>
    /// <returns>-1 if p is smaller than q, 1 if larger, and 0 if equal.</returns>
    public static int Compare( XYZ p, XYZ q )
    {
      int diff = Compare( p.X, q.X );
      if( 0 == diff )
      {
        diff = Compare( p.Y, q.Y );
        if( 0 == diff )
        {
          diff = Compare( p.Z, q.Z );
        }
      }
      return diff;
    }

    /// <summary>
    /// Define equality between XYZ objects, 
    /// ensuring that almost equal points compare equal.
    /// </summary>
    class XyzEqualityComparer : IEqualityComparer<XYZ>
    {
      public bool Equals( XYZ p, XYZ q )
      {
        return p.IsAlmostEqualTo( q );
      }

      public int GetHashCode( XYZ p )
      {
        return PointString( p ).GetHashCode();
      }
    }
    #endregion // Geometrical Comparison

#if DEBUG
    public class MeshTriangleEdge
    {
      public XYZ A { get; set; }
      public XYZ B { get; set; }

      public MeshTriangleEdge( XYZ a, XYZ b )
      {
        int d = Compare( a, b );

        Debug.Assert( 0 != d, "expected non-equal edge vertices" );

        A = ( 0 < d ) ? a : b;
        B = ( 0 < d ) ? b : a;
      }
    }

    public static string EdgeString( MeshTriangleEdge e )
    {
      return string.Format( "{0}->{1}",
        PointString( e.A ),
        PointString( e.B ) );
    }

    /// <summary>
    /// A comparison method for MeshTriangleEdge instances.
    /// </summary>
    /// <returns>-1 if e is smaller than f, 1 if larger, and 0 if equal.</returns>
    public static int Compare( MeshTriangleEdge e, MeshTriangleEdge f )
    {
      int diff = Compare( e.A, f.A );
      if( 0 == diff )
      {
        diff = Compare( e.B, f.B );
      }
      return diff;
    }

    /// <summary>
    /// Define equality between MeshTriangleEdge objects.
    /// </summary>
    class EdgeEqualityComparer 
      : IEqualityComparer<MeshTriangleEdge>
    {
      public bool Equals( 
        MeshTriangleEdge e, 
        MeshTriangleEdge f )
      {
        return e.A.IsAlmostEqualTo( f.A ) 
          && e.B.IsAlmostEqualTo( f.B );
      }

      public int GetHashCode( MeshTriangleEdge e )
      {
        string s = PointString( e.A ) + ":" + PointString( e.B );
        return s.GetHashCode();
      }
    }

    /// <summary>
    /// Map mesh vertices, which are also triangle 
    /// vertices, to a list of the indices of all 
    /// the triangles they belong to.
    /// </summary>
    class MapVertexToTriangleIndices 
      : Dictionary<XYZ, List<int>>
    {
      public MapVertexToTriangleIndices()
        : base( new XyzEqualityComparer() )
      {
      }

      /// <summary>
      /// Add a new XYZ vertex.
      /// If it is already known, append the triangle 
      /// index of the current triangle. Otherwise, 
      /// generate a new key for it.
      /// </summary>
      public void AddVertex( XYZ p, int triangleIndex )
      {
        if( !ContainsKey( p ) )
        {
          Add( p, new List<int>( 3 ) );
        }
        ( this )[p].Add( triangleIndex );
      }
    }

    /// <summary>
    /// Map mesh triangle edges to a list of the 
    /// indices of all the triangles they belong to.
    /// </summary>
    class MapEdgeToTriangleIndices
      : Dictionary<MeshTriangleEdge, List<int>>
    {
      public MapEdgeToTriangleIndices()
        : base( new EdgeEqualityComparer() )
      {
      }

      /// <summary>
      /// Add a new edge.
      /// If it is already known, append the triangle 
      /// index of the current triangle. Otherwise, 
      /// generate a new key for it.
      /// </summary>
      public void AddEdge( 
        MeshTriangleEdge e, 
        int triangleIndex )
      {
        if( !ContainsKey( e ) )
        {
          Add( e, new List<int>( 2 ) );
        }
        ( this )[e].Add( triangleIndex );
      }
    }

    /// <summary>
    /// For each point of the given mesh,
    /// determine how many triangles it belongs to.
    /// </summary>
    void DetermineTriangleCountForPoints( Mesh mesh )
    {
      int np = mesh.Vertices.Count;
      int nt = mesh.NumTriangles;

      Debug.Print( "Mesh has {0} point{1} and {2} triangle{3}.",
        np, PluralSuffix( np ), nt, PluralSuffix( nt ) );

      MapVertexToTriangleIndices map 
        = new MapVertexToTriangleIndices();

      for( int i = 0; i < nt; ++i )
      {
        MeshTriangle t = mesh.get_Triangle( i );

        for( int j = 0; j < 3; ++j )
        {
          map.AddVertex( t.get_Vertex( j ), i );
        }
      }
      List<XYZ> pts = new List<XYZ>( map.Keys );
      pts.Sort( Compare );

      foreach( XYZ p in pts )
      {
        int n = map[p].Count;

        Debug.Print( "  vertex {0} belongs to {1} triangle{2}", 
          PointString( p ), n, PluralSuffix( n ) );
      }
    }

    /// <summary>
    /// For each edge of the given mesh,
    /// determine how many triangles it belongs to.
    /// </summary>
    void DetermineTriangleCountForEdges( Mesh mesh )
    {
      int np = mesh.Vertices.Count;
      int nt = mesh.NumTriangles;

      Debug.Print( "Mesh has {0} point{1} and {2} triangle{3}.",
        np, PluralSuffix( np ), nt, PluralSuffix( nt ) );

      MapEdgeToTriangleIndices map
        = new MapEdgeToTriangleIndices();

      for( int i = 0; i < nt; ++i )
      {
        MeshTriangle t = mesh.get_Triangle( i );

        for( int j = 0; j < 3; ++j )
        {
          MeshTriangleEdge e = new MeshTriangleEdge(
            t.get_Vertex( 0 == j ? 2 : j - 1 ),
            t.get_Vertex( j ) );

          map.AddEdge( e, i );
        }
      }
      List<MeshTriangleEdge> edges = new List<MeshTriangleEdge>( map.Keys );
      edges.Sort( Compare );

      foreach( MeshTriangleEdge e in edges )
      {
        int n = map[e].Count;

        Debug.Assert( 0 < n, "expected each edge to belong to at least one triangle" );
        Debug.Assert( 3 > n, "expected each edge to belong to at most two triangles" );

        Debug.Print( "  edge {0} belongs to {1} triangle{2} and is therefore {3}",
          EdgeString( e ), n, PluralSuffix( n ),
          ( 2 == n ? "interior" : "boundary" ) );
      }
    }
#endif // DEBUG

    /// <summary>
    /// Manage a mesh triangle edge by storing the 
    /// index of the mesh vertex corresponing to
    /// the edge start and end point.
    /// For reliable comparison purposes, the 
    /// lower index is always stored in A and 
    /// the higher in B.
    /// </summary>
    public class JtEdge
    {
      public int A { get; set; }
      public int B { get; set; }

      public JtEdge( int a, int b )
      {
        Debug.Assert( a != b, "expected non-equal edge vertices" );

        A = ( a < b ) ? a : b;
        B = ( a < b ) ? b : a;
      }
    }

    /// <summary>
    /// A comparison method for JtEdge instances.
    /// </summary>
    /// <param name="e"></param>
    /// <param name="f"></param>
    /// <returns>-1 if e is smaller than f, 1 if larger, and 0 if equal.</returns>
    public static int Compare( JtEdge e, JtEdge f )
    {
      int diff = Compare( e.A, f.A );
      if( 0 == diff )
      {
        diff = Compare( e.B, f.B );
      }
      return diff;
    }

    /// <summary>
    /// Define equality between JtEdge objects.
    /// </summary>
    class JtEdgeEqualityComparer
      : IEqualityComparer<JtEdge>
    {
      public bool Equals( 
        JtEdge e, 
        JtEdge f )
      {
        return e.A == f.A 
          && e.B == f.B;
      }

      public int GetHashCode( JtEdge e )
      {
        return (e.A.ToString() + ":" 
          + e.B.ToString() ).GetHashCode();
      }
    }

    /// <summary>
    /// Map mesh triangle edges to a list of the 
    /// indices of all the triangles they belong to.
    /// </summary>
    class MapEdgeToTriangles
      : Dictionary<JtEdge, List<int>>
    {
      public MapEdgeToTriangles()
        : base( new JtEdgeEqualityComparer() )
      {
      }

      /// <summary>
      /// Add a new edge.
      /// If it is already known, append the triangle 
      /// index of the current triangle. Otherwise, 
      /// generate a new key for it.
      /// </summary>
      public void AddEdge(
        JtEdge e,
        int triangleIndex )
      {
        if( !ContainsKey( e ) )
        {
          Add( e, new List<int>( 2 ) );
        }
        ( this )[e].Add( triangleIndex );
      }
    }

    /// <summary>
    /// Map mesh vertex index to a list of the edges 
    /// the vertex belongs to.
    /// </summary>
    class MapVertexToEdges
      : Dictionary<int, List<JtEdge>>
    {
      public MapVertexToEdges( int capacity )
        : base( capacity )
      {
      }

      /// <summary>
      /// Append a new edge for a given vertex.
      /// If the vertex is new, generate a new key for it.
      /// </summary>
      public void AddVertexEdge(
        int vertexIndex,
        JtEdge e )
      {
        if( !ContainsKey( vertexIndex ) )
        {
          Add( vertexIndex, new List<JtEdge>( 2 ) );
        }
        ( this )[vertexIndex].Add( e );
      }
    }

    /// <summary>
    /// For each point of the given mesh,
    /// determine whether it is interior or boundary.
    /// The algorithm goes like this:
    /// Every triangle edge belongs to either one or two triangles,
    /// depending on whether it it boundary or interior.
    /// For each edge, determine whether it is boundary 
    /// or interior. A point is interior if all of the edges it 
    /// belongs to are interior.
    /// </summary>
    /// <returns>A dictionary mapping each mesh vertex index 
    /// to a Boolean which is true if the corresponding point 
    /// is interior and false if it is on the boundary</returns>
    Dictionary<int, bool> ClassifyPoints( Mesh mesh )
    {
      int nv = mesh.Vertices.Count;
      int nt = mesh.NumTriangles;
      int i, n;

      Debug.Print( "\nClassifyPoints: mesh has {0} point{1} and {2} triangle{3}:",
        nv, PluralSuffix( nv ), nt, PluralSuffix( nt ) );

      // set up a map to determine the vertex 
      // index of a given triangle vertex;
      // this is needed because the 
      // MeshTriangle.get_Vertex method
      // returns the XYZ but not the index,
      // and we base our edges on the index:

      Dictionary<XYZ, int> vertexIndex
        = new Dictionary<XYZ, int>( nv, new XyzEqualityComparer() );

      for( i = 0; i < nv; ++i )
      {
        XYZ p = mesh.Vertices[i];
        Debug.Print( "  mesh vertex {0}: {1}", i, PointString( p ) );
        vertexIndex[p] = i;
      }

    #if DEBUG
      for( i = 0; i < nt; ++i )
      {
        MeshTriangle t = mesh.get_Triangle( i );

        for( int j = 0; j < 3; ++j )
        {
          XYZ p = t.get_Vertex( j );
          Debug.Print( "  triangle {0} vertex {1}: {2}", i, j, PointString( p ) );
          Debug.Assert( vertexIndex.ContainsKey( p ),
            "expected all triangle vertices to be part of the mesh vertices" );
        }
      }
    #endif // DEBUG
      
      // set up a map to determine which 
      // edges a given vertex belongs to:

      MapVertexToEdges vertexEdges
        = new MapVertexToEdges( nv );

      // set up a map to determine which 
      // triangles a given edge belongs to;
      // this is used to determine the edge's
      // interior or boundary status:

      MapEdgeToTriangles map
        = new MapEdgeToTriangles();

      for( i = 0; i < nt; ++i )
      {
        MeshTriangle t = mesh.get_Triangle( i );

        for( int j = 0; j < 3; ++j )
        {
          // get the start and end vertex 
          // of the current triangle edge:

          int a = vertexIndex[t.get_Vertex( 0 == j ? 2 : j - 1 )];
          int b = vertexIndex[t.get_Vertex( j )];

          JtEdge e = new JtEdge( a, b );

          map.AddEdge( e, i );

          vertexEdges.AddVertexEdge( a, e );
          vertexEdges.AddVertexEdge( b, e );
        }
      }

    #if DEBUG
      List<JtEdge> edges = new List<JtEdge>( map.Keys );
      edges.Sort( Compare );
      n = edges.Count;

      Debug.Print( "Classify the {0} edge{1}:", n, PluralSuffix( n ) );

      foreach( JtEdge e in edges )
      {
        n = map[e].Count;

        Debug.Assert( 0 < n, "expected each edge to belong to at least one triangle" );
        Debug.Assert( 3 > n, "expected each edge to belong to at most two triangles" );

        Debug.Print( "  edge {0}->{1} belongs to {2} triangle{3} and is therefore {4}",
          e.A, e.B, n, PluralSuffix( n ),
          ( 2 == n ? "interior" : "boundary" ) );
      }
    #endif

      int nBoundaryEdges;
      int nInteriorPoints = 0;

      Dictionary<int, bool> dict = new Dictionary<int,bool>( nv );

      Debug.Print( "Classify the {0} point{1}:", nv, PluralSuffix( nv ) );

      for( i = 0; i < nv; ++i )
      {
        nBoundaryEdges = 0;
        n = vertexEdges[i].Count;

        //foreach( JtEdge e in vertexEdges[i] )
        //{
        //  n = map[e].Count;
        //  if( 1 == n )
        //  {
        //    ++nBoundaryEdges;
        //  }
        //}

        nBoundaryEdges = vertexEdges[i].Count<JtEdge>( 
          e => 1 == map[e].Count );

        dict[i] = ( 0 == nBoundaryEdges );

        XYZ p = mesh.Vertices[i];

        if( 0 == nBoundaryEdges )
        {
          ++nInteriorPoints;
        }

        Debug.Print( "  point {0} {1} belongs to {2} edge{3}, "
          + "{4} interior and {5} boundary and is therefore {6}",
          i, PointString( p ), n, PluralSuffix( n ),
          n - nBoundaryEdges, nBoundaryEdges,
          ( 0 == nBoundaryEdges ? "interior" : "boundary" ) );
      }
      Debug.Print( "{0} boundary and {1} interior points detected.",
        nv - nInteriorPoints, nInteriorPoints );

      return dict;
    }

    /// <summary>
    /// Classify the points of a given topgraphy 
    /// surface mesh into interior and boundary.
    /// </summary>
    void ClassifyPoints( TopographySurface ts, Options opt )
    {
      Mesh mesh = null;
      GeometryElement e = ts.get_Geometry( opt );
      GeometryObjectArray objects = e.Objects;
      foreach( GeometryObject obj in objects )
      {
        mesh = obj as Mesh;
        if( null != mesh )
        {
          break;
        }
      }
      if( null == mesh )
      {
        Debug.Print( 
          "\nNo mesh found for TopographySurface {0}.", 
          ts.Id.IntegerValue );
      }
      else
      {
        Debug.Print( 
          "\nTopographySurface {0}:",
          ts.Id.IntegerValue );

#if DEBUG
        DetermineTriangleCountForPoints( mesh );
        DetermineTriangleCountForEdges( mesh );
#endif // DEBUG

        ClassifyPoints( mesh );
      }
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      Document doc = uiapp.ActiveUIDocument.Document;

      FilteredElementCollector toposurfs
        = new FilteredElementCollector( doc )
          .OfClass( typeof( TopographySurface ) );

      Options opt = new Options();

      foreach( TopographySurface ts in toposurfs )
      {
        ClassifyPoints( ts, opt );
      }
      return Result.Succeeded;
    }
  }
}
