﻿#region Namespaces
using System;
using System.Diagnostics;
using System.Collections.Generic;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
#endregion

namespace Transform3
{
  [Transaction( TransactionMode.Manual )]
  public class Command : IExternalCommand
  {
    UIApplication _uiapp;
    UIDocument _uidoc;
    Application _app;
    Document _doc;

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      _uiapp = commandData.Application;
      _uidoc = _uiapp.ActiveUIDocument;
      _app = _uiapp.Application;
      _doc = commandData.Application.ActiveUIDocument.Document;
      List<Element> a = dwgLinkInstances( _doc );
      foreach( Element e in a )
      {
        checkLinkedDwg( e );
      }
      return Result.Succeeded;
    }

    void checkLinkedDwg( Element linked )
    {
      Instance iLinked = linked as Instance;
      Transform transform = iLinked.GetTransform();
      Options options = new Options();
      options.View = _doc.ActiveView;
      GeometryElement gElem = iLinked.get_Geometry( options );
      Debug.Print( "Point without Transformation: " + fetchSomeMesh( gElem ).Vertices[0].ToString() );
      XYZ vertex = fetchSomeMesh( gElem, transform ).Vertices[0];
      Debug.Print( "Point when Transformed: " + vertex.ToString() );
      Debug.Print( "Point when Transformed + inverse: " + transform.Inverse.OfPoint( vertex ).ToString() );
      Debug.Print( "Point with ID Transformation: " + fetchSomeMeshTransformed( gElem ).Vertices[0].ToString() );
    }

    protected List<Element> dwgLinkInstances( Document document )
    {
      List<Element> instances = new List<Element>();
      FilteredElementCollector collector
        = new FilteredElementCollector( document )
          .OfClass( typeof( Instance ) );

      foreach( Element e in collector )
      {
        if( ( (Instance) e ).ObjectType.Name.EndsWith( ".dwg" ) )
        {
          instances.Add( e );
        }
      }
      return instances;
    }

    protected Mesh fetchSomeMesh( GeometryElement gElem )
    {
      foreach( GeometryObject gObject in gElem.Objects )
      {
        Mesh gMesh = gObject as Mesh;
        if( null != gMesh )
        {
          return gMesh;
        }
      }

      // Loop and seek for geometry instances
      foreach( GeometryObject gObject in gElem.Objects )
      {
        GeometryInstance gInstance = gObject as GeometryInstance;
        if( null != gInstance )
        {
          // If it's GeometryInstance, go recursive
          return fetchSomeMesh( gInstance.SymbolGeometry );
        }
      }

      return null;
    }

    protected Mesh fetchSomeMesh( GeometryElement gElem, Transform transform )
    {
      // Apply transformation and seek for Meshes
      GeometryElement transformed = gElem.GetTransformed( transform );

      foreach( GeometryObject gObject in transformed.Objects )
      {
        Mesh gMesh = gObject as Mesh;
        if( null != gMesh )
        {
          return gMesh;
        }
      }

      // Loop and seek for geometry instances
      foreach( GeometryObject gObject in gElem.Objects )
      {
        GeometryInstance gInstance = gObject as GeometryInstance;
        if( null != gInstance )
        {
          // If it's GeometryInstance, combine transformations and go recursive
          Transform combinedTransform = gInstance.Transform.Multiply( transform );
          return fetchSomeMesh( gInstance.SymbolGeometry, combinedTransform );
        }
      }

      return null;
    }

    protected Mesh fetchSomeMeshTransformed( GeometryElement gElem )
    {
      // Apply transformation and seek for Meshes
      GeometryElement transformed = gElem.GetTransformed( Transform.Identity );

      foreach( GeometryObject gObject in transformed.Objects )
      {
        Mesh gMesh = gObject as Mesh;
        if( null != gMesh )
        {
          return gMesh;
        }
      }

      // Loop and seek for geometry instances
      foreach( GeometryObject gObject in gElem.Objects )
      {
        GeometryInstance gInstance = gObject as GeometryInstance;
        if( null != gInstance )
        {
          // If it's GeometryInstance, combine transformations and go recursive
          //Transform combinedTransform = gInstance.Transform.Multiply( transform );
          return fetchSomeMeshTransformed( gInstance.SymbolGeometry );
        }
      }

      return null;
    }

  }
}
