﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using WinForms = System.Windows.Forms;
using Autodesk.Revit;
using Autodesk.Revit.Collections; // Map
using Autodesk.Revit.Elements;
using Autodesk.Revit.Parameters;
using Autodesk.Revit.MEP;
using Autodesk.Revit.Geometry;
using Autodesk.Revit.Symbols;

using CmdResult = Autodesk.Revit.IExternalCommand.Result;

namespace UK_DB_Schedule
{
  class CircuitReferences
  {
    public
    
      int mPhase1;
      int mPhase2;
      int mPhase3;

    public void Add_UK_References (Autodesk.Revit.Application app)
    {
      Document doc = app.ActiveDocument;

      List<Autodesk.Revit.Element> equipment = new List<Autodesk.Revit.Element>();
      Util.GetElectricalEquipment(equipment, app);
      int n = equipment.Count;

      //
      // determine mapping from panel to circuit == electrical system:
      //
      Dictionary<string, ElectricalSystemSet> mapPanelToSystems = new Dictionary<string, ElectricalSystemSet>();
      List<Autodesk.Revit.Element> systems = new List<Autodesk.Revit.Element>();
      Util.GetElectricalSystems(systems, app);
      n = systems.Count;
      Debug.WriteLine(string.Format("Retrieved {0} electrical system{1}.", n, Util.PluralSuffix(n)));
      
      //
      // all circuits which are fed from the same family instance have 
      // the same panel name, so you can retrieve all of these circuits.
      //
      // todo: there is an issue here if there are several different panels 
      // with the same name! they will get merged in the tree view, 
      // but they should stay separate. possible workaround: add the 
      // element id to keep them separate, and then remove it again
      // when displaying in tree view.
      //
      Debug.WriteLine("-----------------------------------------------------------------------------------");
      foreach (ElectricalSystem system in systems)
      {
        string panelName = system.PanelName;
        Debug.WriteLine("  system " + system.Name + ": panel " + panelName + " load " + system.LoadClassification);
        if (!mapPanelToSystems.ContainsKey(panelName))
        {
          mapPanelToSystems.Add(panelName, new ElectricalSystemSet());
        }
        mapPanelToSystems[panelName].Insert(system);
      }
      Debug.WriteLine("-----------------------------------------------------------------------------------");
      n = mapPanelToSystems.Count;
      
      Debug.WriteLine(string.Format("Mapping from the {0} panel{1} to electrical systems == circuits:", n, Util.PluralSuffix(n)));
      List<string> keys = new List<string>(mapPanelToSystems.Keys);
      keys.Sort();

      // ArrayList outputStringList = new ArrayList();

      // --- Panel Select Dialog
      OutputForm fs = new OutputForm(keys);

      if (fs.ShowDialog() == DialogResult.OK)
      {
        string selectedPanel = fs.mSelectedPanel;

        string s;
        foreach (string panelName in keys)
        {
          if (panelName == selectedPanel)
          {
            int numWays = 0;

            foreach (Autodesk.Revit.Element rvtEquipment in equipment)
            {
              string equipmentType = rvtEquipment.GetType().ToString();
              string objectType = rvtEquipment.ObjectType.ToString();

              if (rvtEquipment.Name == selectedPanel)
              {
                foreach (Autodesk.Revit.Parameter param in rvtEquipment.Parameters)
                {
                  string nn = param.Definition.Name;
                  string paramStr = param.AsString();
                  if (param.Definition.Name == "Max #1 Pole Breakers")
                  {
                    numWays = param.AsInteger();
                    break;
                  }
                }
                break;
              }
            }      
      
            // --- Check we've an even number of circuits
            int evenCheck = numWays % 2;

            // --- Check that the number of ways are divisible by 3
            int div3Check = numWays % 3;

            if (evenCheck != 0)
            {
              WinForms.MessageBox.Show("There should be an even number of circuits");
              return;
            }

            if (div3Check != 0)
            {
              WinForms.MessageBox.Show("The number of circuits should be divisible by 3");
              return;
            }

            int i;

            string[] circuitRefArray = new string[numWays];

            for (i = 0; i < numWays; i++)
            {
              circuitRefArray[i] = "X";
            }

            int num3phases = numWays / 3;

            int num3phasesOnOneSide = num3phases / 2;

            int phaseCount = 1;
            string phaseStr;

            int circuitNum = 0;

            for (i = 0; i < num3phasesOnOneSide; i++)
            {
              int circuit1 = circuitNum;
              int circuit2 = circuitNum + 2;
              int circuit3 = circuitNum + 4;
              circuitRefArray[circuit1] = phaseCount + "L1";
              circuitRefArray[circuit2] = phaseCount + "L2";
              circuitRefArray[circuit3] = phaseCount + "L3";
              circuitNum += 6;
              phaseCount++;
            }

            circuitNum = 1;

            for (i = 0; i < num3phasesOnOneSide; i++)
            {
              int circuit1 = circuitNum;
              int circuit2 = circuitNum + 2;
              int circuit3 = circuitNum + 4;
              circuitRefArray[circuit1] = phaseCount + "L1";
              circuitRefArray[circuit2] = phaseCount + "L2";
              circuitRefArray[circuit3] = phaseCount + "L3";
              circuitNum += 6;
              phaseCount++;
            }

            int halfCircuitNum = numWays - 1;
            
            s = string.Empty;

            foreach (ElectricalSystem system in mapPanelToSystems[panelName])
            {
              ConnectorManager cmgr = system.ConnectorManager;

              int numConnectors = cmgr.Connectors.Size;

              Debug.Assert(cmgr.UnusedConnectors.Size <= cmgr.Connectors.Size, "unused connectors is a subset of connectors");
              Debug.Assert(system.Name.Equals(system.CircuitNumber), "ElectricalSystem Name and CircuitNumber properties are always identical");

              s += (0 < s.Length ? ", " : ": ") + system.Name;

              int polesNunber = system.PolesNumber;

              string circuitNumberStr = system.CircuitNumber;

              mPhase1 = 0;
              mPhase2 = 0;
              mPhase3 = 0;

              string circuitRef = "";

              if (polesNunber == 1)
              {
                mPhase1 = Convert.ToInt32(circuitNumberStr);
                circuitRef = circuitRefArray[mPhase1-1];
              }
              else if (polesNunber == 3)
              {
                extract3phaseCircuitNumbers(circuitNumberStr);
                circuitRef = circuitRefArray[mPhase1-1];
                // --- Knock off the last char (e.g. "6L3" to "6L")
                string crStart = circuitRef.Substring(0, circuitRef.Length - 1);
                // --- And add on "123" (e.g. "6L" to "6L123")
                circuitRef = crStart + "123";
              }

              /*
              outputStringList.Add("System name: " + system.Name);
              outputStringList.Add("CircuitNumber: " + system.CircuitNumber.ToString());
              outputStringList.Add("LoadName: " + system.LoadName);
              outputStringList.Add("PolesNumber: " + system.PolesNumber.ToString());
              outputStringList.Add("Rating: " + system.Rating.ToString());
              */

              foreach (Autodesk.Revit.Parameter param in system.Parameters)
              {
                string name = param.Definition.Name;
                if (name == "CircuitRef")
                  param.Set(circuitRef);
              }

              /*
              outputStringList.Add("Voltage: " + GetStringFromParam(system, BuiltInParameter.RBS_ELEC_VOLTAGE));
              outputStringList.Add("TrueLoad: " + GetStringFromParam(system, BuiltInParameter.RBS_ELEC_TRUE_LOAD));
              outputStringList.Add("TrueLoadPhaseA: " + GetStringFromParam(system, BuiltInParameter.RBS_ELEC_TRUE_LOAD_PHASEA));
              outputStringList.Add("TrueLoadPhaseB: " + GetStringFromParam(system, BuiltInParameter.RBS_ELEC_TRUE_LOAD_PHASEB));
              outputStringList.Add("TrueLoadPhaseC: " + GetStringFromParam(system, BuiltInParameter.RBS_ELEC_TRUE_LOAD_PHASEC));
              */

            }

            Debug.WriteLine("  " + panelName + s);
          }
        }

        List<Autodesk.Revit.Element> lightingFixtures = new List<Autodesk.Revit.Element>();
        Util.GetLightingFixtures(lightingFixtures, app);
        int numLightingFixtures = lightingFixtures.Count;

        foreach (Autodesk.Revit.Element lf in lightingFixtures)
        {
          string circuitNumber = "";
          bool onOurPanel = false;
          string equipmentType = lf.GetType().ToString();
          string objectType = lf.ObjectType.ToString();
          string panelName;

          foreach (Autodesk.Revit.Parameter param in lf.Parameters)
          {
            string nn = param.Definition.Name;
            string paramStr = param.AsString();
            if (param.Definition.Name == "Panel")
            {
              if (paramStr == selectedPanel)
              {
                onOurPanel = true;
                panelName = paramStr;
              }
            }
            if (nn == "Circuit Number")
            {
              circuitNumber = param.AsString();
            }
          }

          if (onOurPanel)
          {
            string circuitRef = "";
            
            foreach (string pn in keys)
            {
              if (pn == selectedPanel)
              {
                foreach (ElectricalSystem system in mapPanelToSystems[pn])
                {
                  string circuitNumberStr = system.CircuitNumber;
                  if (circuitNumber == circuitNumberStr)
                  {
                    foreach (Autodesk.Revit.Parameter param in system.Parameters)
                    {
                      string name = param.Definition.Name;
                      if (name == "CircuitRef")
                      {
                        circuitRef = param.AsString();
                      }
                    }
                  }
                }
              }
            }

            if (circuitRef != "")
            {
              foreach (Autodesk.Revit.Parameter param in lf.Parameters)
              {
                string name = param.Definition.Name;
                if (name == "CircuitRef")
                {
                  param.Set(circuitRef);
                }
              }

            }
          }
        }

        List<Autodesk.Revit.Element> wires = new List<Autodesk.Revit.Element>();
        Util.GetWires(wires, app);
        int numWires = wires.Count;

        foreach (Autodesk.Revit.Element lf in wires)
        {


        }

      }
    }


    private void extract3phaseCircuitNumbers(string circuitStr)
    {
      int c = 0;
      string numStr;
      numStr = "";
      int phase = 1;
      for (c = 0; c < circuitStr.Length; c++)
      {
        if (circuitStr[c] == ',')
        {
          if (phase == 1) mPhase1 = Convert.ToInt32(numStr);
          if (phase == 2) mPhase2 = Convert.ToInt32(numStr);
          phase++;
          numStr = "";
        }
        else
        {
          numStr += circuitStr[c];
        }
      }
      mPhase3 = Convert.ToInt32(numStr);
    }
  }
}


