﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using WinForms = System.Windows.Forms;
using Autodesk.Revit;
using Autodesk.Revit.Collections; // Map
using Autodesk.Revit.Elements;
using Autodesk.Revit.Parameters;
using Autodesk.Revit.MEP;
using Autodesk.Revit.Geometry;
using Autodesk.Revit.Symbols;
using Microsoft.Office.Interop.Excel;

using CmdResult = Autodesk.Revit.IExternalCommand.Result;

namespace UK_DB_Schedule
{
  class CreateDBSchedule
  {

    // ---------------------------------------------------------------------------------------------------------------------

    public void Run(Autodesk.Revit.Application app)
    {
      Document doc = app.ActiveDocument;

      List<Autodesk.Revit.Element> equipment = new List<Autodesk.Revit.Element>();
      Util.GetElectricalEquipment(equipment, app);
      int n = equipment.Count;

      //
      // determine mapping from panel to circuit == electrical system:
      //
      Dictionary<string, ElectricalSystemSet> mapPanelToSystems = new Dictionary<string, ElectricalSystemSet>();
      List<Autodesk.Revit.Element> systems = new List<Autodesk.Revit.Element>();
      Util.GetElectricalSystems(systems, app);
      n = systems.Count;
      Debug.WriteLine(string.Format("Retrieved {0} electrical system{1}.", n, Util.PluralSuffix(n)));
      
      //
      // all circuits which are fed from the same family instance have 
      // the same panel name, so you can retrieve all of these circuits.
      //
      // todo: there is an issue here if there are several different panels 
      // with the same name! they will get merged in the tree view, 
      // but they should stay separate. possible workaround: add the 
      // element id to keep them separate, and then remove it again
      // when displaying in tree view.
      //
      Debug.WriteLine("-----------------------------------------------------------------------------------");
      foreach (ElectricalSystem system in systems)
      {
        string panelName = system.PanelName;
        Debug.WriteLine("  system " + system.Name + ": panel " + panelName + " load " + system.LoadClassification);
        if (!mapPanelToSystems.ContainsKey(panelName))
        {
          mapPanelToSystems.Add(panelName, new ElectricalSystemSet());
        }
        mapPanelToSystems[panelName].Insert(system);
      }
      Debug.WriteLine("-----------------------------------------------------------------------------------");
      n = mapPanelToSystems.Count;
      
      Debug.WriteLine(string.Format("Mapping from the {0} panel{1} to electrical systems == circuits:", n, Util.PluralSuffix(n)));
      List<string> keys = new List<string>(mapPanelToSystems.Keys);
      keys.Sort();

      // ArrayList outputStringList = new ArrayList();

      // --- Panel Select Dialog
      OutputForm fs = new OutputForm(keys);
      
      if (fs.ShowDialog() == DialogResult.OK)
      {
        string selectedPanel = fs.mSelectedPanel;

        int numWays = 0;

        foreach (Autodesk.Revit.Element rvtEquipment in equipment)
        {
          string equipmentType = rvtEquipment.GetType().ToString();
          string objectType = rvtEquipment.ObjectType.ToString();

          if (rvtEquipment.Name == selectedPanel)
          {
            foreach (Autodesk.Revit.Parameter param in rvtEquipment.Parameters)
            {
              string nn = param.Definition.Name;
              string paramStr = param.AsString();
              if (param.Definition.Name == "Max #1 Pole Breakers")
              {
                numWays = param.AsInteger();
                // break;
              }
            }
            break;
          }
        }

        string workbookPath = "c:/PanelSchedule.xls";

        OpenFileDialog dialog = new OpenFileDialog();
        
        dialog.Filter = "xls files|*.xls";

        dialog.InitialDirectory = "c:\\";
        dialog.Title = "Select a Excel Spreadsheet file";

        if (dialog.ShowDialog() == DialogResult.OK)
        {
          workbookPath = dialog.FileName;

          Microsoft.Office.Interop.Excel.Application excelApp = new Microsoft.Office.Interop.Excel.ApplicationClass();

          excelApp.Visible = true;

          Microsoft.Office.Interop.Excel.Workbook xlsWorkbook = excelApp.Workbooks.Open(workbookPath,
            0, false, 5, "", "", false, Microsoft.Office.Interop.Excel.XlPlatform.xlWindows, "",
            true, false, 0, true, false, false);

          Worksheet ws = (Microsoft.Office.Interop.Excel.Worksheet)xlsWorkbook.ActiveSheet;

          string s;

          foreach (string panelName in keys)
          {
            if (panelName == selectedPanel)
            {
              // --- Export panel name to "O3"
              exportToExcel(ws, "O", 3, panelName);

              // --- Export number of ways (slots) to "H5"
              exportToExcel(ws, "H", 5, numWays.ToString());

              s = string.Empty;

              int circuitRow = 12;

              List<string> circuitList = new List<string>();


              // --- Create a list of all the circuits on our panel
              foreach (ElectricalSystem system in mapPanelToSystems[panelName])
              {
                string cr = "?";
                foreach (Autodesk.Revit.Parameter param in system.Parameters)
                {
                  string name = param.Definition.Name;
                  if (name == "CircuitRef")
                  {
                    cr = param.AsString();
                    if (cr == null)
                      cr = "?";
                    break;
                  }
                }
                circuitList.Add(cr);
              }
           
              // --- Sort the list
              circuitList.Sort();

              // --- Loop through all the circuits and pick them out in the order of the sorted list
              foreach (string cref in circuitList)
              {
                // WinForms.MessageBox.Show(cref);

                foreach (ElectricalSystem system in mapPanelToSystems[panelName])
                {
                  string cr = "?";
                  foreach (Autodesk.Revit.Parameter param in system.Parameters)
                  {                 
                    if (param.Definition.Name == "CircuitRef")
                    {
                      cr = param.AsString();
                      
                      if (cr == null)
                        cr = "?";

                      if (cr == cref)
                      {
                        exportToExcel(ws, "A", circuitRow, cr);

                        ConnectorManager cmgr = system.ConnectorManager;

                        int numConnectors = cmgr.Connectors.Size;

                        exportToExcel(ws, "J", circuitRow, numConnectors.ToString());

                        Debug.Assert(cmgr.UnusedConnectors.Size <= cmgr.Connectors.Size, "unused connectors is a subset of connectors");
                        Debug.Assert(system.Name.Equals(system.CircuitNumber), "ElectricalSystem Name and CircuitNumber properties are always identical");

                        s += (0 < s.Length ? ", " : ": ") + system.Name;

                        /*
                        outputStringList.Add("System name: " + system.Name);
                        outputStringList.Add("CircuitNumber: " + system.CircuitNumber.ToString());
                        outputStringList.Add("LoadName: " + system.LoadName);
                        outputStringList.Add("PolesNumber: " + system.PolesNumber.ToString());
                        outputStringList.Add("Rating: " + system.Rating.ToString());
                        */

                        exportToExcel(ws, "C", circuitRow, system.LoadName.ToString());
                        exportToExcel(ws, "Q", circuitRow, system.Rating.ToString());

                        /*
                        Debug.WriteLine("Voltage: " + system.Voltage.ToString());
                        Debug.WriteLine("TrueLoadPhaseA: " + system.TrueLoadPhaseA.ToString());
                        Debug.WriteLine("TrueLoadPhaseB: " + system.TrueLoadPhaseB.ToString());
                        Debug.WriteLine("TrueLoadPhaseC: " + system.TrueLoadPhaseC.ToString());
                        */

                        foreach (Autodesk.Revit.Parameter p in system.Parameters)
                        {
                          if (p.Definition.Name == "CableSize_PH")
                          {
                            cr = p.AsString();
                            exportToExcel(ws, "K", circuitRow, cr);
                          }
                          if (p.Definition.Name == "CableSize_CPC")
                          {
                            cr = p.AsString();
                            exportToExcel(ws, "L", circuitRow, cr);
                          }
                        }

                        circuitRow++;

                        /*
                        outputStringList.Add("Voltage: " + GetStringFromParam(system, BuiltInParameter.RBS_ELEC_VOLTAGE));
                        outputStringList.Add("TrueLoad: " + GetStringFromParam(system, BuiltInParameter.RBS_ELEC_TRUE_LOAD));
                        outputStringList.Add("TrueLoadPhaseA: " + GetStringFromParam(system, BuiltInParameter.RBS_ELEC_TRUE_LOAD_PHASEA));
                        outputStringList.Add("TrueLoadPhaseB: " + GetStringFromParam(system, BuiltInParameter.RBS_ELEC_TRUE_LOAD_PHASEB));
                        outputStringList.Add("TrueLoadPhaseC: " + GetStringFromParam(system, BuiltInParameter.RBS_ELEC_TRUE_LOAD_PHASEC));
                        */
                        break;
                      }
                    }
                  }

                }
              }

              Debug.WriteLine("  " + panelName + s);
            }
          }
        }
      }

      // --- Here is my start to attempt to write a table to the new detail view
      /*
      ViewDrafting view = doc.Create.NewViewDrafting();
      view.ViewName = "Panel Schedule";

      XYZ startPoint = new XYZ(0, 0, 0);
      XYZ endPoint = new XYZ(100, 0, 0);

      drawDetailLine(app, doc, view, startPoint, endPoint);

      copyXYZ(endPoint, startPoint);
      endPoint[1] = 100;
      drawDetailLine(app, doc, view, startPoint, endPoint);

      copyXYZ(endPoint, startPoint);
      endPoint[0] = 0;
      drawDetailLine(app, doc, view, startPoint, endPoint);

      copyXYZ(endPoint, startPoint);
      endPoint[1] = 0;
      drawDetailLine(app, doc, view, startPoint, endPoint);

      drawText(app, doc, view, startPoint);
      drawText(app, doc, view, startPoint);
      */

    }
    
    // ---------------------------------------------------------------------------------------------------------------------
    
    static string GetStringFromParam(Autodesk.Revit.Element e, BuiltInParameter bip)
    {
      Autodesk.Revit.Parameter p = e.get_Parameter(bip);
      return (null == p) ? null
        : (StorageType.String == p.StorageType) ? p.AsString()
        : p.AsValueString();
    }

    // ---------------------------------------------------------------------------------------------------------------------

    private void exportToExcel(Worksheet ws, string col, int row, string str)
    {
      int colNum = 0;
      switch (col)
      {
        case "A": colNum = 1; break;
        case "B": colNum = 2; break;
        case "C": colNum = 3; break;
        case "D": colNum = 4; break;
        case "E": colNum = 5; break;
        case "F": colNum = 6; break;
        case "G": colNum = 7; break;
        case "H": colNum = 8; break;
        case "I": colNum = 9; break;
        case "J": colNum = 10; break;
        case "K": colNum = 11; break;
        case "L": colNum = 12; break;
        case "M": colNum = 13; break;
        case "N": colNum = 14; break;
        case "O": colNum = 15; break;
        case "P": colNum = 16; break;
        case "Q": colNum = 17; break;
        case "R": colNum = 18; break;
        case "S": colNum = 19; break;
        case "T": colNum = 20; break;
        case "U": colNum = 21; break;
        case "V": colNum = 22; break;
        case "W": colNum = 23; break;
        case "X": colNum = 24; break;
        case "Y": colNum = 25; break;
        case "Z": colNum = 26; break;  
      }
      ws.Cells[row, colNum] = str;
    }

    // ---------------------------------------------------------------------------------------------------------------------
    
    private void drawDetailLine(Autodesk.Revit.Application app, Document doc, Autodesk.Revit.Elements.View view, XYZ from, XYZ to)
    {
      Autodesk.Revit.Geometry.Line line = app.Create.NewLineBound(from, to);
      doc.Create.NewDetailCurve(view, line);
    }

    // ---------------------------------------------------------------------------------------------------------------------

    private void drawText(Autodesk.Revit.Application app, Document doc, Autodesk.Revit.Elements.View view, XYZ from)
    {
      string str = "Text";
      
      TextNote newTextNote = doc.Create.NewTextNote(
          view,                  // View
          new XYZ(0, 0, 0),        // origin
          new XYZ(1.0, 0.0, 0.0),  // baseVec
          new XYZ(0.0, 1.0, 0.0),  // upVec
          1.0,                     // lineWidth
          Autodesk.Revit.Enums.TextAlignFlags.TEF_ALIGN_CENTER, // textAlign
          str
          );

      List<Autodesk.Revit.Element> textNoteTypes;

      // doc.get_Elements(typeof(TextNoteType), textNoteTypes);

      // newTextNote.TextNoteType = 
    }

    // ---------------------------------------------------------------------------------------------------------------------

    private void copyXYZ(XYZ from, XYZ to)
    {
      to[0] = from[0]; 
      to[1] = from[1]; 
      to[2] = from[2];
    }
  }
}
