﻿/*
 * Revit Macro created by SharpDevelop
 * User: mycynes
 * Date: 11/29/2012
 * Time: 9:54 AM
 * 
 * To change this template use Tools | Options | Coding | Edit Standard Headers.
 */
using System;
using System.Linq;
using Microsoft.CSharp;
using System.Dynamic;

namespace UiClose
{
	/// <summary>
	/// A helper class to keep a placeholder document open in Revit to allow closing the active document in Revit.
	/// </summary>
	public class UiCloseHelper
	{
		/// <summary>
		/// Constructor -- initializes object with a UIApplication and location of a placeholder document.
		/// </summary>
		/// <param name="uiAppInterface">The Revit UI Application, pass either a UIApplication or an ApplicationEntryPoint</param>
		/// <param name="placeholderPath">Path to a dummy .rvt file.</param>
		public UiCloseHelper(dynamic uiAppInterface, string placeholderPath)
			{
				m_uiAppInterface = uiAppInterface;
				if (m_uiAppInterface == null)
					throw new ArgumentNullException("m_uiAppInterface");
				
				if ((!(uiAppInterface is Autodesk.Revit.UI.UIApplication)) && (!(uiAppInterface is Autodesk.Revit.UI.Macros.ApplicationEntryPoint)))
					throw new ArgumentException("Must pass a UIApplication or ApplicationEntryPoint.");
				
				SetPlaceHolderPath(placeholderPath);
			}
		
		/// <summary>
		/// Closes a UIDocument with the same interface as UIDocument.Close
		/// </summary>
		/// <param name="uiDoc">The document to close</param>
		/// <returns>True if successful, false otherwise.</returns>
		public bool CloseAndSave(Autodesk.Revit.UI.UIDocument uiDoc)
		{
			if (uiDoc == null)
				throw new ArgumentNullException("uiDoc");
			RejectClosingPlaceHolder(uiDoc);
			if (!EnsureActivePlaceHolder(m_uiAppInterface))
				return false;
			return uiDoc.SaveAndClose();
		}
		
		/// <summary>
		/// Closes a DB.Document with the same interface as DB.Document.Close
		/// </summary>
		/// <param name="doc">The DB.Document to close</param>
		/// <returns>True if successful, false otherwise.</returns>
		public bool Close(Autodesk.Revit.DB.Document doc)
		{
			return Close(doc, true);
		}
		
		/// <summary>
		/// Closes a DB.Document with the same interface as DB.Document.Close
		/// </summary>
		/// <param name="doc">The DB.Document to close</param>
		/// <param name="saveModified">true to save the document if modified, false to not save.</param>
		/// <returns>True if successful, false otherwise.</returns>
		public bool Close(Autodesk.Revit.DB.Document doc, bool saveModified)
		{
			if (doc == null)
				throw new ArgumentNullException("doc");
			RejectClosingPlaceHolder(doc);
			if (!EnsureActivePlaceHolder(m_uiAppInterface))
				return false;
			return doc.Close(saveModified);
		}
		
		#region Helper methods
		
	
		/// <summary>
		/// Throws an exception if the user passes the placeholder document.
		/// </summary>
		/// <param name="doc">The document to test to ensure it is not the placeholder document.</param>
		private void RejectClosingPlaceHolder(Autodesk.Revit.DB.Document doc)
		{
			if (doc.Title == System.IO.Path.GetFileName(this.m_placeHolderPath))
				throw new ArgumentException("Cannot close placeholder doc: " + doc.Title);
		}
		
		/// <summary>
		/// Throws an exception if the user passes the placeholder document.
		/// </summary>
		/// <param name="doc">The document to check</param>
		private void RejectClosingPlaceHolder(Autodesk.Revit.UI.UIDocument uiDoc)
		{
			RejectClosingPlaceHolder(uiDoc.Document);
		}
		
		/// <summary>
		/// Ensures that the placeholder document is open and active to that another open document can be closed.
		/// </summary>
		private bool EnsureActivePlaceHolder(dynamic application)
		{
			try 
			{
				if (m_placeHolderDoc == null)
					m_placeHolderDoc =	application.OpenAndActivateDocument(m_placeHolderPath);
				else
				{
					m_placeHolderDoc.SaveAndClose();
					m_placeHolderDoc = application.OpenAndActivateDocument(m_placeHolderPath);
				}
				return true;
			}
			catch(Exception)
			{
				return false;
			}
		}
		
		
		/// <summary>
		/// Sets the path of the placeholder document.
		/// </summary>
		/// <param name="path">Path to placeholder/dummy document.</param>
		private void SetPlaceHolderPath(string path)
		{
				m_placeHolderPath = path;
				if (!System.IO.File.Exists(m_placeHolderPath))
					throw new ArgumentException("File not found: " + path);
				if (!(System.IO.Path.GetExtension(m_placeHolderPath) != "rvt"))
					throw new ArgumentException("Placeholder: " + path+ " is not a revit file.");
		}
		#endregion
		#region Data			
		private Autodesk.Revit.UI.UIDocument m_placeHolderDoc;
		private string m_placeHolderPath;
		private dynamic m_uiAppInterface;
		#endregion
	}
}
