﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Autodesk.Revit.DB;


namespace Revit.SDK.Samples.UnitConversion.CS
{

    /// <summary>
    /// This class performs functions related to handling units of measure in Revit.
    /// It is a static class with static methods, which means the methods can be called without
    /// having to create an instance of the class first.
    /// </summary>
    public static class UnitFunctions
    {

        
#region Module-Level Storage

        public const double MAX_ROUNDING_PRECISION = 0.000000000001;  // Seems to be the most precision Revit will allow

        // Set up a static cache dictionary of DisplayUnitTypes for a given UnitType. This is because
        // determining this information is SLOOOOW, so keeping a cache can greatly improve performance.
        private static Dictionary<UnitType, List<DisplayUnitType>> _DisplayUnitTypesCache;
        
#endregion Module-Level Storage





#region Public Functions

        
        /// <summary>
        /// Returns the current DisplayUnitType (units of measure) for the given UnitType being
        /// used on the specified document (family or project document).  This is the equivalent 
        /// of getting the value for the UnitType as seen in the Project Units dialog.
        /// </summary>
        /// <param name="unitType">The UnitType (e.g. Length, Angle, etc.) for which to return the current DisplayUnitType (units of measure)</param>
        /// <param name="document">The document for which the current DisplayUnitType is desired.</param>
        /// <param name="roundingPrecision">The precision with which rounding occurs, IF SUPPORTED</param>
        /// <returns>Returns the current DisplayUnitType (units of measure) for the given UnitType.</returns>
        public static DisplayUnitType get_CurrentDisplayUnitType(UnitType unitType,
                                                                 Document document,
                                                                 out double? roundingPrecision)
        {

            // Following good SOA practices, don't trust the incoming parameters and verify
            // that they have values that can be worked with before doing anything.
            if (null == document) throw new ArgumentNullException("document");

            // This function does not require the document to be a family document, so don't check for that.

            FormatOptions formatOption;
            DisplayUnitType result;

            ProjectUnit projectUnit = document.ProjectUnit;

            try
            {

                formatOption = projectUnit.get_FormatOptions(unitType);
            }
            catch (Exception ex)
            {
                throw new ArgumentOutOfRangeException("The UnitType '" + unitType.ToString() + "' does not have any DisplayUnitType options.", ex);
            }

            try
            {
                result = formatOption.Units;
            }
            catch (Exception ex)
            {
                throw new Exception("Unable to get the DisplayUnitType for UnitType '" + unitType.ToString() + "'", ex);
            }


            try
            {
                roundingPrecision = formatOption.Rounding;
            }
            catch (Exception)
            {
                // Not all dimensional types support rounding, so if an exception is thrown,
                // store the fact into our nullable double that there is no value.
                roundingPrecision = null;
            }


            return result;
        }



        /// <summary>
        /// This method sets the DisplayUnitType for the given UnitType.  
        /// This is the same as changing the Project Units in Revit.
        /// </summary>
        /// <param name="unitType">The unit type for which to set the current DisplayUnitType (units of measure)</param>
        /// <param name="displayUnitType">The DisplayUnitType value to now be used for the given UnitType</param>
        /// <param name="document">The document for which the new DisplayUnitType should be set.</param>
        /// <param name="roundingPrecision">The rounding precision for the conversion.  E.g. 0.001 = 3 decimal places of accuracy. If no value, uses default rounding.</param>
        /// <exception cref="Autodesk.Revit.UnitsOfMeasure.InvalidDisplayUnitTypeException">Thrown if a provided DisplayUnitType is not supported for the given UnitType.</exception>
        public static void SetCurrentDisplayUnitType(UnitType unitType,
                                                     DisplayUnitType displayUnitType,
                                                     Document document,
                                                     double? roundingPrecision)
        {
            // Following good SOA practices, don't trust the incoming parameters and verify
            // that they have values that can be worked with before doing anything.
            if (null == document) throw new ArgumentNullException("document");

            // This function does not require the document to be a family document, so don't check for that.

            ProjectUnit projectUnit = document.ProjectUnit;

            FormatOptions formatOption = projectUnit.get_FormatOptions(unitType);


            try
            {
                formatOption.Units = displayUnitType;
            }
            catch (Exception)
            {
            }


            if (roundingPrecision.HasValue)
            {
                try
                {
                    if (roundingPrecision < MAX_ROUNDING_PRECISION)
                    {
                        // The user has specified a rounding precision beyond Revit's range.
                        // Set the precision to the best that Revit can do.
                        roundingPrecision = MAX_ROUNDING_PRECISION;
                    }

                    formatOption.Rounding = roundingPrecision.Value;
                }
                catch (Exception)
                {
                    // Some unit types don't support rounding.  Let it work for those that do.
                }
            }

        }



        /// <summary>
        /// Sets the value for a parameter without requiring the value to be in internally stored units
        /// because the caller specifies the units of measure the value is in.
        /// </summary>
        /// <param name="familyDocument">The family document hosting the parameter</param>
        /// <param name="familyParameter">A reference to the parameter being updated</param>
        /// <param name="newValueUnits">The units of measure in which the new value is being provided</param>
        /// <param name="newValue">The new value as a string.</param>
        /// <param name="roundingPrecision">The rounding precision for storing the value.  E.g. 0.001 = 3 decimal places of accuracy. If no value, uses default rounding.</param>
        /// <remarks>
        /// If newValue contains symbols, those will override the newValueUnits.  For example, if 
        /// newValueUnits = DisplayUnitType.DUT_DECIMAL_INCHES but newValue is the string: 1' 6" 
        /// then the value stored is the equivalent to 18 inches.  If newValue is the string:  2.54 cm
        /// then the value stored is the equivalent of 1 inch.
        /// </remarks>
        public static void SetFamilyParameterValue(Document familyDocument,
                                                   FamilyParameter familyParameter,
                                                   DisplayUnitType newValueUnits,
                                                   string newValue,
                                                   double? roundingPrecision)
        {
            // Following good SOA practices, don't trust the incoming parameters and verify
            // that they have values that can be worked with before doing anything.
            if (null == familyDocument) throw new ArgumentNullException("familyDocument");
            if (!familyDocument.IsFamilyDocument) throw new ArgumentOutOfRangeException("familyDocument is not a family document");
            if (null == familyParameter) throw new ArgumentNullException("familyParameter");
            if (string.IsNullOrEmpty(newValue)) throw new ArgumentNullException("newValue");

            if (familyParameter.IsReadOnly)
            {
                throw new InvalidOperationException("Parameter '" + familyParameter.Definition.Name + "' is read-only.");
            }

            UnitType parameterUnitType = ConvertParameterTypeToUnitType(familyParameter.Definition.ParameterType);

            // NOTE:  Storing and restoring the current display units is not needed if the incoming
            // value string contains symbols.  For example, is a value such as "3 cm"  or   1' 6"
            // For those cases, just calling SetValueString (below) would work.


            // Get the current display units ("Project Units") for the parameter's UnitType (e.g. Length, Angle, etc.)
            double? originalRoundingPrecision = null;
            DisplayUnitType originalUnits = get_CurrentDisplayUnitType(parameterUnitType,
                                                                       familyDocument,
                                                                       out originalRoundingPrecision);

            try
            {

                // Temporarily set the current display units for the parameter's UnitType.  This is equivalent to changing
                // the Project Units.  
                SetCurrentDisplayUnitType(parameterUnitType,
                                          newValueUnits,
                                          familyDocument,
                                          roundingPrecision);

                // THIS IS THE KEY:  Use SetValueString instead of Set.  Set requires your data to be in
                // whatever internal units of measure Revit uses. SetValueString expects your value to 
                // be in whatever the current DisplayUnitType (units of measure) the document is set to 
                // for the UnitType associated with the parameter.
                //
                // So SetValueString is basically how the Revit GUI works.
                familyDocument.FamilyManager.SetValueString(familyParameter, newValue);
            }
            catch (Exception ex)
            {
                // Just re-throw the exception, but use our Finally to ensure we restore the document
                // to it's original state before exiting this method.
                throw new Exception("Revit threw an exception in SetParameterValue:  " + ex.Message, ex);
            }
            finally
            {
                // Now put back the DisplayUnitType to whatever it was when we started, to leave the
                // document in the same condition in which we found it.
                SetCurrentDisplayUnitType(parameterUnitType,
                                          originalUnits,
                                          familyDocument,
                                          originalRoundingPrecision);
            }


        }



        /// <summary>
        /// This method returns the value of the parameter in the desired display units for that 
        /// parameter's data type and to the specified precision.  
        /// </summary>
        /// <param name="familyDocument">The family document hosting the parameter</param>
        /// <param name="familyParameter">The parameter whose display value is desired</param>
        /// <param name="desiredUnitsOfMeasure">The units of measure in which the value should be returned</param>
        /// <param name="roundingPrecision">The rounding precision for getting the value.  E.g. 0.001 = 3 decimal places of accuracy. If no value, uses default rounding.</param>
        /// <returns>The display value of the parameter in current project units</returns>
        /// <remarks>
        /// It's a bit like a unit converter.
        /// </remarks>
        public static string get_FamilyParameterDisplayValue(Document familyDocument,
                                                             FamilyParameter familyParameter,
                                                             DisplayUnitType desiredUnitsOfMeasure,
                                                             double? roundingPrecision)
        {

            // Following good SOA practices, don't trust the incoming parameters and verify
            // that they have values that can be worked with before doing anything.
            if (null == familyDocument) throw new ArgumentNullException("familyDocument");
            if (!familyDocument.IsFamilyDocument) throw new ArgumentOutOfRangeException("familyDocument is not a family document");
            if (null == familyParameter) throw new ArgumentNullException("familyParameter");


            string result = string.Empty;

            FamilyManager familyMgr = familyDocument.FamilyManager;

            // Before we can get a parameter's display or internal value, at least one Type must be defined for this family.
            // So we must make one if one doesn't exist.
            if (null == familyMgr.CurrentType)
            {
                familyMgr.NewType("TempType");
            }

            UnitType parameterUnitType = ConvertParameterTypeToUnitType(familyParameter.Definition.ParameterType);

            // Get the current display units ("Project Units") for the parameter's UnitType (e.g. Length, Angle, etc.)
            double? originalRoundingPrecision = null;
            DisplayUnitType originalUnits = get_CurrentDisplayUnitType(parameterUnitType,
                                                                       familyDocument,
                                                                       out originalRoundingPrecision);

            if ((originalUnits != desiredUnitsOfMeasure)
                ||
                (originalRoundingPrecision != roundingPrecision))
            {
                // Must store and re-store the original units of measure for the given unit type.

                // Temporarily set the current display units for the parameter's UnitType.  This is equivalent to changing
                // the Project Units.  
                SetCurrentDisplayUnitType(parameterUnitType,
                                          desiredUnitsOfMeasure,
                                          familyDocument, 
                                          roundingPrecision);
            }


            try
            {
                // Get the value in current display units of measure.
                result = familyMgr.CurrentType.AsValueString(familyParameter);
            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                // Restore what we need to.
                if ((originalUnits != desiredUnitsOfMeasure)
                    ||
                    (originalRoundingPrecision != roundingPrecision))
                {
                    // Now put back the original units of measure.
                    SetCurrentDisplayUnitType(parameterUnitType,
                                              originalUnits,
                                              familyDocument, 
                                              originalRoundingPrecision);
                }
            }

            return result;

        }





        /// <summary>
        /// Returns the value of the parameter in internal (base) units for that parameter's data type.
        /// </summary>
        /// <param name="familyDocument">The family document hosting the parameter</param>
        /// <param name="familyParameter">The parameter whose internal value is desired</param>
        /// <returns>The value of the parameter in Revit's internal units for that data type</returns>
        public static string get_FamilyParameterInternalValue(Document familyDocument,
                                                              FamilyParameter familyParameter)
        {

            // Following good SOA practices, don't trust the incoming parameters and verify
            // that they have values that can be worked with before doing anything.
            if (null == familyDocument) throw new ArgumentNullException("familyDocument");
            if (!familyDocument.IsFamilyDocument) throw new ArgumentOutOfRangeException("familyDocument is not a family document");
            if (null == familyParameter) throw new ArgumentNullException("familyParameter");

            string result = string.Empty;

            FamilyManager familyMgr = familyDocument.FamilyManager;

            // Before we can get a parameter's display or internal value, at least one Type must be defined for this family.
            // So we must make one if one doesn't exist.
            if (null == familyMgr.CurrentType)
            {
                familyMgr.NewType("TempType");
            }

            switch (familyParameter.StorageType)
            {
                case StorageType.Double:
                    double? internalDoubleValue = familyMgr.CurrentType.AsDouble(familyParameter);

                    if (internalDoubleValue.HasValue)
                    {
                        result = internalDoubleValue.Value.ToString();
                    }
                    break;

                case StorageType.Integer:
                    int? internalIntegerValue = familyMgr.CurrentType.AsInteger(familyParameter);

                    if (internalIntegerValue.HasValue)
                    {
                        result = internalIntegerValue.Value.ToString();
                    }
                    break;

                case StorageType.String:
                    result = familyMgr.CurrentType.AsString(familyParameter);
                    break;

                case StorageType.ElementId:

                    ElementId elementIDValue = familyMgr.CurrentType.AsElementId(familyParameter);
                    result = elementIDValue.IntegerValue.ToString();

                    break;

                default:
                    result = string.Empty;
                    break;
            }

            return result;

        }





        /// <summary>
        /// Returns the scale factor by which a value in the given forDisplayUnitType 
        /// units of measure must be multiplied so it can be stored directly into Revit, 
        /// which requires values be stored in internal units (unless using the SetValueString methods)
        /// </summary>
        /// <param name="familyDocument">The family document hosting the parameter</param>
        /// <param name="familyParameter">The parameter whose display value is desired</param>
        /// <param name="forDisplayUnitType">The units of measure for which the multiplier to get to the equivalent internal units value is needed</param>
        /// <returns></returns>
        public static string get_ScaleFactorToInternalUnits(Document familyDocument,
                                                            FamilyParameter familyParameter,
                                                            DisplayUnitType forDisplayUnitType)
        {
            // Following good SOA practices, don't trust the incoming parameters and verify
            // that they have values that can be worked with before doing anything.
            if (null == familyDocument) throw new ArgumentNullException("familyDocument");
            if (!familyDocument.IsFamilyDocument) throw new ArgumentOutOfRangeException("familyDocument is not a family document");
            if (null == familyParameter) throw new ArgumentNullException("familyParameter");

            string result = string.Empty;

            FamilyManager familyMgr = familyDocument.FamilyManager;

            // Store the original value for the parameter in whatever internal units Revit uses.
            string origValue = get_FamilyParameterInternalValue(familyDocument,
                                                                familyParameter);

            double? roundingPrecision = MAX_ROUNDING_PRECISION;

            try
            {

                // Set the value of 1.0 ***in the desired display units*** into the parameter
                SetFamilyParameterValue(familyDocument,
                                        familyParameter,
                                        forDisplayUnitType,
                                        "1.0",
                                        roundingPrecision);

                // Get the internal value back out.  This is the scale factor (multiplier) the user needs
                result = get_FamilyParameterInternalValue(familyDocument, 
                                                          familyParameter);

            }
            catch (Exception)
            {
                string resultMagnified = string.Empty;
                result = string.Empty;
                double doubleResult;

                // 1.0 may not always be a valid value, so we'll try a few others in the hopes
                // of finding a value in valid range.
                for (double magnifiedValue = 1000000; magnifiedValue > 0.0000001; magnifiedValue /= 10.0)
                {

                    try
                    {
                        SetFamilyParameterValue(familyDocument,
                                                familyParameter,
                                                forDisplayUnitType,
                                                magnifiedValue.ToString(),
                                                roundingPrecision);

                        // Get the internal value back out.  This is the scale factor (multiplier) the user needs
                        resultMagnified = get_FamilyParameterInternalValue(familyDocument,
                                                                           familyParameter);


                        if (double.TryParse(resultMagnified, out doubleResult))
                        {
                            // Found a happy multiplier of 1.0.  Divide by that multiplier to get our
                            // final result.
                            result = (doubleResult / magnifiedValue).ToString();
                            break;
                        }

                    }
                    catch (Exception)
                    {
                        continue;
                    }

                }

                if (result == string.Empty)
                {
                    throw new Exception("Unable to determine internal scale factor.");
                }

            }
            finally
            {
                // Put back the original value for the parameter, so we leave it unchanged.
                switch (familyParameter.StorageType)
                {
                    case StorageType.Double:
                        double doubleValue;
                        if (double.TryParse(origValue, out doubleValue))
                        {
                            familyDocument.FamilyManager.Set(familyParameter, doubleValue);
                        }
                        break;
                    case StorageType.Integer:
                        int integerValue;
                        if (int.TryParse(origValue, out integerValue))
                        {
                            familyDocument.FamilyManager.Set(familyParameter, integerValue);
                        }
                        break;
                    case StorageType.String:
                        familyDocument.FamilyManager.Set(familyParameter, origValue);
                        break;
                }
            }

            return result;
        }



        /// <summary>
        /// This function returns the type of parameter to create which is of the given UnitType (where possible).
        /// </summary>
        /// <param name="forUnitType">The UnitType for which we need to know the ParameterType</param>
        /// <returns></returns>
        /// <exception cref="ArgumentException">Thrown if it is not possible to convert the UnitType to a ParameterType.</exception>
        public static ParameterType ConvertUnitTypeToParameterType(UnitType unitType)
        {
            ParameterType result;

            // The conversion will be retrieved from the in-memory dictionary.
            if (BuildUnitTypeToParameterTypeMapping().TryGetValue(unitType, out result))
            {
                return result;
            }
            else
            {
                throw new ArgumentException("Cannot convert UnitType '" + unitType.ToString() +
                                            "' to a ParameterType.");
            }

        }



        /// <summary>
        /// This function returns the UnitType for the ParameterType provided (where possible).
        /// </summary>
        /// <param name="forParameterType">The ParameterType for which we need to know the UnitType</param>
        /// <returns></returns>
        /// <exception cref="ArgumentException">Thrown if it is not possible to convert the ParameterType to a UnitType.</exception>
        public static UnitType ConvertParameterTypeToUnitType(ParameterType parameterType)
        {

            // The conversion will from the in-memory dictionary.
            foreach (KeyValuePair<UnitType, ParameterType> kvp in BuildUnitTypeToParameterTypeMapping())
            {
                if (kvp.Value == parameterType)
                {
                    return kvp.Key;
                }
            }
            // If we made it this far, there's no entry in the dictionary.
            throw new ArgumentException("Cannot convert ParameterType '" + parameterType.ToString() +
                                        "' to a UnitType.");
        }



        /// <summary>
        /// Returns the list of DisplayUnitType values (units of measure) that are supported 
        /// for the specified UnitType (e.g. Length, Angle, HVAC Airflow etc.) 
        /// The returned list is the same as the choices seen in the Project Units
        /// UI when changing the units of measure for a given UnitType.
        /// This method should work on family or project documents.
        /// </summary>
        /// <param name="document">The document on which to operate</param>
        /// <param name="unitType">The UnitType (e.g. Length, Angle, HVAC Airflow etc.) whose list of supported units of measure are desired.</param>
        /// <returns>The list of DisplayUnitTypes for the given UnitType</returns>
        /// <remarks>
        /// NOTE:  This method can be VERY slow, but there may be no other way to do it.
        /// Caching the results in a dictionary helps, but prevents this method from being static.
        /// </remarks>
        public static List<DisplayUnitType> get_DisplayUnitTypeChoices(Document document,
                                                                       UnitType unitType)
        {
            // Following good SOA practices, don't trust the incoming parameters and verify
            // that they have values that can be worked with before doing anything.
            if (null == document) throw new ArgumentNullException("document");

            // Check the cache first, in case this one has already been asked for any time in the past.

            if (null == _DisplayUnitTypesCache)
            {
                _DisplayUnitTypesCache = new Dictionary<UnitType, List<DisplayUnitType>>();
            }

            if (_DisplayUnitTypesCache.ContainsKey(unitType))
            {
                return _DisplayUnitTypesCache[unitType];
            }

            List<DisplayUnitType> result = new List<DisplayUnitType>();

            FormatOptions formatOption = null;
            DisplayUnitType originalDisplayUnitType = DisplayUnitType.DUT_FEET_FRACTIONAL_INCHES;

            ProjectUnit projectUnit = document.ProjectUnit;

            try
            {
                formatOption = projectUnit.get_FormatOptions(unitType);
            }
            catch (Exception)
            {
                formatOption = null;
            }

            if (formatOption != null)
            {
                // Store the original units being used by the project so we can put them back when done.
                try
                {
                    originalDisplayUnitType = formatOption.Units;
                }
                catch (Exception)
                {
                }

                foreach (DisplayUnitType displayUnitType in Enum.GetValues(typeof(DisplayUnitType)))
                {
                    try
                    {
                        formatOption.Units = displayUnitType;
                        result.Add(displayUnitType);
                    }
                    catch (Exception)
                    {
                    }
                }

                // Restore the original units being used by the project
                try
                {
                    formatOption.Units = originalDisplayUnitType;
                }
                catch (Exception)
                {
                }

            }

            // Add this to the cache, in case it's asked for again.
            _DisplayUnitTypesCache.Add(unitType, result);

            return result;
        }


#endregion Public Functions






#region Private Helper Functions

        /// <summary>
        /// This method builds up the dictionary object which relates the UnitType enum values to their
        /// ParameterType enum counterpart values.
        /// </summary>
        /// <remarks>
        /// </remarks>
        /// <example>
        /// UnitType.UT_Angle = ParameterType.Angle
        /// </example>
        private static Dictionary<UnitType, ParameterType> BuildUnitTypeToParameterTypeMapping()
        {

            // This data could come from a file, or (better yet) from Revit itself...

            // Use a local variable with a shorter name as a quick reference to the module-level referenced dictionary
            Dictionary<UnitType, ParameterType> result = new Dictionary<UnitType, ParameterType>();

            result.Add(UnitType.UT_Angle, ParameterType.Angle);
            result.Add(UnitType.UT_Area, ParameterType.Area);
            result.Add(UnitType.UT_AreaForce, ParameterType.AreaForce);
            result.Add(UnitType.UT_AreaForcePerLength, ParameterType.AreaForcePerLength);
            //map.Add(UnitType.UT_AreaForceScale, ParameterType.???);
            result.Add(UnitType.UT_Color_Temperature, ParameterType.ColorTemperature);
            result.Add(UnitType.UT_Currency, ParameterType.Currency);
            //map.Add(UnitType.UT_DecSheetLength, ParameterType.???);
            result.Add(UnitType.UT_Electrical_Apparent_Power, ParameterType.ElectricalApparentPower);
            result.Add(UnitType.UT_Electrical_Current, ParameterType.ElectricalCurrent);
            result.Add(UnitType.UT_Electrical_Efficacy, ParameterType.ElectricalEfficacy);
            result.Add(UnitType.UT_Electrical_Frequency, ParameterType.ElectricalFrequency);
            result.Add(UnitType.UT_Electrical_Illuminance, ParameterType.ElectricalIlluminance);
            result.Add(UnitType.UT_Electrical_Luminance, ParameterType.ElectricalLuminance);
            result.Add(UnitType.UT_Electrical_Luminous_Flux, ParameterType.ElectricalLuminousFlux);
            result.Add(UnitType.UT_Electrical_Luminous_Intensity, ParameterType.ElectricalLuminousIntensity);
            result.Add(UnitType.UT_Electrical_Potential, ParameterType.ElectricalPotential);
            result.Add(UnitType.UT_Electrical_Power, ParameterType.ElectricalPower);
            result.Add(UnitType.UT_Electrical_Power_Density, ParameterType.ElectricalPowerDensity);
            result.Add(UnitType.UT_Electrical_Wattage, ParameterType.ElectricalWattage);
            result.Add(UnitType.UT_Force, ParameterType.Force);
            result.Add(UnitType.UT_ForceLengthPerAngle, ParameterType.ForceLengthPerAngle);
            result.Add(UnitType.UT_ForcePerLength, ParameterType.ForcePerLength);
            //map.Add(UnitType.UT_ForceScale, ParameterType.???);
            result.Add(UnitType.UT_HVAC_Airflow, ParameterType.HVACAirflow);
            result.Add(UnitType.UT_HVAC_Airflow_Density, ParameterType.HVACAirflowDensity);
            result.Add(UnitType.UT_HVAC_Airflow_Divided_By_Cooling_Load, ParameterType.HVACAirflowDividedByCoolingLoad);
            result.Add(UnitType.UT_HVAC_Airflow_Divided_By_Volume, ParameterType.HVACAirflowDividedByVolume);
            result.Add(UnitType.UT_HVAC_Area_Divided_By_Cooling_Load, ParameterType.HVACAreaDividedByCoolingLoad);
            result.Add(UnitType.UT_HVAC_Area_Divided_By_Heating_Load, ParameterType.HVACAreaDividedByHeatingLoad);
            result.Add(UnitType.UT_HVAC_CoefficientOfHeatTransfer, ParameterType.HVACCoefficientOfHeatTransfer);
            result.Add(UnitType.UT_HVAC_Cooling_Load, ParameterType.HVACCoolingLoad);
            result.Add(UnitType.UT_HVAC_Cooling_Load_Divided_By_Area, ParameterType.HVACCoolingLoadDividedByArea);
            result.Add(UnitType.UT_HVAC_Cooling_Load_Divided_By_Volume, ParameterType.HVACCoolingLoadDividedByVolume);
            result.Add(UnitType.UT_HVAC_CrossSection, ParameterType.HVACCrossSection);
            result.Add(UnitType.UT_HVAC_Density, ParameterType.HVACDensity);
            result.Add(UnitType.UT_HVAC_DuctSize, ParameterType.HVACDuctSize);
            result.Add(UnitType.UT_HVAC_Energy, ParameterType.HVACEnergy);
            result.Add(UnitType.UT_HVAC_Factor, ParameterType.HVACFactor);
            result.Add(UnitType.UT_HVAC_Friction, ParameterType.HVACFriction);
            result.Add(UnitType.UT_HVAC_HeatGain, ParameterType.HVACHeatGain);
            result.Add(UnitType.UT_HVAC_Heating_Load, ParameterType.HVACHeatingLoad);
            result.Add(UnitType.UT_HVAC_Heating_Load_Divided_By_Area, ParameterType.HVACHeatingLoadDividedByArea);
            result.Add(UnitType.UT_HVAC_Heating_Load_Divided_By_Volume, ParameterType.HVACHeatingLoadDividedByVolume);
            result.Add(UnitType.UT_HVAC_Power, ParameterType.HVACPower);
            result.Add(UnitType.UT_HVAC_Power_Density, ParameterType.HVACPowerDensity);
            result.Add(UnitType.UT_HVAC_Pressure, ParameterType.HVACPressure);
            result.Add(UnitType.UT_HVAC_Roughness, ParameterType.HVACRoughness);
            result.Add(UnitType.UT_HVAC_Slope, ParameterType.HVACSlope);
            result.Add(UnitType.UT_HVAC_Temperature, ParameterType.HVACTemperature);
            result.Add(UnitType.UT_HVAC_Velocity, ParameterType.HVACVelocity);
            result.Add(UnitType.UT_HVAC_Viscosity, ParameterType.HVACViscosity);
            result.Add(UnitType.UT_Length, ParameterType.Length);
            result.Add(UnitType.UT_LinearForce, ParameterType.LinearForce);
            result.Add(UnitType.UT_LinearForceLengthPerAngle, ParameterType.LinearForceLengthPerAngle);
            result.Add(UnitType.UT_LinearForcePerLength, ParameterType.LinearForcePerLength);
            // map.Add(UnitType.UT_LinearForceScale, ParameterType.???);
            result.Add(UnitType.UT_LinearMoment, ParameterType.LinearMoment);
            // map.Add(UnitType.UT_LinearMomentScale, ParameterType.???);
            result.Add(UnitType.UT_Moment, ParameterType.Moment);
            ///map.Add(UnitType.UT_MomentScale, ParameterType.???);
            result.Add(UnitType.UT_Number, ParameterType.Number);
            result.Add(UnitType.UT_PipeSize, ParameterType.PipeSize);
            result.Add(UnitType.UT_Piping_Density, ParameterType.PipingDensity);
            result.Add(UnitType.UT_Piping_Flow, ParameterType.PipingFlow);
            result.Add(UnitType.UT_Piping_Friction, ParameterType.PipingFriction);
            result.Add(UnitType.UT_Piping_Pressure, ParameterType.PipingPressure);
            result.Add(UnitType.UT_Piping_Roughness, ParameterType.PipingRoughness);
            result.Add(UnitType.UT_Piping_Slope, ParameterType.PipingSlope);
            result.Add(UnitType.UT_Piping_Temperature, ParameterType.PipingTemperature);
            result.Add(UnitType.UT_Piping_Velocity, ParameterType.PipingVelocity);
            result.Add(UnitType.UT_Piping_Viscosity, ParameterType.PipingViscosity);
            result.Add(UnitType.UT_Piping_Volume, ParameterType.PipingVolume);
            //map.Add(UnitType.UT_SheetLength, ParameterType.???);
            //map.Add(UnitType.UT_SiteAngle, ParameterType.???);
            result.Add(UnitType.UT_Slope, ParameterType.Slope);
            result.Add(UnitType.UT_Stress, ParameterType.Stress);
            result.Add(UnitType.UT_TemperalExp, ParameterType.TemperalExp);
            result.Add(UnitType.UT_UnitWeight, ParameterType.UnitWeight);
            result.Add(UnitType.UT_Volume, ParameterType.Volume);
            result.Add(UnitType.UT_WireSize, ParameterType.WireSize);

            return result;
        }


#endregion Private Helper Functions




    }
}
