﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using Autodesk.Revit;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Revit.SDK.Samples.UnitConversion.CS;
using ComboBox = System.Windows.Forms.ComboBox;


namespace Revit.SDK.Samples.UnitConversion.CS
{

    /// <summary>
    /// This form provides an interface for showing how units and conversions can work in Revit.
    /// Following good N-tier architecture, this user interface (presentation layer) is 
    /// relatively light, and the real conversion work gets done by methods in the UnitFunctions class,
    /// which is effectively a Business Logic class.
    /// </summary>
    public partial class frmUnitConversions : System.Windows.Forms.Form
    {


#region Module-level storage
        
        // A place to store a reference to the Revit application
        private UIApplication _Application;

        // A place to store a reference to the current parameter being manipulated
        private FamilyParameter _Parameter;

        private Document _DbDocument;
#endregion Module-level storage




#region Form Construction


        /// <summary>
        /// This is the only constructor for this form.  It requires a reference to the Revit application.
        /// </summary>
        /// <param name="application">A reference to the Revit application being used</param>
        public frmUnitConversions(UIApplication application)
        {
            // Following SOA practices, don't trust the data coming in.  Ensure everything
            // that may be needed exists.
            if (null == application)
            {
                throw new ArgumentNullException("application");
            }

            if (null == application.ActiveUIDocument.Document)
            {
                throw new ArgumentNullException("application.ActiveDocument");
            }

            if (!application.ActiveUIDocument.Document.IsFamilyDocument)
            {
                throw new ArgumentOutOfRangeException("application.ActiveDocument must be a family document.");
            }

            _DbDocument = application.ActiveUIDocument.Document;

            InitializeComponent();
            
            _Application = application;

        }


        /// <summary>
        /// This event fires the first time the form is displayed.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Form1_Load(object sender, EventArgs e)
        {
            // This will put all the unit type choices into the unit type combo box
            RepopulateUnitTypeComboBox(cboUnitType);

            // Preselect the default item.  That pre-selection will trigger associated
            // DisplayUnitType combo boxes to be populated as well.
            SelectUnitType(cboUnitType, UnitType.UT_Length);
            
            // Set the new value units to inches
            SelectDisplayUnitType(cboNewValueUnits, DisplayUnitType.DUT_DECIMAL_INCHES);

            // Set the 'convert to' units to feet and fractional issues
            SelectDisplayUnitType(cboConvertToUnits, DisplayUnitType.DUT_FEET_FRACTIONAL_INCHES);

        }

#endregion Form Construction






#region Form Control Event Handlers

        /// <summary>
        /// This event fires whenever the user wants to change the basic unit type (e.g. Length, Area,
        /// HVAC Airflow, etc.) with which they want to work.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void cboUnitType_SelectedIndexChanged(object sender, EventArgs e)
        {

            Cursor.Current = Cursors.WaitCursor;
            Cursor.Show();
            
            // Determine which Unit Type is now selected 
            UnitType selectedUnitType = GetSelectedUnitType(cboUnitType);

            // First, determine if it's possible to create a parameter of this UnitType so we can work with it.
            ParameterType parameterTypeToCreate = ParameterType.Invalid;

            try
            {
                // Call the busines logic class (UnitFunctions) to determine what ParameterType to create based on the selected UnitType
                parameterTypeToCreate = UnitFunctions.ConvertUnitTypeToParameterType(selectedUnitType);
                EnableConversions(true);
            }
            catch (Exception)
            {
                // Can't create a parameter for this unit type, so disable things.

                lblConvertedValue.Text = "Converted value:  <N/A>";
                lblInternalValue.Text = "Internal value:  <N/A>";
                txtScaleFactor.Text = string.Empty;

                EnableConversions(false);

                Cursor.Current = Cursors.Default;
                Cursor.Show();

                MessageBox.Show("Cannot create a parameter for UnitType '" + selectedUnitType.ToString() + "'", 
                                "No ParameterType Available", 
                                MessageBoxButtons.OK, 
                                MessageBoxIcon.Warning);
                return;
            }

            // If we made it this far, we can create a parameter and do a conversion for this UnitType.

            // Get or create the parameter to use.  For simplicity, give it the same name as the parameter type.
            _Parameter = _DbDocument.FamilyManager.get_Parameter(parameterTypeToCreate.ToString());

            if (null == _Parameter)
            {
                // Create a new parameter.
                _Parameter = _DbDocument.FamilyManager.AddParameter(parameterTypeToCreate.ToString(),
                                                                                    BuiltInParameterGroup.INVALID,
                                                                                    parameterTypeToCreate,
                                                                                    false);
            }

            // Fill the two Display Unit Type (units of measure) combo boxes with the 
            // choices for the newly-selected Unit Type
            RepopulateDisplayUnitTypeComboBox(cboNewValueUnits,
                                              selectedUnitType,
                                               _DbDocument);

            RepopulateDisplayUnitTypeComboBox(cboConvertToUnits,
                                              selectedUnitType,
                                               _DbDocument);

            // Now pre-select their choices to match the currently selected units of measure
            // for the selected UnitType for this document's settings.
            double? roundingPrecision;

            try
            {
                // Call the busines logic class (UnitFunctions) to get the current display units for the selected unit type
                DisplayUnitType projectUnits = UnitFunctions.get_CurrentDisplayUnitType(selectedUnitType,
                                                                                         _DbDocument,
                                                                                        out roundingPrecision);

                SelectDisplayUnitType(cboNewValueUnits, projectUnits);
                SelectDisplayUnitType(cboConvertToUnits, projectUnits);

            }
            catch (Exception)
            {
                Cursor.Current = Cursors.Default;
                Cursor.Show();

                MessageBox.Show("Unable to determine the display units for UnitType '" + selectedUnitType.ToString() + "'");

            }
            finally
            {
                double doubleTemp;
                if (double.TryParse(txtNewValue.Text, out doubleTemp))
                {
                    // Now click the "Set New Value" button to apply the current value to the new units.
                    btnSetNewValue_Click(this, null);
                }

                Cursor.Current = Cursors.Default;
                Cursor.Show();
            }

        }




        /// <summary>
        /// This event fires when the user wants to change the value for the current parameter.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnSetNewValue_Click(object sender, EventArgs e)
        {
            try
            {

                double? roundingPrecision = null;
                double convertedRoundingPrecision;

                if (double.TryParse(txtRoundingPrecision.Text, out convertedRoundingPrecision))
                {
                    roundingPrecision = convertedRoundingPrecision;
                }

                // Find the DisplayUnitType enum for the currently selected combo box text
                DisplayUnitType dataUnitsOfMeasure = GetSelectedDisplayUnitType(cboNewValueUnits);

                // Call the busines logic class (UnitFunctions) to change the value of the parameter on the document
                // to whatever the user typed in.  This is equivalent to how the Revit UI works...the value sent in
                // is in whatever units of measure the user has selected, but ultimately it gets stored in whatever
                // internal units Revit uses.
                UnitFunctions.SetFamilyParameterValue(_DbDocument,
                                                      _Parameter,
                                                      dataUnitsOfMeasure,
                                                      txtNewValue.Text,
                                                      roundingPrecision);

                // Now update the values for the parameter as seen on screen.
                RefreshCurrentParameterValues();

            }
            catch (Exception ex)
            {

                // Clear out the old results, because they are not valid.
                lblConvertedValue.Text = "Converted value:  <Cannot process new value>";
                lblInternalValue.Text = "Internal value:  <Cannot process new value>";
                
                MessageBox.Show("Error setting value " + txtNewValue.Text + 
                                " -- This value may be out of range for the current units of measure.\r\n" + 
                                "A smaller or larger new value may be required.\r\n\r\n" +
                                "Error Message:  " + ex.Message, 
                                "Error Setting New Value", 
                                MessageBoxButtons.OK, 
                                MessageBoxIcon.Error);
            }
        }



        /// <summary>
        /// This event fires whenever the user changes how they want the converted value to be displayed
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void cboConvertToUnits_SelectedIndexChanged(object sender, EventArgs e)
        {
            RefreshCurrentParameterValues();
        }


        /// <summary>
        /// This event fires whenever the user changes the precision in which to show the converted value.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void txtConversionRoundingPrecision_TextChanged(object sender, EventArgs e)
        {
            RefreshCurrentParameterValues();
        }



        /// <summary>
        /// This event fires whenever the user changes the units of measure from which their data is
        /// to be entered.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void cboNewValueUnits_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                // Update the scale factor value to show it to the user.
                DisplayUnitType selectedDisplayUnits = GetSelectedDisplayUnitType(cboNewValueUnits);

                // Call the busines logic class (UnitFunctions) to get the conversion factor the user
                // would use to multiply their value by to get the internal value Revit expects for 
                // the desired display unit type.
                string scaleFactor = UnitFunctions.get_ScaleFactorToInternalUnits(_DbDocument,
                                                                                  _Parameter,
                                                                                  selectedDisplayUnits);

                txtScaleFactor.Text = scaleFactor;

                // Set the value in now-changed units so the conversion will appear.
                btnSetNewValue_Click(this, null);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.ToString(), "Error");
            }

        }


#endregion Form Control Event Handlers






#region Helper Functions


        /// <summary>
        /// This function simply updates the labels that show the current display and
        /// internal value for the parameter being used.
        /// </summary>
        private void RefreshCurrentParameterValues()
        {
            try
            {
                double? roundingPrecision = null;
                double convertedRoundingPrecision;

                if (double.TryParse(txtConversionRoundingPrecision.Text, out convertedRoundingPrecision))
                {
                    roundingPrecision = convertedRoundingPrecision;
                }

                // Call the busines logic class (UnitFunctions) to get the internal value currently being used
                // for the parameter
                string internalValue = UnitFunctions.get_FamilyParameterInternalValue(_DbDocument,
                                                                                      _Parameter);


                DisplayUnitType desiredUnitsOfMeasure = GetSelectedDisplayUnitType(cboConvertToUnits);

                // Call the busines logic class (UnitFunctions) to get the display value for the
                // parameter when providing the specific display units that are desired (may be different
                // than current project display units).
                string displayValue = UnitFunctions.get_FamilyParameterDisplayValue(_DbDocument,
                                                                                    _Parameter,
                                                                                    desiredUnitsOfMeasure,
                                                                                    roundingPrecision);

                lblConvertedValue.Text = "Converted value:  " + displayValue;
                lblInternalValue.Text = "Internal value:  " + internalValue;

            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.ToString(), "RefreshCurrentParameterValues Unhandled Exception");
            }

        }



        /// <summary>
        /// Enables or disables controls to allow or prevent the user from converting values
        /// (e.g. based on whether or not a parameter can be created / used)
        /// </summary>
        /// <param name="bNowEnabled"></param>
        private void EnableConversions(bool nowEnabled)
        {
            cboNewValueUnits.Items.Clear();
            cboNewValueUnits.Enabled = nowEnabled;
            txtNewValue.Enabled = nowEnabled;
            txtRoundingPrecision.Enabled = nowEnabled;
            btnSetNewValue.Enabled = nowEnabled;
            cboConvertToUnits.Items.Clear();
            cboConvertToUnits.Enabled = nowEnabled;
            txtConversionRoundingPrecision.Enabled = nowEnabled;
        }

                
#endregion Helper Functions




        



#region Combo Box Data Functions
        

        /// <summary>
        /// Adds the Unit Type enum descriptions as choices to a combo box.
        /// </summary>
        /// <param name="comboBox">The combo box being operated on</param>
        private void RepopulateUnitTypeComboBox(ComboBox comboBox)
        {
            // Following good SOA practices, don't trust the incoming parameters and verify
            // that they have values that can be worked with before doing anything.
            if (null == comboBox) throw new ArgumentNullException("comboBox");

            comboBox.Items.Clear();

            // Populate the combo box choices.
            foreach (UnitType unitType in Enum.GetValues(typeof(UnitType)))
            {
                comboBox.Items.Add(unitType.ToString());
            }
        }


        /// <summary>
        /// Returns the UnitType enum associated with the selected combo box item.
        /// </summary>
        /// <param name="comboBox">The combo box being operated on</param>
        /// <returns></returns>
        private UnitType GetSelectedUnitType(ComboBox comboBox)
        {
            // Following good SOA practices, don't trust the incoming parameters and verify
            // that they have values that can be worked with before doing anything.
            if (null == comboBox) throw new ArgumentNullException("comboBox");

            // Find the UnitType enum for the currently selected combo box text
            foreach (UnitType candidate in Enum.GetValues(typeof(UnitType)))
            {
                if (candidate.ToString() == comboBox.Text)
                {
                    return candidate;
                }
            }

            return UnitType.UT_Length;
        }



        /// <summary>
        /// Reloads the combo box with the DisplayUnitType choices for the given unit type.
        /// </summary>
        /// <param name="comboBox">The combo box to reload</param>
        /// <param name="unitTypeFilter">The UnitType for which available DisplayUnitType values are needed</param>
        /// <param name="document">The document being operated on</param>
        private void RepopulateDisplayUnitTypeComboBox(ComboBox comboBox,
                                                      UnitType unitTypeFilter,
                                                       Document document)
        {
            // Following good SOA practices, don't trust the incoming parameters and verify
            // that they have values that can be worked with before doing anything.
            if (null == comboBox) throw new ArgumentNullException("comboBox");
            if (null == document) throw new ArgumentNullException("document");
            
            comboBox.Items.Clear();

            // Call the busines logic class (UnitFunctions) to get the choices for the units
            // of measure available for the given UnitType.  This is the list normally seen
            // when selecting new settings in the Projet Units dialog.
            List<DisplayUnitType> oDisplayUnitTypeChoices = UnitFunctions.get_DisplayUnitTypeChoices(document, 
                                                                                                     unitTypeFilter); 

            // Populate the combo box choices.
            foreach (DisplayUnitType displayUnitType in oDisplayUnitTypeChoices)
            {
                comboBox.Items.Add(displayUnitType.ToString());
            }

        }


        /// <summary>
        /// Returns the selected DisplayUnitType from the given combo box.
        /// </summary>
        /// <param name="comboBox">The combo box on which to operate</param>
        /// <returns></returns>
        private DisplayUnitType GetSelectedDisplayUnitType(ComboBox comboBox)
        {
            // Following good SOA practices, don't trust the incoming parameters and verify
            // that they have values that can be worked with before doing anything.
            if (null == comboBox) throw new ArgumentNullException("comboBox");

            // Find the UnitType enum for the currently selected combo box text
            foreach (DisplayUnitType candidate in Enum.GetValues(typeof(DisplayUnitType)))
            {
                if (candidate.ToString() == comboBox.Text)
                {
                    return candidate;
                }
            }

            // If we made it this far, something is wrong, just return a generic value
            return DisplayUnitType.DUT_GENERAL;
        }



        /// <summary>
        /// Selects the combo box item matching the desired DisplayUnitType
        /// </summary>
        /// <param name="comboBox">The combo box on which to operate</param>
        /// <param name="displayUnitType">The desired DisplayUnitType to select</param>
        private void SelectDisplayUnitType(ComboBox comboBox,
                                           DisplayUnitType displayUnitType)
        {
            // Following good SOA practices, don't trust the incoming parameters and verify
            // that they have values that can be worked with before doing anything.
            if (null == comboBox) throw new ArgumentNullException("comboBox");

            string desiredDisplayUnitType = displayUnitType.ToString();

            for (int counter = 0; counter < comboBox.Items.Count; counter++)
            {
                if (desiredDisplayUnitType == (string)comboBox.Items[counter])
                {
                    comboBox.SelectedIndex = counter;
                    return;
                }
            }

        }



        /// <summary>
        /// Selects the combo box item matching the desired UnitType.
        /// </summary>
        /// <param name="comboBox">The combo box on which to operate</param>
        /// <param name="unitType">The desired UnitType to select</param>
        private void SelectUnitType(ComboBox comboBox,
                                    UnitType unitType)
        {
            // Following good SOA practices, don't trust the incoming parameters and verify
            // that they have values that can be worked with before doing anything.
            if (null == comboBox) throw new ArgumentNullException("comboBox");

            string desiredUnitType = unitType.ToString();

            for (int counter = 0; counter < comboBox.Items.Count; counter++)
            {
                if (desiredUnitType == (string)comboBox.Items[counter])
                {
                    comboBox.SelectedIndex = counter;
                    return;
                }
            }

        }

#endregion Combo Box Data Functions




    }
}
