﻿#region Header
//
// CmdListMarks.cs - list all door marks
//
// Copyright (C) 2009 by Jeremy Tammik,
// Autodesk Inc. All rights reserved.
//
#endregion // Header

#region Namespaces
using System.Collections.Generic;
using System.Diagnostics;
using Autodesk.Revit;
using Autodesk.Revit.Elements;
using Autodesk.Revit.Parameters;
using CmdResult = Autodesk.Revit.IExternalCommand.Result;
#endregion // Namespaces

namespace BuildingCoder
{
  class CmdListMarks : IExternalCommand
  {
    static bool _modify_existing_marks = true;
    const string _the_answer = "42";

    public CmdResult Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      Application app = commandData.Application;
      Document doc = app.ActiveDocument;

      Autodesk.Revit.Creation.Filter cf
        = app.Create.Filter;

      List<Element> doors
        = new List<Element>();

      Filter f1 = cf.NewTypeFilter(
        typeof( FamilyInstance ) );

      Filter f2 = cf.NewCategoryFilter(
        BuiltInCategory.OST_Doors );

      Filter f = cf.NewLogicAndFilter( f1, f2 );

      int n = doc.get_Elements( f, doors );

      Debug.Print( "{0} door{1} found.", 
        n, Util.PluralSuffix( n ) );

      if( 0 < n )
      {
        Dictionary<string, List<Element>> marks 
          = new Dictionary<string, List<Element>>();

        foreach( FamilyInstance door in doors )
        {
          string mark = door.get_Parameter( 
            BuiltInParameter.ALL_MODEL_MARK )
            .AsString();

          if( !marks.ContainsKey( mark ) )
          {
            marks.Add( mark, new List<Element>() );
          }
          marks[mark].Add( door );
        }
        
        List<string> keys = new List<string>( 
          marks.Keys );

        keys.Sort();

        n = keys.Count;

        Debug.Print( "{0} door mark{1} found{2}",
          n, Util.PluralSuffix( n ), 
          Util.DotOrColon( n ) );

        foreach( string mark in keys )
        {
          n = marks[mark].Count;

          Debug.Print( "  {0}: {1} door{2}",
            mark, n, Util.PluralSuffix( n ) );
        }
      }

      n = 0; // count how many elements are modified

      if( _modify_existing_marks )
      {
        ElementSet els = doc.Selection.Elements;

        foreach( Element e in els )
        {
          if( e is FamilyInstance
            && null != e.Category
            && (int) BuiltInCategory.OST_Doors 
              == e.Category.Id.Value )
          {
            e.get_Parameter( 
              BuiltInParameter.ALL_MODEL_MARK )
              .Set( _the_answer );

            ++n;
          }
        }
      }

      // return Succeeded only if we wish to commit 
      // the transaction to modify the database:

      return 0 < n 
        ? CmdResult.Succeeded 
        : CmdResult.Failed;
    }
  }
}
