﻿#region Header
//
// CmdUnrotateNorth.cs - transform element location back to 
// original coordinates to cancel effect of rotating project north
//
// Copyright (C) 2009 by Jeremy Tammik,
// Autodesk Inc. All rights reserved.
//
#endregion // Header

#region Namespaces
using System;
using System.Diagnostics;
using Autodesk.Revit;
using Autodesk.Revit.Site;
using XYZ = Autodesk.Revit.Geometry.XYZ;
using Transform = Autodesk.Revit.Geometry.Transform;
#endregion // Namespaces

namespace BuildingCoder
{
  class CmdUnrotateNorth
  {
    /// <summary>
    /// Return a location for the given element.
    /// Use either the element's LocationPoint Point property,
    /// or its LocationCurve start point, whichever is available.
    /// </summary>
    /// <param name="p">Return element location point</param>
    /// <param name="e">Revit Element</param>
    /// <returns>True if a location point is available for the given element, 
    /// otherwise false.</returns>
    bool GetElementLocation( 
      out XYZ p, 
      Element e )
    {
      p = XYZ.Zero;
      bool rc = false;
      Location loc = e.Location;
      if( null != loc )
      {
        LocationPoint lp = loc as LocationPoint;
        if( null != lp )
        {
          p = lp.Point;
          rc = true;
        }
        else
        {
          LocationCurve lc = loc as LocationCurve;

          Debug.Assert( null != lc, 
            "expected location to be either point or curve" );

          p = lc.Curve.get_EndPoint( 0 );
          rc = true;
        }
      }
      return rc;
    }

    public IExternalCommand.Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      Application app = commandData.Application;
      Document doc = app.ActiveDocument;

      ElementSet els = doc.Selection.Elements;

      if( 1 != els.Size )
      {
        message = "Please select a single element.";
      }
      else
      {
        ElementSetIterator it = els.ForwardIterator();
        it.MoveNext();

        Element e = it.Current as Element;

        XYZ p;
        if( !GetElementLocation( out p, e ) )
        {
          message
            = "Selected element has no location defined.";

          Debug.Print( message );
        }
        else
        {
          string msg 
            = "Selected element location: "
            + Util.PointString( p );

          double pna = 0;

          foreach( ProjectLocation location
            in doc.ProjectLocations )
          {
            ProjectPosition projectPosition
              = location.get_ProjectPosition( XYZ.Zero );

            pna = projectPosition.Angle;

            msg +=
              "\nAngle between project north and true north: "
              + Util.AngleString( pna );

            Transform t = Transform.get_Rotation(
              XYZ.Zero, XYZ.BasisZ, pna );

            msg +=
              "\nUnrotated element location: "
              + Util.PointString( t.OfPoint( p ) );

            Util.InfoMsg( msg );
          }
        }
      }
      return IExternalCommand.Result.Failed;
    }
  }
}
