﻿#region Header
//
// CmdMirror.cs - mirror some elements.
//
// Copyright (C) 2009-2010 by Jeremy Tammik,
// Autodesk Inc. All rights reserved.
//
#endregion // Header

#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
#endregion // Namespaces

namespace BuildingCoder
{
  [Transaction( TransactionMode.Automatic )]
  [Regeneration( RegenerationOption.Manual )]
  class CmdMirror : IExternalCommand
  {
    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;

      Application app = uiapp.Application;
      Document doc = uidoc.Document;

      ElementSet els = uidoc.Selection.Elements;

      Line line = app.Create.NewLine(
        XYZ.Zero, XYZ.BasisX, true );

      doc.Mirror( els, line );

      return Result.Succeeded;
    }
  }

  [Transaction( TransactionMode.Automatic )]
  [Regeneration( RegenerationOption.Manual )]
  class CmdMirrorListAdded : IExternalCommand
  {
    string _msg = "The following elements were mirrored:\r\n";

    /// <summary>
    /// Return all elements that are not ElementType objects.
    /// </summary>
    FilteredElementCollector GetElements( Document doc )
    {
      FilteredElementCollector collector
        = new FilteredElementCollector( doc );
      return collector.WhereElementIsNotElementType();
    }

    /// <summary>
    /// Return the current number of non-ElementType elements.
    /// </summary>
    int GetElementCount( Document doc )
    {
      return GetElements( doc ).ToElements().Count;
    }

    /// <summary>
    /// Return all database elements after the given number n.
    /// </summary>
    List<Element> GetElementsAfter( int n, Document doc )
    {
      List<Element> a = new List<Element>();
      FilteredElementCollector c = GetElements( doc );
      int i = 0;

      foreach( Element e in c )
      {
        ++i;

        if( n < i )
        {
          a.Add( e );
        }
      }
      return a;
    }

    /// <summary>
    /// Return all elements in the entire document 
    /// whose element id is greater than 'lastId'.
    /// </summary>
    FilteredElementCollector GetElementsAfter(
      Document doc,
      ElementId lastId )
    {
      BuiltInParameter bip = BuiltInParameter.ID_PARAM;

      ParameterValueProvider provider
        = new ParameterValueProvider( 
          new ElementId( bip ) );

      FilterNumericRuleEvaluator evaluator
        = new FilterNumericGreater();

      FilterRule rule = new FilterElementIdRule(
        provider, evaluator, lastId );

      ElementParameterFilter filter
        = new ElementParameterFilter( rule );

      FilteredElementCollector collector
        = new FilteredElementCollector( doc );

      return collector.WherePasses( filter );
    }

    /// <summary>
    /// Return all elements from the given collector
    /// whose element id is greater than 'lastId'.
    /// </summary>
    FilteredElementCollector GetElementsAfter(
      FilteredElementCollector input,
      ElementId lastId )
    {
      BuiltInParameter bip = BuiltInParameter.ID_PARAM;

      ParameterValueProvider provider 
        = new ParameterValueProvider( 
          new ElementId( bip ) );

      FilterNumericRuleEvaluator evaluator 
        = new FilterNumericGreater();

      FilterRule rule = new FilterElementIdRule( 
        provider, evaluator, lastId );

      ElementParameterFilter filter 
        = new ElementParameterFilter( rule );

      return input.WherePasses( filter );
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Application app = uiapp.Application;
      Document doc = uidoc.Document;

      ElementSet els = uidoc.Selection.Elements;

      Line line = app.Create.NewLine(
        XYZ.Zero, XYZ.BasisX, true );

      using( SubTransaction t = new SubTransaction( doc ) )
      {
        // this approach works in both Revit 2010 and 2011:

        t.Start();

        int n = GetElementCount( doc );

        doc.Mirror( els, line );

        List<Element> a = GetElementsAfter( n, doc );

        string s = _msg;

        foreach( Element e in a )
        {
          s += string.Format( "\r\n  {0}",
            Util.ElementDescription( e ) );
        }
        Util.InfoMsg( s );
        
        t.RollBack();
      }

      using( SubTransaction t = new SubTransaction( doc ) )
      {
        // here is an idea for a new approach in 2011:

        t.Start();

        FilteredElementCollector a = GetElements( doc );
        int i = a.Max<Element>( e => e.Id.IntegerValue );
        ElementId maxId = new ElementId( i );

        doc.Mirror( els, line );

        // get all elements in document with an
        // element id greater than maxId:

        a = GetElementsAfter( doc, maxId );

        string s = _msg;

        foreach( Element e in a )
        {
          s += string.Format( "\r\n  {0}",
            Util.ElementDescription( e ) );
        }
        Util.InfoMsg( s );
        
        t.RollBack();
      }

      using( SubTransaction t = new SubTransaction( doc ) )
      {
        t.Start();

        FilteredElementCollector a = GetElements( doc );
        int i = a.Max<Element>( e => e.Id.IntegerValue );
        ElementId maxId = new ElementId( i );

        doc.Mirror( els, line );

        // only look at non-ElementType elements
        // instead of all document elements:

        a = GetElements( doc );
        a = GetElementsAfter( a, maxId );

        string s = _msg;

        foreach( Element e in a )
        {
          s += string.Format( "\r\n  {0}",
            Util.ElementDescription( e ) );
        }
        Util.InfoMsg( s );

        t.RollBack();
      }
      return Result.Succeeded;
    }
  }
}
