﻿#region Header
//
// CmdCollectorPerformance.cs - benchmark Revit 2011 API collector performance
//
// Copyright (C) 2010 by Jeremy Tammik, Autodesk Inc. All rights reserved.
//
#endregion // Header

#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
#endregion // Namespaces

namespace BuildingCoder
{
  [Transaction( TransactionMode.Automatic )]
  [Regeneration( RegenerationOption.Manual )]
  class CmdCollectorPerformance : IExternalCommand
  {
    Document _doc;

    #region Retrieve openings in wall
    /// <summary>
    /// Retrieve all openings in a given wall.
    /// </summary>
    void GetOpeningsInWall( 
      Document doc, 
      Wall wall )
    {
      ElementId id = wall.Id;

      BuiltInCategory bic
        = BuiltInCategory.OST_SWallRectOpening;

      FilteredElementCollector collector
        = new FilteredElementCollector( doc );

      collector.OfClass( typeof( Opening ) );
      collector.OfCategory( bic );

      // explicit iteration and manual 
      // checking of a property:

      List<Element> openings = new List<Element>();

      foreach( Opening e in collector )
      {
        if( e.Host.Id.Equals( id ) )
        {
          openings.Add( e );
        }
      }

      // using LINQ:

      IEnumerable<Opening> openingsOnLevelLinq =
        from e in collector.Cast<Opening>()
        where e.Host.Level.Id.Equals( id )
        select e;

      // using an anonymous method:

      IEnumerable<Opening> openingsOnLevelAnon =
        collector.Cast<Opening>().Where<Opening>( e
          => e.Host.Id.Equals( id ) );
    }
    #endregion // Retrieve openings in wall

    #region Retrieve stairs on level
    /// <summary>
    /// Retrieve all stairs on a given level.
    /// </summary>
    FilteredElementCollector
      GetStairsOnLevel(
        Document doc,
        Level level )
    {
      ElementId id = level.Id;

      BuiltInCategory bic 
        = BuiltInCategory.OST_Stairs;

      FilteredElementCollector collector
        = new FilteredElementCollector( doc );

      collector.OfCategory( bic );

      // explicit iteration and manual 
      // checking of a property:

      List<Element> stairs = new List<Element>();

      foreach( Element e in collector )
      {
        if( e.Level.Id.Equals( id ) )
        {
          stairs.Add( e );
        }
      }

      // using LINQ:

      IEnumerable<Element> stairsOnLevelLinq =
        from e in collector
        where e.Level.Id.Equals( id )
        select e;

      // using an anonymous method:

      IEnumerable<Element> stairsOnLevelAnon =
        collector.Where<Element>( e 
          => e.Level.Id.Equals( id ) );

      // using a parameter filter:

      BuiltInParameter bip
        = BuiltInParameter.STAIRS_BASE_LEVEL_PARAM;

      ParameterValueProvider provider
        = new ParameterValueProvider(
          new ElementId( bip ) );

      FilterNumericRuleEvaluator evaluator
        = new FilterNumericEquals();

      FilterRule rule = new FilterElementIdRule(
        provider, evaluator, id );

      ElementParameterFilter filter
        = new ElementParameterFilter( rule );

      return collector.WherePasses( filter );
    }
    #endregion // Retrieve stairs on level

    #region Helper method to create some elements to play with
    /// <summary>
    /// Create a new level at the given elevation.
    /// </summary>
    Level CreateLevel( int elevation )
    {
      Level level = _doc.Create.NewLevel( elevation );
      level.Name = "Level " + elevation.ToString();
      return level;
    }
    #endregion // Helper method to create some elements to play with

    #region Methods to measure collector and post processing speed
    /// <summary>
    /// An empty method that does nothing.
    /// </summary>
    Element EmptyMethod( Type type )
    {
      return null;
    }

    /// <summary>
    /// An empty method that does nothing.
    /// </summary>
    Element EmptyMethod( Type type, string name )
    {
      return null;
    }

    /// <summary>
    /// Return all non ElementType elements.
    /// </summary>
    /// <returns></returns>
    FilteredElementCollector GetNonElementTypeElements()
    {
      return new FilteredElementCollector( _doc )
        .WhereElementIsNotElementType();
    }

    /// <summary>
    /// Return a collector of all elements of the given type.
    /// </summary>
    FilteredElementCollector GetElementsOfType( 
      Type type )
    {
      return new FilteredElementCollector( _doc )
        .OfClass( type );
    }

    /// <summary>
    /// Return the first element of the given 
    /// type without any further filtering.
    /// </summary>
    Element GetFirstElementOfType(
      Type type )
    {
      return new FilteredElementCollector( _doc )
        .OfClass( type )
        .FirstElement();
    }

    /// <summary>
    /// Return the first element of the given 
    /// type and name using a parameter filter.
    /// </summary>
    Element GetFirstNamedElementOfTypeUsingParameterFilter(
      Type type,
      string name )
    {
      FilteredElementCollector a
        = GetElementsOfType( type );

      BuiltInParameter bip 
        = BuiltInParameter.ELEM_NAME_PARAM;

      ParameterValueProvider provider 
        = new ParameterValueProvider( 
          new ElementId( bip ) );

      FilterStringRuleEvaluator evaluator 
        = new FilterStringEquals();

      FilterRule rule = new FilterStringRule( 
        provider, evaluator, name, true );

      ElementParameterFilter filter 
        = new ElementParameterFilter( rule );

      return a.WherePasses( filter ).FirstElement();
    }

    #region Methods to measure post processing speed retrieving all elements
    /// <summary>
    /// Return a list of all elements matching 
    /// the given type using explicit code to test 
    /// the element type.
    /// </summary>
    List<Element> GetElementsOfTypeUsingExplicitCode( 
      Type type )
    {
      FilteredElementCollector a 
        = GetNonElementTypeElements();

      List<Element> b = new List<Element>();
      foreach( Element e in a )
      {
        if( e.GetType().Equals( type ) )
        {
          b.Add( e );
        }
      }
      return b;
    }

    /// <summary>
    /// Return a list of all elements matching 
    /// the given type using a LINQ query to test 
    /// the element type.
    /// </summary>
    IEnumerable<Element> GetElementsOfTypeUsingLinq( 
      Type type )
    {
      FilteredElementCollector a 
        = GetNonElementTypeElements();

      IEnumerable<Element> b =
        from e in a
        where e.GetType().Equals( type )
        select e;

      return b;
    }
    #endregion // Methods to measure post processing speed retrieving all elements

    #region Methods to measure post processing speed retrieving a named element
    /// <summary>
    /// Return the first element of the given 
    /// type and name using explicit code.
    /// </summary>
    Element GetFirstNamedElementOfTypeUsingExplicitCode(
      Type type,
      string name )
    {
      FilteredElementCollector a 
        = GetElementsOfType( type );
      //
      // explicit iteration and manual checking of a property:
      //
      Element ret = null;
      foreach( Element e in a )
      {
        if( e.Name.Equals( name ) )
        {
          ret = e;
          break;
        }
      }
      return ret;
    }

    /// <summary>
    /// Return the first element of the given 
    /// type and name using LINQ.
    /// </summary>
    Element GetFirstNamedElementOfTypeUsingLinq(
      Type type,
      string name )
    {
      FilteredElementCollector a
        = GetElementsOfType( type );
      //
      // using LINQ:
      //
      IEnumerable<Element> elementsByName =
        from e in a
        where e.Name.Equals( name )
        select e;

      return elementsByName.First<Element>();
    }

    /// <summary>
    /// Return the first element of the given 
    /// type and name using an anonymous method
    /// to define a named method.
    /// </summary>
    Element GetFirstNamedElementOfTypeUsingAnonymousButNamedMethod(
      Type type,
      string name )
    {
      FilteredElementCollector a
        = GetElementsOfType( type );
      //
      // using an anonymous method to define a named method:
      //
      Func<Element, bool> nameEquals = e => e.Name.Equals( name );
      return a.First<Element>( nameEquals );
    }

    /// <summary>
    /// Return the first element of the given 
    /// type and name using an anonymous method.
    /// </summary>
    Element GetFirstNamedElementOfTypeUsingAnonymousMethod(
      Type type,
      string name )
    {
      FilteredElementCollector a
        = GetElementsOfType( type );
      //
      // using an anonymous method:
      //
      return a.First<Element>(
        e => e.Name.Equals( name ) );
    }
    #endregion // Methods to measure post processing speed retrieving a named element

    #endregion // Methods to measure collector and post processing speed

    /// <summary>
    /// Benchmark several different approaches to 
    /// using filtered collectors to retrieve 
    /// all levels in the model, 
    /// and measure the time required to 
    /// create IList and List collections
    /// from them.
    /// </summary>
    void BenchmarkAllLevels( int nLevels )
    {
      Type t = typeof( Level );
      int n;

      using( JtTimer pt = new JtTimer( 
        "Empty method *" ) )
      {
        EmptyMethod( t );
      }

      using( JtTimer pt = new JtTimer( 
        "NotElementType *" ) )
      {
        FilteredElementCollector a 
          = GetNonElementTypeElements();
      }

      using( JtTimer pt = new JtTimer( 
        "NotElementType as IList *" ) )
      {
        IList<Element> a 
          = GetNonElementTypeElements().ToElements();
        n = a.Count;
      }
      Debug.Assert( nLevels <= n, 
        "expected to retrieve all non-element-type elements" );

      using( JtTimer pt = new JtTimer( 
        "NotElementType as List *" ) )
      {
        List<Element> a = new List<Element>( 
          GetNonElementTypeElements() );

        n = a.Count;
      }
      Debug.Assert( nLevels <= n, 
        "expected to retrieve all non-element-type elements" );

      using( JtTimer pt = new JtTimer( "Explicit" ) )
      {
        List<Element> a 
          = GetElementsOfTypeUsingExplicitCode( t );

        n = a.Count;
      }
      Debug.Assert( nLevels == n, 
        "expected to retrieve all levels" );

      using( JtTimer pt = new JtTimer( "Linq" ) )
      {
        IEnumerable<Element> a = 
          GetElementsOfTypeUsingLinq( t );

        n = a.Count<Element>();
      }
      Debug.Assert( nLevels == n, 
        "expected to retrieve all levels" );

      using( JtTimer pt = new JtTimer( 
        "Linq as List" ) )
      {
        List<Element> a = new List<Element>( 
          GetElementsOfTypeUsingLinq( t ) );

        n = a.Count;
      }
      Debug.Assert( nLevels == n, 
        "expected to retrieve all levels" );

      using( JtTimer pt = new JtTimer( "Collector" ) )
      {
        FilteredElementCollector a 
          = GetElementsOfType( t );
      }

      using( JtTimer pt = new JtTimer( 
        "Collector as IList" ) )
      {
        IList<Element> a 
          = GetElementsOfType( t ).ToElements();

        n = a.Count;
      }
      Debug.Assert( nLevels == n, 
        "expected to retrieve all levels" );

      using( JtTimer pt = new JtTimer( 
        "Collector as List" ) )
      {
        List<Element> a = new List<Element>( 
          GetElementsOfType( t ) );

        n = a.Count;
      }
      Debug.Assert( nLevels == n, 
        "expected to retrieve all levels" );
    }

    /// <summary>
    /// Benchmark the use of a parameter filter versus 
    /// various kinds of post processing of the 
    /// results returned by the filtered element 
    /// collector to find the level specified by 
    /// iLevel.
    /// </summary>
    void BenchmarkSpecificLevel( int iLevel )
    {
      Type t = typeof( Level );
      string name = "Level " + iLevel.ToString();

      using( JtTimer pt = new JtTimer( 
        "Empty method *" ) )
      {
        Element level
          = EmptyMethod(
            t, name );
      }

      using( JtTimer pt = new JtTimer( 
        "Collector with no name check *" ) )
      {
        Element level = GetFirstElementOfType( t );
      }

      using( JtTimer pt = new JtTimer( 
        "Parameter filter" ) )
      {
        Element level
          = GetFirstNamedElementOfTypeUsingParameterFilter(
            t, name );
      }

      using( JtTimer pt = new JtTimer( "Explicit" ) )
      {
        Element level
          = GetFirstNamedElementOfTypeUsingExplicitCode(
            t, name );
      }
      using( JtTimer pt = new JtTimer( "Linq" ) )
      {
        Element level
          = GetFirstNamedElementOfTypeUsingLinq(
            t, name );
      }
      using( JtTimer pt = new JtTimer( 
        "Anonymous named" ) )
      {
        Element level
          = GetFirstNamedElementOfTypeUsingAnonymousButNamedMethod(
            t, name );
      }
      using( JtTimer pt = new JtTimer( "Anonymous" ) )
      {
        Element level
          = GetFirstNamedElementOfTypeUsingAnonymousMethod( 
            t, name );
      }
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication app = commandData.Application;
      _doc = app.ActiveUIDocument.Document;

      // create a number of levels for us to play with:

      int maxLevel = 1000;
      for( int i = 3; i < maxLevel; ++i )
      {
        CreateLevel( i );
      }

      // run a specified number of tests 
      // to retrieve all levels in different 
      // ways:

      int nLevels = GetElementsOfType( typeof( Level ) )
        .ToElements().Count;

      int nRuns = 1000;

      JtTimer totalTimer = new JtTimer(
        "TOTAL TIME" );

      using( totalTimer )
      {
        for( int i = 0; i < nRuns; ++i )
        {
          BenchmarkAllLevels( nLevels );
        }
      }

      totalTimer.Report( "Retrieve all levels:" );

      // run a specified number of tests 
      // to retrieve a randomly selected 
      // specific level:

      nRuns = 1000;
      Random rand = new Random();
      totalTimer.Restart( "TOTAL TIME" );

      using( totalTimer )
      {
        for( int i = 0; i < nRuns; ++i )
        {
          int iLevel = rand.Next( 1, maxLevel );
          BenchmarkSpecificLevel( iLevel );
        }
      }

      totalTimer.Report( 
        "Retrieve specific named level:" );

      return Result.Failed;
    }
  }
}
