#region Header
//
// CmdColumnRound.cs - determine whether a
// selected column instance is cylindrical
//
// Copyright (C) 2009-2010 by Jeremy Tammik,
// Autodesk Inc. All rights reserved.
//
#endregion // Header

#region Namespaces
using System;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.Structure;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
#endregion // Namespaces

namespace BuildingCoder
{
  [Transaction( TransactionMode.Automatic )]
  [Regeneration( RegenerationOption.Manual )]
  class CmdColumnRound : IExternalCommand
  {

#if REQUIRES_REVIT_2009_API
    //
    // works in Revit Structure 2009 API, but not in 2010:
    //
    bool IsColumnRound(
      FamilySymbol symbol )
    {
      GenericFormSet solid = symbol.Family.SolidForms;
      GenericFormSetIterator i = solid.ForwardIterator();
      i.MoveNext();
      Extrusion extr = i.Current as Extrusion;
      CurveArray cr = extr.Sketch.CurveLoop;
      CurveArrayIterator i2 = cr.ForwardIterator();
      i2.MoveNext();
      String s = i2.Current.GetType().ToString();
      return s.Contains( "Arc" );
    }
#endif // REQUIRES_REVIT_2009_API

#if REQUIRES_REVIT_2010_API
    //
    // works in Revit Structure 2010 API, but not in 2011:
    // works in Revit Structure, but not in other flavours of Revit:
    //
    bool ContainsArc( AnalyticalModel a )
    {
      bool rc = false;
      AnalyticalModel amp = a.Profile;
      Profile p = amp.SweptProfile;
      foreach( Curve c in p.Curves )
      {
        if( c is Arc )
        {
          rc = true;
          break;
        }
      }
      return rc;
    }
#endif // REQUIRES_REVIT_2010_API

    /// <summary>
    /// Return true if the given Revit element looks
    /// like it might be a column family instance.
    /// </summary>
    bool IsColumn( Element e )
    {
      return e is FamilyInstance
        && null != e.Category
        && e.Category.Name.ToLower().Contains( "column" );
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication app = commandData.Application;
      UIDocument uidoc = app.ActiveUIDocument;
      Document doc = uidoc.Document;

      Selection sel = uidoc.Selection;
      Element column = null;

      if( 1 == sel.Elements.Size )
      {
        foreach( Element e in sel.Elements )
        {
          column = e;
        }
        if( !IsColumn( column ) )
        {
          column = null;
        }
      }

      if( null == column )
      {

#if _2010
        sel.Elements.Clear();
        sel.StatusbarTip = "Please select a column";
        if( sel.PickOne() )
        {
          ElementSetIterator i
            = sel.Elements.ForwardIterator();
          i.MoveNext();
          column = i.Current as Element;
        }
#endif // _2010

        Reference r = uidoc.Selection.PickObject( ObjectType.Element,
          "Please select a column" );

        if( null != r )
        {
          column = r.Element;

          if( !IsColumn( column ) )
          {
            message = "Please select a single column instance";
          }
        }
      }

      if( null != column )
      {
        Options opt = app.Application.Create.NewGeometryOptions();
        GeometryElement geo = column.get_Geometry( opt );
        GeometryObjectArray objects = geo.Objects;
        GeometryInstance i = null;
        foreach( GeometryObject obj in objects )
        {
          i = obj as GeometryInstance;
          if( null != i )
          {
            break;
          }
        }
        if( null == i )
        {
          message = "Unable to obtain geometry instance";
        }
        else
        {
          bool isCylindrical = false;
          geo = i.SymbolGeometry;
          objects = geo.Objects;
          foreach( GeometryObject obj in objects )
          {
            Solid solid = obj as Solid;
            if( null != solid )
            {
              foreach( Face face in solid.Faces )
              {
                if( face is CylindricalFace )
                {
                  isCylindrical = true;
                  break;
                }
              }
            }
          }
          message = string.Format(
            "Selected column instance is{0} cylindrical",
            ( isCylindrical ? "" : " NOT" ) );
        }
      }
      return Result.Failed;
    }
  }
}
