﻿using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.Creation;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace RevitCommand
{
    [Transaction(TransactionMode.Manual)]
    [Regeneration(RegenerationOption.Manual)]
    public class RevitCommand : IExternalCommand
    {
        public Result Execute(ExternalCommandData commandData,ref string message, ElementSet elements)
        {
            using (Transaction trans = new Transaction(commandData.Application.ActiveUIDocument.Document,"testing")) 
            {
                trans.Start();

                FilteredElementCollector collector = new FilteredElementCollector(commandData.Application.ActiveUIDocument.Document).OfCategory(BuiltInCategory.OST_SpecialityEquipment).OfClass(typeof(FamilyInstance));
                List<Element> elementSet = collector.ToList();
                foreach(Element e in elementSet)
                {
                    TaskDialog td = new TaskDialog("filterDialog");
                    ElementIntersectsElementFilter filter = new ElementIntersectsElementFilter(e);
                    td.MainInstruction = "CURRENT ELEM: " + e.Id;
                    string outStr = "";
                    List<Solid> thisSolids = GetInstanceSolids(e);
                    Solid overall = null;
                    if(thisSolids.Where(s => s.Volume > 0).Count() > 1)
                    {
                        foreach (Solid s in thisSolids)
                        {
                            overall = overall == null ? s : BooleanOperationsUtils.ExecuteBooleanOperation(overall, s, BooleanOperationsType.Union);
                        }
                    }
                    else
                    {
                        overall = thisSolids.Where(s => s.Volume > 0).First();
                    }

                    foreach (Element other in elementSet)
                    {
                        if (!e.Equals(other))
                        {
                            if (filter.PassesFilter(other))
                                outStr += "ElementIntersectsElement: " + other.Id + System.Environment.NewLine;
                            List<Solid> otherSolids = GetInstanceSolids(other);
                            bool EIS_passed = false;
                            bool boolean_passed = false;
                            foreach (Solid s in otherSolids)
                            {
                                ElementIntersectsSolidFilter EiSFilter = new ElementIntersectsSolidFilter(s);
                                if (EiSFilter.PassesFilter(e) && !EIS_passed)
                                {
                                    foreach (Solid tS in thisSolids)
                                    {
                                        ElementIntersectsSolidFilter EiSFilter_this = new ElementIntersectsSolidFilter(tS);
                                        if (EiSFilter_this.PassesFilter(other))
                                        {
                                            outStr += "ElementIntersectsSolid: " + other.Id + System.Environment.NewLine;
                                            EIS_passed = true;
                                            break;
                                        }
                                    }
                                }
                                if ((!boolean_passed) && BooleanOperationsUtils.ExecuteBooleanOperation(overall, s, BooleanOperationsType.Intersect).Volume > 0)
                                {
                                    outStr += "Boolean Intersection: " + other.Id + System.Environment.NewLine;
                                    boolean_passed = true;
                                }                                    
                            }
                        }                        
                    }
                    td.ExpandedContent = outStr;
                    commandData.Application.ActiveUIDocument.Selection.SetElementIds(new List<ElementId>() { e.Id });
                    td.Show();
                }
                trans.Commit();

                return Result.Succeeded;
            }
            
        }

        public static List<Solid> GetInstanceSolids(Element elem)
        {
            List<Solid> solidsFound = new List<Solid>();
            Options options = new Options()
            {
                ComputeReferences = false,
                DetailLevel = ViewDetailLevel.Fine,
                IncludeNonVisibleObjects = false
            };


            Solid solid = null;
            // get the solid geometry from the element
            GeometryElement baseGeomElem = elem.get_Geometry(options);
            try
            {
                foreach (GeometryObject geomObj in baseGeomElem)
                {
                    solid = geomObj as Solid;
                    if (solid != null && solid.Faces.Size != 0 && solid.Edges.Size != 0)
                    {
                        solidsFound.Add(solid);
                    }
                }

                // get the symbol geometry if the element has no instance geometry
                if (solid == null)
                {
                    foreach (GeometryInstance instance in baseGeomElem.OfType<GeometryInstance>())
                    {
                        foreach (GeometryObject instObj in instance.GetInstanceGeometry())
                        {
                            solid = instObj as Solid;
                            if (solid != null && solid.Faces.Size != 0 && solid.Edges.Size != 0)
                            {
                                solidsFound.Add(solid);
                            }

                        }
                    }
                }
            }
            catch (NullReferenceException)
            {
                return solidsFound;
            }

            return solidsFound;
        }
    }
}
