﻿#Region "Imports"

Imports System
Imports System.Collections.Generic
Imports System.Linq
Imports System.Windows
Imports XML = System.Xml

Imports RVT = Autodesk.Revit
Imports RUI = Autodesk.Revit.UI
Imports RDB = Autodesk.Revit.DB
Imports RDV = Autodesk.Revit.DB.Visual
Imports RES = Autodesk.Revit.DB.ExtensibleStorage

#End Region

'repeat atributes and set as required for each class in the assembly
<RVT.Attributes.Transaction(RVT.Attributes.TransactionMode.Manual)>
<RVT.Attributes.Regeneration(RVT.Attributes.RegenerationOption.Manual)>
<RVT.Attributes.Journaling(RVT.Attributes.JournalingMode.NoCommandData)>
Public Class MaterialTools
	Implements RUI.IExternalCommand

	Private thisPath As String

	Private thisXMLDocument As New XML.XmlDocument()
	Private rootXMLNode As XML.XmlElement
	Private currentXMLNode As XML.XmlElement

	'launching function
	Public Function Execute(commandData As RUI.ExternalCommandData, ByRef message As String, elements As RDB.ElementSet) As RUI.Result _
	Implements RUI.IExternalCommand.Execute

		' Get the application, and document.
		''create these in Visual Studio Execute Method to pass along to the Sub and/or Function calls
		'modify them as needed when importing into a macro
		Dim thisUIApplication As RUI.UIApplication = commandData.Application
		Dim thisApplication As RVT.ApplicationServices.Application = commandData.Application.Application
		Dim thisUIDocument As RUI.UIDocument = commandData.Application.ActiveUIDocument
		Dim thisDocument As RDB.Document = commandData.Application.ActiveUIDocument.Document
		Dim thisDocumentView As RDB.View = commandData.View

		'set a working path for output and test image path
		'comment the following for release
		'thisPath = "E:\Resource\Misc Utilities\_RandD Resource files\Sample Package"

		'uncomment the following for release
		Dim folderPicker As New Windows.Forms.FolderBrowserDialog
		folderPicker.Description = "Select the desired working folder for log files and bitmap files"
		'folderPicker.ShowNewFolderButton = True
		Dim folderResult As Windows.Forms.DialogResult = folderPicker.ShowDialog()
		If folderResult <> Forms.DialogResult.Cancel AndAlso
			folderPicker.SelectedPath <> "" Then
			thisPath = folderPicker.SelectedPath
		Else
			Return RUI.Result.Cancelled
			Exit Function
		End If

		' Creates a Revit task dialog to get user preferences.
		Dim mainDialog As New RUI.TaskDialog("Command Options")
		mainDialog.MainInstruction = "Select the function from the options below that you would like to run:"
		mainDialog.MainContent = "Select Ok to run all of them, or Cancel to Exit" & vbCrLf &
									vbCrLf & "Log Files will be generated in *.txt and *.xml formats"

		' Add commmandLink options to task dialog
		mainDialog.AddCommandLink(RUI.TaskDialogCommandLinkId.CommandLink1, "Get All Revit Library Material Assets",
								  "This could take a minute or two on the first run in a session to expand the library")
		mainDialog.AddCommandLink(RUI.TaskDialogCommandLinkId.CommandLink2, "Get All Document Material Assets",
								  "Another task dialog will follow for additional options")
		mainDialog.AddCommandLink(RUI.TaskDialogCommandLinkId.CommandLink3, "Get All Document Materials",
								  "Another task dialog will follow for additional options")
		mainDialog.AddCommandLink(RUI.TaskDialogCommandLinkId.CommandLink4, "Create Material Sample",
								  "Creates a material with new fill patterns, an Appearance Asset with bitmaps, a Structural Asset, and a Thermal Asset")

		' Set common buttons and default button.
		Dim optButtons As RUI.TaskDialogCommonButtons = RUI.TaskDialogCommonButtons.Ok _
														Or RUI.TaskDialogCommonButtons.Cancel
		mainDialog.CommonButtons = optButtons
		mainDialog.DefaultButton = RUI.TaskDialogResult.Cancel

		Dim diaResult As RUI.TaskDialogResult = mainDialog.Show()

		Select Case diaResult
			Case RUI.TaskDialogResult.CommandLink1
				'run 1
				GetRevitAppearanceAssets(thisDocument, thisApplication)
			Case RUI.TaskDialogResult.CommandLink2
				'run 2
				GetDocMaterialAssets(thisDocument)
			Case RUI.TaskDialogResult.CommandLink3
				'run 3
				GetDocMaterials(thisDocument)
			Case RUI.TaskDialogResult.CommandLink4
				'run 4
				CreateSimpleMaterial(thisDocument)
			Case RUI.TaskDialogResult.Ok
				'run all
				GetRevitAppearanceAssets(thisDocument, thisApplication)
				GetDocMaterialAssets(thisDocument)
				GetDocMaterials(thisDocument)
				CreateSimpleMaterial(thisDocument)
			Case RUI.TaskDialogResult.Cancel
				'place tempory test functions here for testing
				'GetLibraryPaths(thisApplication)
				'uncomment the following for release
				Return RUI.Result.Cancelled
				Exit Function
		End Select

		Return RUI.Result.Succeeded
	End Function

	'list all Revit Library Material Appearance Assets
	'can't get Physical (Structural) or Thermal Property Sets this way due to current API limitations
	Public Sub GetRevitAppearanceAssets(thisDoc As RDB.Document, thisApp As RVT.ApplicationServices.Application)

		'start a log file:
		Dim logFile As IO.FileInfo = New IO.FileInfo(My.Computer.FileSystem.CombinePath(thisPath, "Revit Material Assets.txt"))
		Dim writer As System.IO.StreamWriter = logFile.CreateText()
		writer.WriteLine()
		writer.WriteLine("All Revit Material Appearance Assets" & vbCrLf)

		Dim xmlFile As String = My.Computer.FileSystem.CombinePath(thisPath, "Revit Material Assets.xml")
		thisXMLDocument.LoadXml("<?xml version=""1.0"" encoding=""utf-8""?>" & ControlChars.NewLine &
								"<Assets> </Assets>")
		'do not preserve whitespace so the xmldocument class will create indents and the like instead of you having to do it
		thisXMLDocument.PreserveWhitespace = False

		'Why are these material Assets accessed through the Application?
		'Because they're not the document assets, they're from a built-in Autodesk Asset Library
		'and where does this library exist anyway? Library name is the same for every appearance asset returned
		'regardless of them coming from the app level or the document level and it doesn't match any file name in the system
		Dim appAssets As IList(Of RDV.Asset) = thisApp.GetAssets(RDV.AssetType.Appearance)
		writer.WriteLine(" Number of Library Assets:  " & appAssets.Count.ToString & vbCrLf)

		'Get our rootNode
		rootXMLNode = thisXMLDocument.DocumentElement
		rootXMLNode.SetAttribute("Source", "Revit Appearance Asset Library")
		rootXMLNode.SetAttribute("Count", appAssets.Count.ToString)


		' Creates a Revit task dialog to communicate information to the user.
		Dim mainDialog As New RUI.TaskDialog("Command Options")
		mainDialog.MainInstruction = "? Add to Document ?"
		mainDialog.MainContent = "Select Yes to Create One Asset" & vbCrLf &
								"Of each Schema in the current document"
		Dim optButtons As RUI.TaskDialogCommonButtons = RUI.TaskDialogCommonButtons.Yes _
														Or RUI.TaskDialogCommonButtons.No
		mainDialog.CommonButtons = optButtons
		mainDialog.DefaultButton = RUI.TaskDialogResult.No
		Dim diaResult As RUI.TaskDialogResult = mainDialog.Show()

		'create a dictionary that contains keys from the base schemas
		'and values containing dictionaries whose keys are the AssetName and whose values are the assets themselves
		Dim dictAssetType As New SortedDictionary(Of String, SortedDictionary(Of String, RDV.Asset))

		For Each thisAsset As RDV.Asset In appAssets
			Dim schemaType As RDV.AssetProperty = thisAsset.FindByName(RDV.SchemaCommon.BaseSchema)
			Dim thisProp As RDV.AssetPropertyString = TryCast(schemaType, RDV.AssetPropertyString)
			Dim dictAsset As SortedDictionary(Of String, RDV.Asset)

			'if this schema type doesn't exist in the dictionary then create it and add it
			If dictAssetType.ContainsKey(thisProp.Value) = False Then
				dictAssetType.Add(thisProp.Value, New SortedDictionary(Of String, RDV.Asset))
			End If

			dictAsset = dictAssetType.Item(thisProp.Value)
			dictAsset.Add(thisAsset.Name, thisAsset)

		Next

		Dim listAsset As New List(Of String)
		Dim thisTrans As New RDB.Transaction(thisDoc, "Create Assets")
		Dim listNewAsset As New List(Of String)

		If diaResult = RUI.TaskDialogResult.Yes Then
			thisTrans.Start()
			Dim assetCollector As New RDB.FilteredElementCollector(thisDoc)
			Dim allAppearanceAssets As IList(Of RDB.AppearanceAssetElement) = assetCollector.OfClass(GetType(RDB.AppearanceAssetElement)).OfType(Of RDB.AppearanceAssetElement).ToList

			For Each thisAssetElem As RDB.AppearanceAssetElement In allAppearanceAssets
				If listAsset.Contains(thisAssetElem.Name) = False Then
					listAsset.Add(thisAssetElem.Name)
				End If
			Next
		End If

		For Each thisSchema As KeyValuePair(Of String, SortedDictionary(Of String, RDV.Asset)) In dictAssetType
			writer.WriteLine(vbTab & "Asset Schema Type: " & thisSchema.Key & vbTab & "| Count: " & thisSchema.Value.Count.ToString)
			Dim schemaNode As XML.XmlElement = thisXMLDocument.CreateElement("Schema")
			schemaNode.SetAttribute("Type", thisSchema.Key)
			schemaNode.SetAttribute("Count", thisSchema.Value.Count.ToString)
			rootXMLNode.AppendChild(schemaNode)

			Dim runOnce As Boolean = False

			For Each thisAssetKVP As KeyValuePair(Of String, RDV.Asset) In thisSchema.Value
				Dim thisAsset As RDV.Asset = thisAssetKVP.Value
				writer.WriteLine(vbCrLf & vbTab & vbTab & "Asset Name: " & thisAssetKVP.Key &
								 "  |  Size: " & thisAsset.Size.ToString)

				Dim thisXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("Asset")
				thisXmlNode.SetAttribute("Name", thisAssetKVP.Key)
				thisXmlNode.SetAttribute("Size", thisAsset.Size.ToString)
				schemaNode.AppendChild(thisXmlNode)
				'use asset.size to ensure that we have something to process
				If thisAsset.Size > 0 Then
					GetAssetProperties(thisAsset, thisXmlNode, writer, 2)
				End If
				If diaResult = RUI.TaskDialogResult.Yes AndAlso runOnce = False Then
					Dim nameStr As String = ""
					'create a new Document Appearance Asset from this Asset
					'error checking for existence of name
					nameStr = GenerateName("_" & thisSchema.Key, listAsset)
					Dim newAsset As RDB.AppearanceAssetElement = RDB.AppearanceAssetElement.Create(thisDoc, nameStr, thisAsset)
					listAsset.Add(nameStr)
					listNewAsset.Add(nameStr)
					runOnce = True
				End If
			Next

		Next

		If thisTrans.HasStarted = True Then
			thisTrans.Commit()
		End If

		'question: why can't we get Physical (Structural) and Thermal library assets using similar methods?

		'finalize the log files

		thisXMLDocument.Save(xmlFile)

		writer.Flush()
		writer.Close()
		'System.Diagnostics.Process.Start("notepad.exe", logFile.FullName)

		If listNewAsset.Count > 0 Then
			Dim returnStr As String = "Added the following to the document"
			For Each name As String In listNewAsset
				returnStr += vbCrLf & name
			Next
			RUI.TaskDialog.Show("Notification", returnStr)
		End If

	End Sub

	'list all document material assets: Appearance, Physical (Structural) PSE, and Thermal PSE
	'still can't firmly determine difference between Structural and Thermal Property Set Elements
	'without getting a material first, then pulling them specifically from that
	Public Sub GetDocMaterialAssets(thisDoc As RDB.Document)

		' Creates a Revit task dialog to communicate information to the user.
		Dim mainDialog As New RUI.TaskDialog("Command Options")
		mainDialog.MainInstruction = "? Include Revit Element Parameters ?"
		mainDialog.MainContent = "Select Yes to include Element Parameters" & vbCrLf &
									"Select No to skip Element Parameters"
		Dim optButtons As RUI.TaskDialogCommonButtons = RUI.TaskDialogCommonButtons.Yes _
														Or RUI.TaskDialogCommonButtons.No
		mainDialog.CommonButtons = optButtons
		mainDialog.DefaultButton = RUI.TaskDialogResult.No
		Dim diaResult As RUI.TaskDialogResult = mainDialog.Show()

		'start a log file:
		Dim docName As String = thisDoc.Title
		If docName = "" Then
			docName = "Doc"
		End If
		Dim logFile As IO.FileInfo = New IO.FileInfo(My.Computer.FileSystem.CombinePath(thisPath, docName & " Material Assets.txt"))
		Dim writer As System.IO.StreamWriter = logFile.CreateText()

		writer.WriteLine()
		writer.WriteLine("All Document Material Assets" & vbCrLf)

		Dim xmlFile As String = My.Computer.FileSystem.CombinePath(thisPath, docName & " Material Assets.xml")
		thisXMLDocument.LoadXml("<?xml version=""1.0"" encoding=""utf-8""?>" & ControlChars.NewLine &
								"<Assets> </Assets>")
		'do not preserve whitespace so the xmldocument class will create indents and the like instead of you having to do it
		thisXMLDocument.PreserveWhitespace = False
		'Get our rootNode
		rootXMLNode = thisXMLDocument.DocumentElement
		rootXMLNode.SetAttribute("Source", docName)

		'must inherit from Element for filtered element collector so we need the document element level,
		'then use that to get the underlying Asset
		Dim assetCollector As New RDB.FilteredElementCollector(thisDoc)
		Dim allAppearanceAssets As IList(Of RDB.AppearanceAssetElement) = assetCollector.OfClass(GetType(RDB.AppearanceAssetElement)).OfType(Of RDB.AppearanceAssetElement).ToList

		writer.WriteLine()
		writer.WriteLine("**Appearance Assets:  " & allAppearanceAssets.Count.ToString)

		currentXMLNode = thisXMLDocument.CreateElement("Category")
		currentXMLNode.SetAttribute("Name", "Appearance Assets")
		currentXMLNode.SetAttribute("Count", allAppearanceAssets.Count.ToString)
		rootXMLNode.AppendChild(currentXMLNode)

		Dim dictAssetType As New SortedDictionary(Of String, SortedDictionary(Of String, RDB.AppearanceAssetElement))

		For Each thisAssetElem As RDB.AppearanceAssetElement In allAppearanceAssets
			Dim thisAsset As RDV.Asset = thisAssetElem.GetRenderingAsset
			Dim schemaStr As String = "UnKnown"
			Dim schemaType As RDV.AssetProperty = thisAsset.FindByName(RDV.SchemaCommon.BaseSchema)
			Dim thisProp As RDV.AssetPropertyString = TryCast(schemaType, RDV.AssetPropertyString)
			If thisProp IsNot Nothing Then
				schemaStr = thisProp.Value
			End If
			Dim dictAsset As SortedDictionary(Of String, RDB.AppearanceAssetElement)

			'if this schema type doesn't exist in the dictionary then create it and add it
			If dictAssetType.ContainsKey(schemaStr) = False Then
				dictAssetType.Add(schemaStr, New SortedDictionary(Of String, RDB.AppearanceAssetElement))
			End If

			dictAsset = dictAssetType.Item(schemaStr)
			If dictAsset.ContainsKey(thisAssetElem.Name) = False Then
				dictAsset.Add(thisAssetElem.Name, thisAssetElem)
			End If
		Next

		For Each thisSchema As KeyValuePair(Of String, SortedDictionary(Of String, RDB.AppearanceAssetElement)) In dictAssetType

			writer.WriteLine(vbTab & "Asset Schema Type: " & thisSchema.Key & vbTab & "  | Count: " & thisSchema.Value.Count.ToString)

			Dim schemaNode As XML.XmlElement = thisXMLDocument.CreateElement("Schema")
			schemaNode.SetAttribute("Type", thisSchema.Key)
			schemaNode.SetAttribute("Count", thisSchema.Value.Count.ToString)
			currentXMLNode.AppendChild(schemaNode)

			For Each thisAssetKVP As KeyValuePair(Of String, RDB.AppearanceAssetElement) In thisSchema.Value
				Dim thisAssetElem As RDB.AppearanceAssetElement = thisAssetKVP.Value

				writer.WriteLine(vbTab & vbTab & "Asset Name: " & thisAssetKVP.Key & "  | UniqueId: " & thisAssetElem.UniqueId)

				Dim elemXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("Element")
				elemXmlNode.SetAttribute("Name", thisAssetKVP.Key)
				elemXmlNode.SetAttribute("UniqueId", thisAssetElem.UniqueId)
				schemaNode.AppendChild(elemXmlNode)

				If thisAssetElem.Parameters.Size > 0 Then
					writer.WriteLine(vbTab & vbTab & vbTab & " -- Pset Parameters:  " & thisAssetElem.Parameters.Size.ToString)

					Dim paramXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("Parameters")
					paramXmlNode.SetAttribute("Count", thisAssetElem.Parameters.Size.ToString)
					elemXmlNode.AppendChild(paramXmlNode)

					If diaResult = RUI.TaskDialogResult.Yes Then
						For Each para As RDB.Parameter In thisAssetElem.Parameters
							Try
								LogParam(para, thisDoc, paramXmlNode, writer, 4)
							Catch ex As Exception
								writer.WriteLine(vbTab & vbTab & vbTab & " !! Cannot retrieve Parameter: " & para.Id.ToString)
							End Try
						Next
						writer.WriteLine(vbTab & vbTab & vbTab & " -- End Parameters" & vbCrLf)
					End If
				End If

				Dim thisAsset As RDV.Asset = thisAssetElem.GetRenderingAsset

				Dim thisXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("Asset")
				thisXmlNode.SetAttribute("Name", thisAsset.Name)
				thisXmlNode.SetAttribute("Size", thisAsset.Size.ToString)
				elemXmlNode.AppendChild(thisXmlNode)

				'use asset.size to ensure that we have something to process
				If thisAsset.Size > 0 Then
					GetAssetProperties(thisAsset, thisXmlNode, writer, 3)
				End If
				writer.WriteLine()

			Next
			writer.WriteLine()
		Next

		Dim pSetCollector As New RDB.FilteredElementCollector(thisDoc)
		Dim allPropertyAssets As IList(Of RDB.PropertySetElement) = pSetCollector.OfClass(GetType(RDB.PropertySetElement)).OfType(Of RDB.PropertySetElement).ToList

		writer.WriteLine()
		writer.WriteLine("**Property Set Assets:  " & allPropertyAssets.Count.ToString)

		currentXMLNode = thisXMLDocument.CreateElement("Category")
		currentXMLNode.SetAttribute("Name", "Physical Assets")
		currentXMLNode.SetAttribute("Count", allPropertyAssets.Count.ToString)
		rootXMLNode.AppendChild(currentXMLNode)

		'try to categorize the returns for selection by type (stru or Thermal) when duplicating an asset
		'although this would double the processing time

		Dim dictPsetType As New SortedDictionary(Of String, SortedDictionary(Of String, RDB.PropertySetElement))
		Dim dictTherm As New SortedDictionary(Of String, RDB.PropertySetElement)
		Dim dictStruct As New SortedDictionary(Of String, RDB.PropertySetElement)
		Dim dictUnKnown As New SortedDictionary(Of String, RDB.PropertySetElement)

		For Each thisAssetElem As RDB.PropertySetElement In allPropertyAssets
			'place tests here and place Assets in appropriate list
			Dim thermAsset As RDB.ThermalAsset = Nothing
			Dim strucAsset As RDB.StructuralAsset = Nothing
			Dim tryOther As Boolean = False

			Dim testParam As RDB.Parameter = thisAssetElem.Parameter(RDB.BuiltInParameter.PROPERTY_SET_KEYWORDS)
			If testParam IsNot Nothing Then
				'we have to test it
				If LCase(testParam.AsString) Like LCase("*structural*") Then
					strucAsset = thisAssetElem.GetStructuralAsset
				ElseIf LCase(testParam.AsString) Like LCase("*thermal*") Then
					thermAsset = thisAssetElem.GetThermalAsset
				Else
					'it's an unknown PSE class that we probably shouldn't collect for reuse
					'(but we may need to know it's name to avoid any naming conflicts against either type)
					tryOther = True
				End If
			Else
				'it may be an older material and we think it's structural because
				'older structural Psets didn't seem to have it, but we don't know for sure
				'either way we shouldn't be using it to duplicate, although we may want replace it???
				'(but we may need to know it's name to avoid any naming conflicts)
				tryOther = True
			End If

			If tryOther = True Then
				'use a Try... Catch... block to see if it can be
				'turned into a thermal asset that way, and if not it will most likely cast to structural
				Try
					thermAsset = thisAssetElem.GetThermalAsset
				Catch exT As Exception
					Try
						strucAsset = thisAssetElem.GetStructuralAsset
					Catch exS As Exception
						'do nothing
					End Try
				End Try
			End If

			If thermAsset IsNot Nothing Then
				If dictTherm.ContainsKey(thermAsset.Name) = False Then
					dictTherm.Add(thermAsset.Name, thisAssetElem)
				End If
			ElseIf strucAsset IsNot Nothing Then
				If dictStruct.ContainsKey(strucAsset.Name) = False Then
					dictStruct.Add(strucAsset.Name, thisAssetElem)
				End If
			Else
				'should not need this since all Pset elements seen to be able to convert to structural
				If dictUnKnown.ContainsKey(thisAssetElem.Name) = False Then
					dictUnKnown.Add(thisAssetElem.Name, thisAssetElem)
				End If
			End If

		Next

		dictPsetType.Add("Structural", dictStruct)
		dictPsetType.Add("Thermal", dictTherm)
		dictPsetType.Add("Unknown", dictUnKnown)

		For Each thisSchema As KeyValuePair(Of String, SortedDictionary(Of String, RDB.PropertySetElement)) In dictPsetType

			If thisSchema.Value.Count > 0 Then
				writer.WriteLine(vbTab & "Asset Type: " & thisSchema.Key & vbTab & "| Count: " & thisSchema.Value.Count.ToString)

				Dim schemaNode As XML.XmlElement = thisXMLDocument.CreateElement("Schema")
				schemaNode.SetAttribute("Type", thisSchema.Key)
				schemaNode.SetAttribute("Count", thisSchema.Value.Count.ToString)
				currentXMLNode.AppendChild(schemaNode)

				For Each thisAssetKVP As KeyValuePair(Of String, RDB.PropertySetElement) In thisSchema.Value
					writer.WriteLine(vbTab & vbTab & "Asset Name: " & thisAssetKVP.Key)

					Dim thisAssetElem As RDB.PropertySetElement = thisAssetKVP.Value

					Dim elemXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("Element")
					elemXmlNode.SetAttribute("Name", thisAssetKVP.Key)
					elemXmlNode.SetAttribute("UniqueId", thisAssetElem.UniqueId)
					schemaNode.AppendChild(elemXmlNode)

					If thisAssetElem.Parameters.Size > 0 Then
						writer.WriteLine(vbTab & vbTab & vbTab & " -- Pset Parameters:  " & thisAssetElem.Parameters.Size.ToString)

						Dim paramXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("Parameters")
						paramXmlNode.SetAttribute("Count", thisAssetElem.Parameters.Size.ToString)
						elemXmlNode.AppendChild(paramXmlNode)

						If diaResult = RUI.TaskDialogResult.Yes Then
							For Each para As RDB.Parameter In thisAssetElem.Parameters
								Try
									LogParam(para, thisDoc, paramXmlNode, writer, 4)
								Catch ex As Exception
									writer.WriteLine(vbTab & vbTab & vbTab & vbTab & " !! Cannot retrieve Parameter: " & para.Id.ToString)
								End Try
							Next
							writer.WriteLine(vbTab & vbTab & vbTab & " -- End Parameters" & vbCrLf)
						End If
					End If

					If UCase(thisSchema.Key) Like UCase("Structural") Then
						Dim strucAsset As RDB.StructuralAsset = thisAssetElem.GetStructuralAsset
						GetStructAsset(strucAsset, elemXmlNode, writer, 3)
					ElseIf UCase(thisSchema.Key) Like UCase("Thermal") Then
						Dim thermAsset As RDB.ThermalAsset = thisAssetElem.GetThermalAsset
						GetThermAsset(thermAsset, elemXmlNode, writer, 3)
					Else

					End If
				Next
				writer.WriteLine()
			End If

		Next


		'finalize the log files

		thisXMLDocument.Save(xmlFile)

		writer.Flush()
		writer.Close()
		'System.Diagnostics.Process.Start("notepad.exe", logFile.FullName)

	End Sub

	'list all document materials and their connected assets.
	Public Sub GetDocMaterials(thisDoc As RDB.Document)

		' Creates a Revit task dialog to communicate information to the user.
		Dim mainDialog As New RUI.TaskDialog("Command Options")
		mainDialog.MainInstruction = "? Include Revit Element Parameters ?"
		mainDialog.MainContent = "Select Yes to include Element Parameters" & vbCrLf &
									"Select No to skip Element Parameters"
		Dim optButtons As RUI.TaskDialogCommonButtons = RUI.TaskDialogCommonButtons.Yes _
														Or RUI.TaskDialogCommonButtons.No
		mainDialog.CommonButtons = optButtons
		mainDialog.DefaultButton = RUI.TaskDialogResult.No
		Dim diaResult As RUI.TaskDialogResult = mainDialog.Show()

		'start a log file:
		Dim docName As String = thisDoc.Title
		If docName = "" Then
			docName = "Doc"
		End If
		Dim logFile As IO.FileInfo = New IO.FileInfo(My.Computer.FileSystem.CombinePath(thisPath, docName & " Materials.txt"))
		Dim writer As System.IO.StreamWriter = logFile.CreateText()

		Dim matCollector As RDB.FilteredElementCollector = New RDB.FilteredElementCollector(thisDoc)
		Dim allMats As IList(Of RDB.Material) = matCollector.OfClass(GetType(RDB.Material)).OfType(Of RDB.Material).ToList

		writer.WriteLine()
		writer.WriteLine("All Document Materials:  " & allMats.Count.ToString & vbCrLf)

		Dim xmlFile As String = My.Computer.FileSystem.CombinePath(thisPath, docName & " Materials.xml")
		thisXMLDocument.LoadXml("<?xml version=""1.0"" encoding=""utf-8""?>" & ControlChars.NewLine &
								"<Materials> </Materials>")
		'do not preserve whitespace so the xmldocument class will create indents and the like instead of you having to do it
		thisXMLDocument.PreserveWhitespace = False
		'Get our rootNode
		rootXMLNode = thisXMLDocument.DocumentElement
		rootXMLNode.SetAttribute("Source", docName)
		rootXMLNode.SetAttribute("Count", allMats.Count.ToString)


		Dim dictMatType As New SortedDictionary(Of String, SortedDictionary(Of String, RDB.Material))
		For Each thisMat As RDB.Material In allMats
			'if this schema type doesn't exist in the dictionary then create it and add it
			Dim className As String = ""
			If thisMat.MaterialClass <> "" Then
				className = thisMat.MaterialClass
			Else
				className = "Unknown"
			End If
			If dictMatType.ContainsKey(className) = False Then
				dictMatType.Add(className, New SortedDictionary(Of String, RDB.Material))
			End If

			Dim dictMat As SortedDictionary(Of String, RDB.Material)
			dictMat = dictMatType.Item(className)

			If dictMat.ContainsKey(thisMat.Name) = False Then
				dictMat.Add(thisMat.Name, thisMat)
			End If
		Next


		For Each thisClass As KeyValuePair(Of String, SortedDictionary(Of String, RDB.Material)) In dictMatType
			writer.WriteLine(vbTab & "Material Class: " & thisClass.Key & vbTab & "  | Count: " & thisClass.Value.Count.ToString)

			Dim classNode As XML.XmlElement = thisXMLDocument.CreateElement("Class")
			classNode.SetAttribute("Type", thisClass.Key)
			classNode.SetAttribute("Count", thisClass.Value.Count.ToString)
			rootXMLNode.AppendChild(classNode)

			For Each thisMatKVP As KeyValuePair(Of String, RDB.Material) In thisClass.Value
				writer.WriteLine(vbTab & vbTab & "Material Name:  " & thisMatKVP.Key)

				Dim elemXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("Material")
				elemXmlNode.SetAttribute("Name", thisMatKVP.Key)
				classNode.AppendChild(elemXmlNode)

				Dim thisMaterial As RDB.Material = thisMatKVP.Value

				'get the info here
				If diaResult = RUI.TaskDialogResult.Yes Then
					GetMaterialInfo(thisDoc, thisMaterial, elemXmlNode, writer, 3, True)
				Else
					GetMaterialInfo(thisDoc, thisMaterial, elemXmlNode, writer, 3, False)
				End If

				writer.WriteLine()
			Next
			writer.WriteLine()
		Next


		'finalize the log files

		thisXMLDocument.Save(xmlFile)

		writer.Flush()
		writer.Close()
		'System.Diagnostics.Process.Start("notepad.exe", logFile.FullName)

	End Sub

	'create a material using what we've learned
	Public Sub CreateSimpleMaterial(thisDoc As RDB.Document)

		'collect our resources to test for naming conflicts and for reuse
#Region "Collect resources"

		Dim matCollector As RDB.FilteredElementCollector = New RDB.FilteredElementCollector(thisDoc)
		Dim allMats As IList(Of RDB.Material) = matCollector.OfClass(GetType(RDB.Material)).OfType(Of RDB.Material).ToList

		Dim dictMat As New SortedDictionary(Of String, RDB.Material)
		For Each thisMat As RDB.Material In allMats
			If dictMat.ContainsKey(thisMat.Name) = False Then
				dictMat.Add(thisMat.Name, thisMat)
			End If
		Next

		Dim assetCollector As New RDB.FilteredElementCollector(thisDoc)
		Dim allAppearanceAssets As IList(Of RDB.AppearanceAssetElement) = assetCollector.OfClass(GetType(RDB.AppearanceAssetElement)).OfType(Of RDB.AppearanceAssetElement).ToList

		Dim dictAssetElem As New SortedDictionary(Of String, RDB.AppearanceAssetElement)
		Dim listAsset As New List(Of RDV.Asset)

		For Each thisAssetElem As RDB.AppearanceAssetElement In allAppearanceAssets
			If dictAssetElem.ContainsKey(thisAssetElem.Name) = False Then
				dictAssetElem.Add(thisAssetElem.Name, thisAssetElem)
			End If
			Try
				Dim thisAsset As RDV.Asset = thisAssetElem.GetRenderingAsset
				listAsset.Add(thisAsset)
			Catch ex As Exception
				'do nothing, just keep going and add the next, we don't need everything
			End Try
		Next

		Dim pSetCollector As New RDB.FilteredElementCollector(thisDoc)
		Dim allPropertyAssets As IList(Of RDB.PropertySetElement) = pSetCollector.OfClass(GetType(RDB.PropertySetElement)).OfType(Of RDB.PropertySetElement).ToList

		Dim dictTherm As New SortedDictionary(Of String, RDB.PropertySetElement)
		Dim dictStruct As New SortedDictionary(Of String, RDB.PropertySetElement)
		Dim dictUnKnown As New SortedDictionary(Of String, RDB.PropertySetElement)

		For Each thisAssetElem As RDB.PropertySetElement In allPropertyAssets
			'place tests here and place Assets in appropriate list
			Dim thermAsset As RDB.ThermalAsset = Nothing
			Dim strucAsset As RDB.StructuralAsset = Nothing
			Dim tryOther As Boolean = False

			Dim testParam As RDB.Parameter = thisAssetElem.Parameter(RDB.BuiltInParameter.PROPERTY_SET_KEYWORDS)
			If testParam IsNot Nothing Then
				'we have to test it
				If LCase(testParam.AsString) Like LCase("*structural*") Then
					strucAsset = thisAssetElem.GetStructuralAsset
				ElseIf LCase(testParam.AsString) Like LCase("*thermal*") Then
					thermAsset = thisAssetElem.GetThermalAsset
				Else
					'it has the parameter but doesn't have one of the required keywords
					'so we'll use the try... catch method testing for ability to return the thermal PSE
					tryOther = True
				End If
			Else
				'it may be an older material and we think it will be structural because older Physical only Psets
				'didn't seem to have this parameter and all seem to default to struct, although we don't know for sure
				'so we'll use the try... catch method testing for ability to return the thermal PSE
				tryOther = True
			End If

			If tryOther = True Then
				'use a Try... Catch... block to see if it can be
				'turned into a thermal PSE asset that way, and if not it will most likely cast to structural
				Try
					thermAsset = thisAssetElem.GetThermalAsset
				Catch exT As Exception
					Try
						strucAsset = thisAssetElem.GetStructuralAsset
					Catch exS As Exception
						'do nothing since the next block will wrap it
					End Try
				End Try
			End If

			If thermAsset IsNot Nothing Then
				If dictTherm.ContainsKey(thermAsset.Name) = False Then
					dictTherm.Add(thermAsset.Name, thisAssetElem)
				End If
			ElseIf strucAsset IsNot Nothing Then
				If dictStruct.ContainsKey(strucAsset.Name) = False Then
					dictStruct.Add(strucAsset.Name, thisAssetElem)
				End If
			Else
				'should not need this since all Pset elements seen to be able to convert to structural
				'but we keep it in just in case so we can double check for naming conflicts when creating our Psets
				If dictUnKnown.ContainsKey(thisAssetElem.Name) = False Then
					dictUnKnown.Add(thisAssetElem.Name, thisAssetElem)
				End If
			End If
		Next

		Dim patCollector As RDB.FilteredElementCollector = New RDB.FilteredElementCollector(thisDoc)
		Dim allPats As IList(Of RDB.FillPatternElement) = patCollector.OfClass(GetType(RDB.FillPatternElement)).OfType(Of RDB.FillPatternElement).ToList

		Dim dictPat As New SortedDictionary(Of String, RDB.FillPatternElement)
		For Each thisPat As RDB.FillPatternElement In allPats
			If dictPat.ContainsKey(thisPat.GetFillPattern.Name) = False Then
				dictPat.Add(thisPat.GetFillPattern.Name, thisPat)
			End If
		Next

#End Region


#Region "Schema Mapping for lookup"

		'Dim thisAsset As RDV.Asset = thisAssetElem.GetRenderingAsset
		'Dim schemaName As String
		'Dim schemaType As RDV.AssetProperty = thisAsset.FindByName(RDV.SchemaCommon.BaseSchema)
		'Dim thisProp As RDV.AssetPropertyString = TryCast(schemaType, RDV.AssetPropertyString)
		'If thisProp IsNot Nothing Then
		'	schemaName = thisProp.Value
		'End If

		'Try
		'	schemaName = TryCast(thisAsset.FindByName(RDV.SchemaCommon.BaseSchema), RDV.AssetPropertyString).Value
		'	'would this do the same?
		'	'schemaName = thisAsset.FindByName(RDV.SchemaCommon.BaseSchema).Value
		'Catch ex As Exception
		'	schemaName = "Unknown"
		'End Try

		'	'find values using:
		'Select Case schemaName
		'	Case "CeramicSchema"
		'			'RDV.Ceramic
		'	Case "ConcreteSchema"
		'			'RDV.Concrete
		'	Case "DecalAppearanceSchema"
		'			'??
		'	Case "DecalSchema"
		'			'??
		'	Case "GenericSchema"
		'			'RDV.Generic
		'	Case "GlazingSchema"
		'			'RDV.Glazing
		'	Case "HardwoodSchema"
		'			'RDV.Hardwood
		'	Case "MasonryCMUSchema"
		'			'RDV.MasonryCMU
		'	Case "MetallicPaintSchema"
		'			'RDV.MetallicPaint
		'	Case "MetalSchema"
		'			'RDV.Metal
		'	Case "MirrorSchema"
		'			'RDV.Mirror
		'	Case "PlasticVinylSchema"
		'			'RDV.PlasticVinyl
		'	Case "PrismGlazingSchema"
		'			'RDV.AdvancedGlazing
		'	Case "PrismLayeredSchema"
		'			'RDV.AdvancedLayered
		'	Case "PrismMetalSchema"
		'			'RDV.AdvancedMetal
		'	Case "PrismOpaqueSchema"
		'			'RDV.AdvancedOpaque
		'	Case "PrismTransparentSchema"
		'			'RDV.AdvancedTransparent
		'	Case "PrismWoodSchema"
		'			'RDV.AdvancedWood
		'	Case "SolidGlassSchema"
		'			'RDV.SolidGlass
		'	Case "StoneSchema"
		'			'RDV.Stone
		'	Case "TilingAppearanceSchema"
		'			'??
		'	Case "TilingPatternSchema"
		'			'??
		'	Case "WallPaintSchema"
		'			'RDV.WallPaint
		'	Case "WaterSchema"
		'		'RDV.Water
		'	Case Else 'this never happens just including for all API schema values
		'		'RDV.SchemaCommon
		'End Select

		'Mapping Types:
		'Checker
		'Gradient
		'Marble
		'Noise
		'Speckle
		'Tile
		'Wave
		'Wood
		'BumpMap
		'UnifiedBitmap

#End Region

		'now we're ready to create our new material in the document, so we start a transaction
		'but first we will define our container for the material
		'so we can do stuff with it after the transaction goes out of scope
		Dim thisMaterial As RDB.Material = Nothing
		Using thisTrans As New RDB.Transaction(thisDoc, "Create new material")
			thisTrans.Start()

			Dim nameStr As String = ""

			'Create the material
			'error checking for existence of name
			nameStr = GenerateName("_My Material", dictMat.Keys.ToList)
			'RUI.TaskDialog.Show("Material Name", nameStr)
			Dim materialId As RDB.ElementId = RDB.Material.Create(thisDoc, nameStr)
			thisMaterial = TryCast(thisDoc.GetElement(materialId), RDB.Material)
			'properties and parameters often overlap, some can be used to set the other,
			'some will be overridden by the Appearance Asset Properties.
			'some will fail if you don't set them in the correct scope.
			'should always consider checking for read only and/or can be set,
			'but we're skipping some of that for now

#Region "Set some properties for the Material"
			'set a parameter on the material
			Dim descriptionParameter As RDB.Parameter = thisMaterial.Parameter(RDB.BuiltInParameter.ALL_MODEL_DESCRIPTION)
			descriptionParameter.Set("My First Material")
			'set a couple properties on the material
			thisMaterial.MaterialClass = "My Classification"
			thisMaterial.UseRenderAppearanceForShading = False
			'neither of the following apply if the above line is True
			thisMaterial.Color = New RDB.Color(127, 127, 127)
			thisMaterial.Transparency = 0

			'set some hatch patterns 
			'create a Model hatch pattern for the surface foreground
			'error checking for existence of name
			nameStr = GenerateName("_My Model Pattern ", dictPat.Keys.ToList)
			Dim modelPatDef As RDB.FillPattern = New RDB.FillPattern(nameStr, RDB.FillPatternTarget.Model,
																  RDB.FillPatternHostOrientation.ToHost, 0, 2, 2)
			Dim modelPat As RDB.FillPatternElement = RDB.FillPatternElement.Create(thisDoc, modelPatDef)
			thisMaterial.SurfaceForegroundPatternId = modelPat.Id
			'create a drafting hatch pattern for the cut foreground
			'error checking for existence of name
			nameStr = GenerateName("_My Drafting Pattern", dictPat.Keys.ToList)
			Dim draftPatDef As RDB.FillPattern = New RDB.FillPattern(nameStr, RDB.FillPatternTarget.Drafting,
																  RDB.FillPatternHostOrientation.ToHost, 0, 1 / 12, 1 / 12)
			Dim draftPat As RDB.FillPatternElement = RDB.FillPatternElement.Create(thisDoc, draftPatDef)
			thisMaterial.CutForegroundPatternId = draftPat.Id

#End Region

			'create an Appearance Asset
#Region "Create an Appearance Asset"
			Dim genericAsset As RDV.Asset

			'we should be able to search for any of the schema types by matching SchemaCommon.BaseSchema
			'Property to a specific schema type (this would require some casting). but we can't, see next attempt comment
			'genericAsset = listAsset.FirstOrDefault(Function(eachAsset) eachAsset.FindByName(RDV.SchemaCommon.BaseSchema) = GetType(RDV.Generic).Name)

			'or from the asset itself as the asset.name = the schema
			'we should be able to use this but there is no one-to-one mapping between the Property Values and the API Classes
			'genericAsset = listAsset.FirstOrDefault(Function(eachAsset) eachAsset.Name() = GetType(RDV.Generic).Name)

			'this one is for the Generic type by checking for a Generic Schema specific property,
			'but this would be tedious across multiple schemas

			'try existing doc assets first for one to duplicate since we can't just create one from scratch

			genericAsset = listAsset.FirstOrDefault(Function(eachAsset) eachAsset.FindByName(RDV.Generic.GenericDiffuse) IsNot Nothing)
			'genericAsset = listAsset.FirstOrDefault(Function(eachAsset) eachAsset.FindByName("generic_diffuse") IsNot Nothing)

			'if no suitable doc asset found then we'll get a Revit Library Asset
			If genericAsset Is Nothing Then
				Dim assetList As List(Of RDV.Asset) = thisDoc.Application.GetAssets(RDV.AssetType.Appearance)
				genericAsset = assetList.FirstOrDefault(Function(eachAsset) eachAsset.FindByName(RDV.Generic.GenericDiffuse) IsNot Nothing)

				'	RUI.TaskDialog.Show("Using Library Asset", "Cannot get doc assets: " & genericAsset.Name &
				'						vbCrLf & "Type Name: " & GetType(RDV.Generic).Name)
				'Else
				'	RUI.TaskDialog.Show("Using Document Asset", "Using Document asset: " & genericAsset.Name &
				'						vbCrLf & "Type Name: " & GetType(RDV.Generic).Name)

			End If

			'create a new appearance Asset from the collected Asset
			'error checking for existence of name
			nameStr = GenerateName("_My Appearance Asset", dictAssetElem.Keys.ToList)
			Dim thisAsset As RDB.AppearanceAssetElement = RDB.AppearanceAssetElement.Create(thisDoc, nameStr, genericAsset)

			'assign the new asset to the material
			thisMaterial.AppearanceAssetId = thisAsset.Id

			'enable editing of Appearance Assets within the document
			Using editScope As New RDV.AppearanceAssetEditScope(thisDoc)

				'add the newly created and assigned Appearance Asset
				'to the editing session and set it to a variable
				Dim editableAsset As RDV.Asset = editScope.Start(thisAsset.Id)

				'Now let's set some Properties for the Asset:
#Region "Set some Properties on the Asset"
				'** create some reusable variables (not all property types are included in this example)
				Dim strProperty As RDV.AssetPropertyString
				Dim booleanProperty As RDV.AssetPropertyBoolean
				Dim doubleProperty As RDV.AssetPropertyDouble
				Dim colorProperty As RDV.AssetPropertyDoubleArray4d

				'IsEditable and IsValidValue tests should be included as a rule, although we skip some here

				'** set a couple common Schema Properties
				'language neutral call:
				strProperty = CType(editableAsset.FindByName(RDV.SchemaCommon.Description), RDV.AssetPropertyString)
				'language specific call:
				'Dim descriptionProperty As RDV.AssetPropertyString = TryCast(editableAsset.FindByName("description"), RDV.AssetPropertyString)
				strProperty.Value = "My First Generic Appearance Asset"
				strProperty = CType(editableAsset.FindByName(RDV.SchemaCommon.Keyword), RDV.AssetPropertyString)
				strProperty.Value = "Generic,Custom,Concrete"
				strProperty = CType(editableAsset.FindByName(RDV.SchemaCommon.Category), RDV.AssetPropertyString)
				'IsReadOnly failed to catch this call
				'If strProperty.IsReadOnly = False Then
				'	strProperty.Value = ":Generic:Custom:Concrete"
				'End If
				'IsEditable caught it and correctly skipped it
				If strProperty.IsEditable = True Then
					strProperty.Value = ":Generic:Custom:Concrete"
				End If

				'** set a few specific Generic Schema Properties (since we have no idea what we started with)
				booleanProperty = CType(editableAsset.FindByName(RDV.Generic.CommonTintToggle), RDV.AssetPropertyBoolean)
				booleanProperty.Value = True
				booleanProperty = CType(editableAsset.FindByName(RDV.Generic.GenericIsMetal), RDV.AssetPropertyBoolean)
				booleanProperty.Value = False

				doubleProperty = CType(editableAsset.FindByName(RDV.Generic.GenericDiffuseImageFade), RDV.AssetPropertyDouble)
				If doubleProperty.IsEditable = True AndAlso doubleProperty.IsValidValue(0.5) = True Then
					doubleProperty.Value = 0.5
				End If
				doubleProperty = CType(editableAsset.FindByName(RDV.Generic.GenericTransparency), RDV.AssetPropertyDouble)
				If doubleProperty.IsEditable = True AndAlso doubleProperty.IsValidValue(0) = True Then
					doubleProperty.Value = 0
				End If
				doubleProperty = CType(editableAsset.FindByName(RDV.Generic.GenericGlossiness), RDV.AssetPropertyDouble)
				If doubleProperty.IsEditable = True AndAlso doubleProperty.IsValidValue(0.1) = True Then
					doubleProperty.Value = 0.1
				End If

				doubleProperty = CType(editableAsset.FindByName(RDV.Generic.GenericReflectivityAt0deg), RDV.AssetPropertyDouble)
				If doubleProperty.IsEditable = True AndAlso doubleProperty.IsValidValue(0.1) = True Then
					doubleProperty.Value = 0.1
				End If
				doubleProperty = CType(editableAsset.FindByName(RDV.Generic.GenericReflectivityAt90deg), RDV.AssetPropertyDouble)
				If doubleProperty.IsEditable = True AndAlso doubleProperty.IsValidValue(0) = True Then
					doubleProperty.Value = 0
				End If

				'let's try a more complex method for the more complicated Properties
				colorProperty = CType(editableAsset.FindByName(RDV.Generic.CommonTintColor), RDV.AssetPropertyDoubleArray4d)
				'set the value as a color:
				colorProperty.SetValueAsColor(New RDB.Color(127, 127, 127))
				colorProperty = CType(editableAsset.FindByName(RDV.Generic.GenericDiffuse), RDV.AssetPropertyDoubleArray4d)
				'another method is set the value as a list of doubles
				colorProperty.SetValueAsDoubles({0.5, 0.5, 0.5, 1})
#End Region

				'let's attach some images
#Region "Set some bitmaps to a couple of the Properties"
				'create a reusable string for the images
				Dim imagePath As String

				'** add a generic Diffuse image to the Appearance Asset by
				'getting the correct asset property from the asset we just created
				''Dim diffuseMapProperty As RDV.AssetProperty = editableAsset.FindByName("generic_diffuse")
				Dim diffuseMapProperty As RDV.AssetProperty = editableAsset.FindByName(RDV.Generic.GenericDiffuse)
				'then get or create the required connected Bitmap asset,
				'although this might be a bad route if there is more than one connected asset (checker?)
				Dim connectedDiffAsset As RDV.Asset = diffuseMapProperty.GetSingleConnectedAsset()
				' Add a new connected asset if it doesn't already have one
				If connectedDiffAsset Is Nothing Then
					'diffuseMapProperty.AddConnectedAsset("UnifiedBitmap")
					diffuseMapProperty.AddConnectedAsset(GetType(RDV.UnifiedBitmap).Name)
					connectedDiffAsset = diffuseMapProperty.GetSingleConnectedAsset()
				End If
				'test for success before trying to set a path
				If connectedDiffAsset IsNot Nothing Then
					' Find the target asset path property
					'Dim diffuseBitmapProperty As RDV.AssetPropertyString = TryCast(connectedDiffAsset.FindByName("unifiedbitmap_Bitmap"), RDV.AssetPropertyString)
					Dim diffuseBitmapProperty As RDV.AssetPropertyString = TryCast(connectedDiffAsset.FindByName(RDV.UnifiedBitmap.UnifiedbitmapBitmap), RDV.AssetPropertyString)
					'build a path to an image
					imagePath = My.Computer.FileSystem.CombinePath(thisPath, "Concrete.Cast-In-Place.Exposed Aggregate.Medium.jpg")
					If diffuseBitmapProperty.IsValidValue(imagePath) Then
						diffuseBitmapProperty.Value = imagePath
					End If
				End If

				'** add a generic BumpMap Image  to the Appearance Asset
				'Dim bumpMapProperty As RDV.AssetProperty = editableAsset.FindByName("generic_bump_map")
				Dim bumpMapProperty As RDV.AssetProperty = editableAsset.FindByName(RDV.Generic.GenericBumpMap)
				Dim connectedBumpAsset As RDV.Asset = bumpMapProperty.GetSingleConnectedAsset()
				' Add a new connected asset if it doesn't already have one
				If connectedBumpAsset Is Nothing Then
					'bumpMapProperty.AddConnectedAsset("UnifiedBitmap")
					bumpMapProperty.AddConnectedAsset(GetType(RDV.UnifiedBitmap).Name)
					connectedBumpAsset = bumpMapProperty.GetSingleConnectedAsset()
				End If
				If connectedBumpAsset IsNot Nothing Then
					' Find the target asset path property
					'Dim bumpmapBitmapProperty As RDV.AssetPropertyString = TryCast(connectedBumpAsset.FindByName("unifiedbitmap_Bitmap"), RDV.AssetPropertyString)
					Dim bumpmapBitmapProperty As RDV.AssetPropertyString = TryCast(connectedBumpAsset.FindByName(RDV.UnifiedBitmap.UnifiedbitmapBitmap), RDV.AssetPropertyString)
					'build a path to an image
					imagePath = My.Computer.FileSystem.CombinePath(thisPath, "Concrete.Cast-In-Place.Exposed Aggregate.Medium.bump.jpg")
					If bumpmapBitmapProperty.IsValidValue(imagePath) Then
						bumpmapBitmapProperty.Value = imagePath
					End If
				End If
#End Region

				editScope.Commit(False)
			End Using
#End Region

#Region "Add some Physical Property Sets"

			'Create a new Structural property set that can be used by this material
			'error checking for existence of name
			nameStr = GenerateName("_My Structural Asset", dictStruct.Keys.ToList)
			'doublecheck name against unknown dictionary
			Do While dictUnKnown.ContainsKey(nameStr) = True
				nameStr += "_1"
			Loop
			'RUI.TaskDialog.Show("Structural Asset Name", nameStr)
			'create a Concrete structural Asset
			Dim strucAsset As New RDB.StructuralAsset(nameStr, RDB.StructuralAssetClass.Concrete)
			'Set a couple of generic values
			strucAsset.Behavior = RDB.StructuralBehavior.Isotropic
			strucAsset.Density = 232.0
			'create a property set element from the structural asset.
			Dim strucPSE As RDB.PropertySetElement = RDB.PropertySetElement.Create(thisDoc, strucAsset)
			'Assign the property set element to the material.
			thisMaterial.SetMaterialAspectByPropertySet(RDB.MaterialAspect.Structural, strucPSE.Id)
			'set a parameter value on the PSE
			Dim strucPSEParameter As RDB.Parameter = strucPSE.Parameter(RDB.BuiltInParameter.PROPERTY_SET_DESCRIPTION)
			strucPSEParameter.Set("My First Structural Asset")

			'Create a new Thermal property set that can be used by this material
			'error checking for existence of name
			nameStr = GenerateName("_My Thermal Asset", dictTherm.Keys.ToList)
			'doublecheck name against unknown dictionary
			Do While dictUnKnown.ContainsKey(nameStr) = True
				nameStr += "_1"
			Loop
			'RUI.TaskDialog.Show("Thermal Asset Name", nameStr)
			'create a Solid thermal Asset
			Dim thermAsset As New RDB.ThermalAsset(nameStr, RDB.ThermalMaterialType.Solid)
			'set a couple of gereric values
			thermAsset.Behavior = RDB.StructuralBehavior.Isotropic
			thermAsset.Density = 232.0
			'create a property set element from the thermal asset.
			Dim thermPSE As RDB.PropertySetElement = RDB.PropertySetElement.Create(thisDoc, thermAsset)
			'Assign the property set to the material.
			thisMaterial.SetMaterialAspectByPropertySet(RDB.MaterialAspect.Thermal, thermPSE.Id)
			'set a parameter value on the PSE
			Dim thermPSEParameter As RDB.Parameter = thermPSE.Parameter(RDB.BuiltInParameter.PROPERTY_SET_DESCRIPTION)
			thermPSEParameter.Set("My First Thermal Asset")

#End Region

			thisTrans.Commit()
		End Using


#Region "Log the resulting Material"
		'now let's create a report about the material that we just created.
		' Creates a Revit task dialog to communicate information to the user.
		Dim mainDialog As New RUI.TaskDialog("Command Options")
		mainDialog.MainInstruction = "? Include Reports ?"
		mainDialog.MainContent = "Select Yes to Create Reports on the new material" & vbCrLf &
									"Select No to skip Report Creation"
		Dim optButtons As RUI.TaskDialogCommonButtons = RUI.TaskDialogCommonButtons.Yes _
														Or RUI.TaskDialogCommonButtons.No
		mainDialog.CommonButtons = optButtons
		mainDialog.DefaultButton = RUI.TaskDialogResult.No
		Dim diaResult As RUI.TaskDialogResult = mainDialog.Show()

		If diaResult = RUI.TaskDialogResult.Yes Then
			'start a log file:
			Dim docName As String = thisMaterial.Name
			If docName = "" Then
				docName = "Material"
			End If
			Dim logFile As IO.FileInfo = New IO.FileInfo(My.Computer.FileSystem.CombinePath(thisPath, docName & " Material Report.txt"))
			Dim writer As System.IO.StreamWriter = logFile.CreateText()

			writer.WriteLine()
			writer.WriteLine("Document: " & vbTab & thisDoc.Title & vbCrLf)
			writer.WriteLine(vbTab & "Material: " & vbTab & thisMaterial.Name)

			Dim xmlFile As String = My.Computer.FileSystem.CombinePath(thisPath, docName & " Material Report.xml")
			thisXMLDocument.LoadXml("<?xml version=""1.0"" encoding=""utf-8""?>" & ControlChars.NewLine &
									"<Document> </Document>")
			'do not preserve whitespace so the xmldocument class will create indents and the like instead of you having to do it
			thisXMLDocument.PreserveWhitespace = False
			'Get our rootNode
			rootXMLNode = thisXMLDocument.DocumentElement
			rootXMLNode.SetAttribute("Source", thisDoc.Title)

			Dim elemXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("Material")
			elemXmlNode.SetAttribute("Name", thisMaterial.Name)
			rootXMLNode.AppendChild(elemXmlNode)

			'get the info here
			GetMaterialInfo(thisDoc, thisMaterial, elemXmlNode, writer, 2, True)

			'finalize the log files

			thisXMLDocument.Save(xmlFile)

			writer.Flush()
			writer.Close()
			'System.Diagnostics.Process.Start("notepad.exe", logFile.FullName)
		End If

#End Region

	End Sub


	'begin our processing functions
	Private Sub GetMaterialInfo(thisDoc As RDB.Document, thisMat As RDB.Material, parentXmlNode As XML.XmlElement,
								Writer As System.IO.StreamWriter, Optional inDent As Integer = 0,
								Optional incParam As Boolean = False)

		Dim prefixStr As String = ""
		If inDent > 0 Then
			For cntr = 1 To inDent
				prefixStr = prefixStr & vbTab
			Next
		End If

		If thisMat.Parameters.Size > 0 Then
			Writer.WriteLine(prefixStr & " -- Parameters:  " & thisMat.Parameters.Size.ToString)

			Dim paramXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("Parameters")
			paramXmlNode.SetAttribute("Count", thisMat.Parameters.Size.ToString)
			parentXmlNode.AppendChild(paramXmlNode)

			If incParam = True Then
				For Each para As RDB.Parameter In thisMat.Parameters
					Try
						LogParam(para, thisDoc, paramXmlNode, Writer, inDent + 1)
					Catch ex As Exception
						Writer.WriteLine(prefixStr & vbTab & " !! Cannot retrieve Parameter: " & para.Id.ToString)
					End Try
				Next
				Writer.WriteLine(prefixStr & " -- End Parameters" & vbCrLf)
			End If
		End If

#Region "Material Properties"

		''reusing on each asset
		currentXMLNode = thisXMLDocument.CreateElement("MaterialProperties")
		currentXMLNode.SetAttribute("Name", thisMat.Name)
		currentXMLNode.SetAttribute("UniqueId", thisMat.UniqueId)
		parentXmlNode.AppendChild(currentXMLNode)

		Dim propXmlNode As XML.XmlElement

		Writer.WriteLine(vbCrLf & prefixStr & " * Begin Material Properties")
		Dim matCat As String = thisMat.MaterialCategory
		Writer.WriteLine(prefixStr & vbTab & " - Category:  " & matCat)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "Category")
		propXmlNode.SetAttribute("Value", matCat)
		currentXMLNode.AppendChild(propXmlNode)

		Dim matClass As String = thisMat.MaterialClass
		Writer.WriteLine(prefixStr & vbTab & " - Class:  " & matClass)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "Class")
		propXmlNode.SetAttribute("Value", matClass)
		currentXMLNode.AppendChild(propXmlNode)

		Dim matShin As Integer = thisMat.Shininess
		Writer.WriteLine(prefixStr & vbTab & " - Shininess:  " & matShin.ToString)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "Shininess")
		propXmlNode.SetAttribute("Value", matShin.ToString)
		currentXMLNode.AppendChild(propXmlNode)

		Dim matSmooth As Integer = thisMat.Smoothness
		Writer.WriteLine(prefixStr & vbTab & " - Smoothness:  " & matSmooth.ToString)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "Smoothness")
		propXmlNode.SetAttribute("Value", matSmooth.ToString)
		currentXMLNode.AppendChild(propXmlNode)

		Dim matTrans As Integer = thisMat.Transparency
		Writer.WriteLine(prefixStr & vbTab & " - Transparency:  " & matTrans.ToString)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "Transparency")
		propXmlNode.SetAttribute("Value", matTrans.ToString)
		currentXMLNode.AppendChild(propXmlNode)

		Dim matShade As Boolean = thisMat.UseRenderAppearanceForShading
		Writer.WriteLine(prefixStr & vbTab & " - Use Render Asset for Shading:  " & matShade.ToString)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "Use Render Asset for Shading?")
		propXmlNode.SetAttribute("Value", matShade.ToString)
		currentXMLNode.AppendChild(propXmlNode)

		Dim valColor As RDB.Color = thisMat.Color
		Writer.WriteLine(prefixStr & vbTab & " - Color: " & valColor.Red.ToString & ", " & valColor.Green.ToString &
								 ", " & valColor.Blue.ToString)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "Color")
		propXmlNode.SetAttribute("Value", valColor.Red.ToString & ", " & valColor.Green.ToString &
										 ", " & valColor.Blue.ToString)
		currentXMLNode.AppendChild(propXmlNode)


		Dim cutBackColor As RDB.Color = thisMat.CutBackgroundPatternColor
		Writer.WriteLine(prefixStr & vbTab & " - CutBackGroundPatternColor: " & cutBackColor.Red.ToString & ", " & cutBackColor.Green.ToString &
								 ", " & cutBackColor.Blue.ToString)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "CutBackGroundPatternColor")
		propXmlNode.SetAttribute("Value", cutBackColor.Red.ToString & ", " & cutBackColor.Green.ToString &
										 ", " & cutBackColor.Blue.ToString)
		currentXMLNode.AppendChild(propXmlNode)

		Dim cutBackPat As RDB.FillPatternElement = TryCast(thisDoc.GetElement(thisMat.CutBackgroundPatternId), RDB.FillPatternElement)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "CutBackGroundPattern")
		If cutBackPat IsNot Nothing Then
			Writer.WriteLine(prefixStr & vbTab & " - CutBackGroundPattern Name: " & cutBackPat.Name)
			propXmlNode.SetAttribute("Value", cutBackPat.Name)
		Else
			Writer.WriteLine(prefixStr & vbTab & " - CutBackGroundPattern Not Assigned")
			propXmlNode.SetAttribute("Value", "Not Assigned")
		End If
		currentXMLNode.AppendChild(propXmlNode)

		Dim cutFrontColor As RDB.Color = thisMat.CutForegroundPatternColor
		Writer.WriteLine(prefixStr & vbTab & " - CutForeGroundPatternColor: " & cutFrontColor.Red.ToString & ", " & cutFrontColor.Green.ToString &
								 ", " & cutFrontColor.Blue.ToString)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "CutForeGroundPatternColor")
		propXmlNode.SetAttribute("Value", cutFrontColor.Red.ToString & ", " & cutFrontColor.Green.ToString &
										 ", " & cutFrontColor.Blue.ToString)
		currentXMLNode.AppendChild(propXmlNode)

		Dim cutFrontPat As RDB.FillPatternElement = TryCast(thisDoc.GetElement(thisMat.CutForegroundPatternId), RDB.FillPatternElement)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "CutForeGroundPattern")
		If cutFrontPat IsNot Nothing Then
			Writer.WriteLine(prefixStr & vbTab & " - CutForeGroundPattern Name: " & cutFrontPat.Name)
			propXmlNode.SetAttribute("Value", cutFrontPat.Name)
		Else
			Writer.WriteLine(prefixStr & vbTab & " - CutForeGroundPattern Not Assigned")
			propXmlNode.SetAttribute("Value", "Not Assigned")
		End If
		currentXMLNode.AppendChild(propXmlNode)

		Dim surBackColor As RDB.Color = thisMat.SurfaceBackgroundPatternColor
		Writer.WriteLine(prefixStr & vbTab & " - SurfaceBackGroundPatternColor: " & surBackColor.Red.ToString & ", " & surBackColor.Green.ToString &
								 ", " & surBackColor.Blue.ToString)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "SurfaceBackGroundPatternColor")
		propXmlNode.SetAttribute("Value", surBackColor.Red.ToString & ", " & surBackColor.Green.ToString &
										 ", " & surBackColor.Blue.ToString)
		currentXMLNode.AppendChild(propXmlNode)

		Dim surBackPat As RDB.FillPatternElement = TryCast(thisDoc.GetElement(thisMat.SurfaceBackgroundPatternId), RDB.FillPatternElement)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "SurfaceBackGroundPattern")
		If surBackPat IsNot Nothing Then
			Writer.WriteLine(prefixStr & vbTab & " - SurfaceBackGroundPattern Name: " & surBackPat.Name)
			propXmlNode.SetAttribute("Value", surBackPat.Name)
		Else
			Writer.WriteLine(prefixStr & vbTab & " - SurfaceBackGroundPattern Not Assigned")
			propXmlNode.SetAttribute("Value", "Not Assigned")
		End If
		currentXMLNode.AppendChild(propXmlNode)

		Dim surFrontColor As RDB.Color = thisMat.SurfaceForegroundPatternColor
		Writer.WriteLine(prefixStr & vbTab & " - SurfaceForeGroundPatternColor: " & surFrontColor.Red.ToString & ", " & surFrontColor.Green.ToString &
								 ", " & surFrontColor.Blue.ToString)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "SurfaceForeGroundPatternColor")
		propXmlNode.SetAttribute("Value", surFrontColor.Red.ToString & ", " & surFrontColor.Green.ToString &
										 ", " & surFrontColor.Blue.ToString)
		currentXMLNode.AppendChild(propXmlNode)

		Dim surFrontPat As RDB.FillPatternElement = TryCast(thisDoc.GetElement(thisMat.SurfaceForegroundPatternId), RDB.FillPatternElement)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "SurfaceForeGroundPattern")
		If surFrontPat IsNot Nothing Then
			Writer.WriteLine(prefixStr & vbTab & " - SurfaceForeGroundPattern Name: " & surFrontPat.Name)
			propXmlNode.SetAttribute("Value", surFrontPat.Name)
		Else
			Writer.WriteLine(prefixStr & vbTab & " - SurfaceForeGroundPattern Not Assigned")
			propXmlNode.SetAttribute("Value", "Not Assigned")
		End If
		currentXMLNode.AppendChild(propXmlNode)

#End Region

#Region "Begin Assets"

		If thisMat.AppearanceAssetId <> RDB.ElementId.InvalidElementId Then
			Dim renderAssetElem As RDB.AppearanceAssetElement = TryCast(thisDoc.GetElement(thisMat.AppearanceAssetId), RDB.AppearanceAssetElement)

			If renderAssetElem IsNot Nothing Then
				Writer.WriteLine(vbCrLf & prefixStr & "Render Asset Element Name:  " & renderAssetElem.Name &
										 "  Unique ID:  " & renderAssetElem.UniqueId)

				''reusing on each asset
				Dim assetXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("RenderAsset")
				assetXmlNode = thisXMLDocument.CreateElement("RenderAsset")
				assetXmlNode.SetAttribute("Name", renderAssetElem.Name)
				assetXmlNode.SetAttribute("UniqueId", renderAssetElem.UniqueId)
				parentXmlNode.AppendChild(assetXmlNode)

				If renderAssetElem.Parameters.Size > 0 Then
					Writer.WriteLine(prefixStr & vbTab & " -- Asset Parameters:  " & renderAssetElem.Parameters.Size.ToString)

					Dim paramXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("Parameters")
					paramXmlNode.SetAttribute("Count", renderAssetElem.Parameters.Size.ToString)
					assetXmlNode.AppendChild(paramXmlNode)

					If incParam = True Then
						For Each para As RDB.Parameter In renderAssetElem.Parameters
							Try
								LogParam(para, thisDoc, paramXmlNode, Writer, inDent + 2)
							Catch ex As Exception
								Writer.WriteLine(prefixStr & vbTab & vbTab & " !! Cannot retrieve Parameter: " & para.Id.ToString)
							End Try
						Next
						Writer.WriteLine(prefixStr & vbTab & " -- End Parameters" & vbCrLf)
					End If
				End If

				Dim thisAsset As RDV.Asset = renderAssetElem.GetRenderingAsset()
				Writer.WriteLine(vbCrLf & prefixStr & vbTab &
									 "Name:  " & thisAsset.Name &
									 "  |  Asset Type: " & thisAsset.AssetType.ToString)

				Dim thisXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("Asset")
				thisXmlNode.SetAttribute("Name", thisAsset.Name)
				thisXmlNode.SetAttribute("Size", thisAsset.Size.ToString)
				assetXmlNode.AppendChild(thisXmlNode)

				If thisAsset.Size > 0 Then
					GetAssetProperties(thisAsset, thisXmlNode, Writer, inDent + 2)
				End If
			Else
				Writer.WriteLine(prefixStr & vbTab & "**Render Asset Element not found**")
			End If
		Else
			Writer.WriteLine(vbCrLf & prefixStr & vbTab & "**No Appearance Asset Element attached**")
		End If

		If thisMat.StructuralAssetId <> RDB.ElementId.InvalidElementId Then
			Dim struAssetElem As RDB.PropertySetElement = TryCast(thisDoc.GetElement(thisMat.StructuralAssetId), RDB.PropertySetElement)

			If struAssetElem IsNot Nothing Then
				Writer.WriteLine(vbCrLf & prefixStr & vbTab & "Structural Asset Element Name:  " & struAssetElem.Name &
										 "  Unique ID:  " & struAssetElem.UniqueId)
				''save this one for reusing on each asset
				Dim assetXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("StructuralAsset")
				'currentXMLNode = thisXMLDocument.CreateElement("StructuralAsset")
				assetXmlNode.SetAttribute("Name", struAssetElem.Name)
				assetXmlNode.SetAttribute("UniqueId", struAssetElem.UniqueId)
				parentXmlNode.AppendChild(assetXmlNode)

				If struAssetElem.Parameters.Size > 0 Then
					Writer.WriteLine(prefixStr & vbTab & " -- Asset Parameters:  " & struAssetElem.Parameters.Size.ToString)

					Dim paramXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("Parameters")
					paramXmlNode.SetAttribute("Count", struAssetElem.Parameters.Size.ToString)
					assetXmlNode.AppendChild(paramXmlNode)

					If incParam = True Then
						For Each para As RDB.Parameter In struAssetElem.Parameters
							Try
								LogParam(para, thisDoc, paramXmlNode, Writer, inDent + 2)
							Catch ex As Exception
								Writer.WriteLine(prefixStr & vbTab & vbTab & " !! Cannot retrieve Parameter: " & para.Id.ToString)
							End Try
						Next
						Writer.WriteLine(prefixStr & vbTab & " -- End Parameters" & vbCrLf)
					End If
				End If

				Dim struAsset As RDB.StructuralAsset = struAssetElem.GetStructuralAsset

				GetStructAsset(struAsset, assetXmlNode, Writer, inDent + 1)
			Else
				Writer.WriteLine(prefixStr & vbTab & vbTab & "**Structural Asset Element not found**")
			End If
		Else
			Writer.WriteLine(vbCrLf & prefixStr & vbTab & "**No Structural Asset Element attached**")
		End If

		If thisMat.ThermalAssetId <> RDB.ElementId.InvalidElementId Then
			Dim thermAssetElem As RDB.PropertySetElement = TryCast(thisDoc.GetElement(thisMat.ThermalAssetId), RDB.PropertySetElement)

			If thermAssetElem IsNot Nothing Then
				Writer.WriteLine(vbCrLf & prefixStr & vbTab & "Thermal Asset Element Name:  " & thermAssetElem.Name & "  Unique ID:  " & thermAssetElem.UniqueId)

				'sreusing on each asset
				Dim assetXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("ThermalAsset")
				'currentXMLNode = thisXMLDocument.CreateElement("Asset")
				assetXmlNode.SetAttribute("Name", thermAssetElem.Name)
				assetXmlNode.SetAttribute("UniqueId", thermAssetElem.UniqueId)
				parentXmlNode.AppendChild(assetXmlNode)

				If thermAssetElem.Parameters.Size > 0 Then
					Writer.WriteLine(prefixStr & vbTab & " -- Pset Parameters:  " & thermAssetElem.Parameters.Size.ToString)

					Dim paramXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("Parameters")
					paramXmlNode.SetAttribute("Count", thermAssetElem.Parameters.Size.ToString)
					assetXmlNode.AppendChild(paramXmlNode)

					If incParam = True Then
						For Each para As RDB.Parameter In thermAssetElem.Parameters
							Try
								LogParam(para, thisDoc, paramXmlNode, Writer, inDent + 2)
							Catch ex As Exception
								Writer.WriteLine(prefixStr & vbTab & vbTab & " !! Cannot retrieve Parameter: " & para.Id.ToString)
							End Try
						Next
						Writer.WriteLine(prefixStr & vbTab & " -- End Parameters" & vbCrLf)
					End If
				End If

				Dim thermAsset As RDB.ThermalAsset = thermAssetElem.GetThermalAsset

				GetThermAsset(thermAsset, assetXmlNode, Writer, inDent + 1)
			Else
				Writer.WriteLine(vbCrLf & prefixStr & vbTab & vbTab & "**Thermal Asset Element not found**")
			End If
		Else
			Writer.WriteLine(vbCrLf & prefixStr & vbTab & "**No Thermal Asset Element attached**")
		End If

#End Region

	End Sub

	Private Sub GetStructAsset(thisAsset As RDB.StructuralAsset, parentXmlNode As XML.XmlElement,
							   writer As StreamWriter, Optional inDent As Integer = 0)

		Dim prefixStr As String = ""
		If inDent > 0 Then
			For cntr = 1 To inDent
				prefixStr = prefixStr & vbTab
			Next
		End If

		writer.WriteLine(prefixStr & "  -  Structural Asset Name: " & thisAsset.Name &
						 "  |  Class: " & thisAsset.StructuralAssetClass.ToString &
						 "  |  Sub-Class: " & thisAsset.SubClass)

		inDent += 1
		prefixStr = prefixStr & vbTab

		writer.WriteLine(prefixStr & "  : Structural Properties : ")

		Dim thisXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("Asset")
		thisXmlNode.SetAttribute("Name", thisAsset.Name)
		thisXmlNode.SetAttribute("Class", thisAsset.StructuralAssetClass.ToString)
		thisXmlNode.SetAttribute("SubClass", thisAsset.SubClass)
		parentXmlNode.AppendChild(thisXmlNode)

		Dim propXmlNode As XML.XmlElement

		'26 total possible?
		'it looks like all of these have (or return) values all of the time, regardless of type and what is shown in the UI
		'but we will break them down by AssetClass anyway to reduce the printed lines

		'let's start breaking it down:
		If thisAsset.StructuralAssetClass <> RDB.StructuralAssetClass.Undefined Then
			'common
			'under Basic Mechanical; everything except perhaps "Undefined"?
			Dim assetDensity As Double = thisAsset.Density
			writer.WriteLine(prefixStr & "  -  Density: " & assetDensity.ToString)

			propXmlNode = thisXMLDocument.CreateElement("Property")
			propXmlNode.SetAttribute("Name", "Density")
			propXmlNode.SetAttribute("Value", assetDensity.ToString)
			thisXmlNode.AppendChild(propXmlNode)

			If thisAsset.StructuralAssetClass <> RDB.StructuralAssetClass.Basic Then
				'everything except Basic
				'under Basic Thermal;
				'ThermalExpansionCoefficient As RDB.XYZ **Wood or isotrophic behavior elements, use setThermalExpansionCoefficient to set
				Dim assetTEC As RDB.XYZ = thisAsset.ThermalExpansionCoefficient
				writer.WriteLine(prefixStr & "  -  Thermal Expansion Coefficient: " & assetTEC.X.ToString & ", " & assetTEC.Y.ToString & ", " & assetTEC.Z.ToString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Thermal Expansion Coefficient")
				propXmlNode.SetAttribute("Value", assetTEC.X.ToString & ", " & assetTEC.Y.ToString & ", " & assetTEC.Z.ToString)
				thisXmlNode.AppendChild(propXmlNode)

				If thisAsset.StructuralAssetClass <> RDB.StructuralAssetClass.Gas Then
					'everything except Gas and Basic
					'under behavior;
					Dim assetBehavior As RDB.StructuralBehavior = thisAsset.Behavior
					writer.WriteLine(prefixStr & "  -  Structural Behavior: " & assetBehavior.ToString)
					propXmlNode = thisXMLDocument.CreateElement("Property")
					propXmlNode.SetAttribute("Name", "Structural Behavior")
					propXmlNode.SetAttribute("Value", assetBehavior.ToString)
					thisXmlNode.AppendChild(propXmlNode)

					If thisAsset.StructuralAssetClass <> RDB.StructuralAssetClass.Liquid Then
						'everything except Gas, liquid, or basic
						'Basic Mechanical;
						'PoissonRatio As RDB.XYZ **Wood or isotrophic behavior elements, use setPoissonRatio to set
						Dim assetPoRa As RDB.XYZ = thisAsset.PoissonRatio
						writer.WriteLine(prefixStr & "  -  Poisson Ratio: " & assetPoRa.X.ToString & ", " & assetPoRa.Y.ToString & ", " & assetPoRa.Z.ToString)
						propXmlNode = thisXMLDocument.CreateElement("Property")
						propXmlNode.SetAttribute("Name", "Poisson Ratio")
						propXmlNode.SetAttribute("Value", assetPoRa.X.ToString & ", " & assetPoRa.Y.ToString & ", " & assetPoRa.Z.ToString)
						thisXmlNode.AppendChild(propXmlNode)

						'ShearModulus As RDB.XYZ **Wood or isotrophic behavior elements, use setShearModulus to set
						Dim assetShMo As RDB.XYZ = thisAsset.ShearModulus
						writer.WriteLine(prefixStr & "  -  Shear Modulus: " & assetShMo.X.ToString & ", " & assetShMo.Y.ToString & ", " & assetShMo.Z.ToString)
						propXmlNode = thisXMLDocument.CreateElement("Property")
						propXmlNode.SetAttribute("Name", "Shear Modulus")
						propXmlNode.SetAttribute("Value", assetShMo.X.ToString & ", " & assetShMo.Y.ToString & ", " & assetShMo.Z.ToString)
						thisXmlNode.AppendChild(propXmlNode)

						'YoungModulus As RDB.XYZ **Wood or isotrophic behavior elements, use setYoungModulus to set
						Dim assetYoMo As RDB.XYZ = thisAsset.YoungModulus
						writer.WriteLine(prefixStr & "  -  Young Modulus: " & assetYoMo.X.ToString & ", " & assetYoMo.Y.ToString & ", " & assetYoMo.Z.ToString)
						propXmlNode = thisXMLDocument.CreateElement("Property")
						propXmlNode.SetAttribute("Name", "Young Modulus")
						propXmlNode.SetAttribute("Value", assetYoMo.X.ToString & ", " & assetYoMo.Y.ToString & ", " & assetYoMo.Z.ToString)
						thisXmlNode.AppendChild(propXmlNode)


						'location varies under strength or concrete;
						Dim assetMTS As Double = thisAsset.MinimumTensileStrength
						writer.WriteLine(prefixStr & "  -  Minimum Tensile Strength: " & assetMTS.ToString)

						propXmlNode = thisXMLDocument.CreateElement("Property")
						propXmlNode.SetAttribute("Name", "Minimum Tensile Strength")
						propXmlNode.SetAttribute("Value", assetMTS.ToString)
						thisXmlNode.AppendChild(propXmlNode)

						Dim assetMYS As Double = thisAsset.MinimumTensileStrength
						writer.WriteLine(prefixStr & "  -  Minimum Yield Stress: " & assetMYS.ToString)
						propXmlNode = thisXMLDocument.CreateElement("Property")
						propXmlNode.SetAttribute("Name", "Minimum Yield Stress")
						propXmlNode.SetAttribute("Value", assetMYS.ToString)
						thisXmlNode.AppendChild(propXmlNode)


						'then get into specific cases
						Select Case thisAsset.StructuralAssetClass
							Case RDB.StructuralAssetClass.Concrete
								'Concrete specific
								writer.WriteLine(prefixStr & "  * Concrete Specific Properties: ")
								Dim assetCBR As Double = thisAsset.ConcreteBendingReinforcement
								writer.WriteLine(prefixStr & "  -  Concrete Bending Reinforcement: " & assetCBR.ToString)
								propXmlNode = thisXMLDocument.CreateElement("Property")
								propXmlNode.SetAttribute("Name", "Concrete Bending Reinforcement")
								propXmlNode.SetAttribute("Value", assetCBR.ToString)
								thisXmlNode.AppendChild(propXmlNode)

								Dim assetCCS As Double = thisAsset.ConcreteCompression
								writer.WriteLine(prefixStr & "  -  Concrete Compression Strength: " & assetCCS.ToString)
								propXmlNode = thisXMLDocument.CreateElement("Property")
								propXmlNode.SetAttribute("Name", "Concrete Compression Strength")
								propXmlNode.SetAttribute("Value", assetCCS.ToString)
								thisXmlNode.AppendChild(propXmlNode)

								Dim assetCSR As Double = thisAsset.ConcreteShearReinforcement
								writer.WriteLine(prefixStr & "  -  Concrete Shear Reinforcement: " & assetCSR.ToString)
								propXmlNode = thisXMLDocument.CreateElement("Property")
								propXmlNode.SetAttribute("Name", "Concrete Shear Reinforcement")
								propXmlNode.SetAttribute("Value", assetCSR.ToString)
								thisXmlNode.AppendChild(propXmlNode)

								Dim assetCSSR As Double = thisAsset.ConcreteShearStrengthReduction
								writer.WriteLine(prefixStr & "  -  Concrete Shear Strength Reduction: " & assetCSSR.ToString)
								propXmlNode = thisXMLDocument.CreateElement("Property")
								propXmlNode.SetAttribute("Name", "Concrete Shear Strength Reduction")
								propXmlNode.SetAttribute("Value", assetCSSR.ToString)
								thisXmlNode.AppendChild(propXmlNode)

								Dim assetCWeight As Boolean = thisAsset.Lightweight
								writer.WriteLine(prefixStr & "  -  Light Weight Concrete: " & assetCWeight.ToString)
								propXmlNode = thisXMLDocument.CreateElement("Property")
								propXmlNode.SetAttribute("Name", "Light Weight Concrete")
								propXmlNode.SetAttribute("Value", assetCWeight.ToString)
								thisXmlNode.AppendChild(propXmlNode)

							Case RDB.StructuralAssetClass.Metal
								'Metal specific
								writer.WriteLine(prefixStr & "  * Metal Specific Properties: ")
								Dim assetMetalRF As Double = thisAsset.MetalReductionFactor
								writer.WriteLine(prefixStr & "  -  Metal Reduction Factor: " & assetMetalRF.ToString)
								propXmlNode = thisXMLDocument.CreateElement("Property")
								propXmlNode.SetAttribute("Name", "Metal Reduction Factor")
								propXmlNode.SetAttribute("Value", assetMetalRF.ToString)
								thisXmlNode.AppendChild(propXmlNode)

								Dim assetMetalRCS As Double = thisAsset.MetalResistanceCalculationStrength
								writer.WriteLine(prefixStr & "  -  Metal Resistance Calculation Strength: " & assetMetalRCS.ToString)
								propXmlNode = thisXMLDocument.CreateElement("Property")
								propXmlNode.SetAttribute("Name", "Metal Resistance Calculation Strength")
								propXmlNode.SetAttribute("Value", assetMetalRCS.ToString)
								thisXmlNode.AppendChild(propXmlNode)

								Dim assetMetalTT As Boolean = thisAsset.MetalThermallyTreated
								writer.WriteLine(prefixStr & "  -  Metal Thermally Treated: " & assetMetalTT.ToString)
								propXmlNode = thisXMLDocument.CreateElement("Property")
								propXmlNode.SetAttribute("Name", "Metal Thermally Treated")
								propXmlNode.SetAttribute("Value", assetMetalTT.ToString)
								thisXmlNode.AppendChild(propXmlNode)

							Case RDB.StructuralAssetClass.Wood
								'wood specific (all properties here shown under strength)
								writer.WriteLine(prefixStr & "  * Wood Specific Properties: ")
								Dim assetWGrade As String = thisAsset.WoodGrade
								writer.WriteLine(prefixStr & "  -  Wood Grade: " & assetWGrade)
								propXmlNode = thisXMLDocument.CreateElement("Property")
								propXmlNode.SetAttribute("Name", "Wood Grade")
								propXmlNode.SetAttribute("Value", assetWGrade)
								thisXmlNode.AppendChild(propXmlNode)

								Dim assetWSp As String = thisAsset.WoodSpecies
								writer.WriteLine(prefixStr & "  -  Wood Species: " & assetWSp)
								propXmlNode = thisXMLDocument.CreateElement("Property")
								propXmlNode.SetAttribute("Name", "Wood Species")
								propXmlNode.SetAttribute("Value", assetWSp)
								thisXmlNode.AppendChild(propXmlNode)

								Dim assetWoodBS As Double = thisAsset.WoodBendingStrength
								writer.WriteLine(prefixStr & "  -  Wood Bending Strength: " & assetWoodBS.ToString)
								propXmlNode = thisXMLDocument.CreateElement("Property")
								propXmlNode.SetAttribute("Name", "Wood Bending Strength")
								propXmlNode.SetAttribute("Value", assetWoodBS.ToString)
								thisXmlNode.AppendChild(propXmlNode)

								Dim assetWoodPCS As Double = thisAsset.WoodParallelCompressionStrength
								writer.WriteLine(prefixStr & "  -  Wood Parallel Compression Strength: " & assetWoodPCS.ToString)
								propXmlNode = thisXMLDocument.CreateElement("Property")
								propXmlNode.SetAttribute("Name", "Wood Parallel Compression Strength")
								propXmlNode.SetAttribute("Value", assetWoodPCS.ToString)
								thisXmlNode.AppendChild(propXmlNode)

								Dim assetWoodPSS As Double = thisAsset.WoodParallelShearStrength
								writer.WriteLine(prefixStr & "  -  Wood Parallel Shear Strength: " & assetWoodPSS.ToString)
								propXmlNode = thisXMLDocument.CreateElement("Property")
								propXmlNode.SetAttribute("Name", "Wood Parallel Shear Strength")
								propXmlNode.SetAttribute("Value", assetWoodPSS.ToString)
								thisXmlNode.AppendChild(propXmlNode)

								Dim assetWoodCS As Double = thisAsset.WoodPerpendicularCompressionStrength
								writer.WriteLine(prefixStr & "  -  Wood Perpendicular Compression Strength: " & assetWoodCS.ToString)
								propXmlNode = thisXMLDocument.CreateElement("Property")
								propXmlNode.SetAttribute("Name", "Wood Perpendicular Compression Strength")
								propXmlNode.SetAttribute("Value", assetWoodCS.ToString)
								thisXmlNode.AppendChild(propXmlNode)

								Dim assetWoodSS As Double = thisAsset.WoodPerpendicularShearStrength
								writer.WriteLine(prefixStr & "  -  Wood Perpendicular Shear Strength: " & assetWoodSS.ToString)
								propXmlNode = thisXMLDocument.CreateElement("Property")
								propXmlNode.SetAttribute("Name", "Wood Perpendicular Shear Strength")
								propXmlNode.SetAttribute("Value", assetWoodSS.ToString)
								thisXmlNode.AppendChild(propXmlNode)

								'Case RDB.StructuralAssetClass.Basic
								'	'Basic specific
								'	writer.WriteLine(vbTab & vbTab & "  * Basic Properties: ")
								'Case RDB.StructuralAssetClass.Gas
								'	'Gas specific
								'	writer.WriteLine(vbTab & vbTab & "  * Gas Specific Properties: ")
								'Case RDB.StructuralAssetClass.Generic
								'	'Genric specific
								'	writer.WriteLine(vbTab & vbTab & "  * Generic Properties: ")
								'Case RDB.StructuralAssetClass.Liquid
								'	'Liquid specific
								'	writer.WriteLine(vbTab & vbTab & "  * Liquid Specific Properties: ")
								'Case RDB.StructuralAssetClass.Plastic
								'	'Plastic specific
								'	writer.WriteLine(vbTab & vbTab & "  * Plastic Specific Properties: ")
								'Case RDB.StructuralAssetClass.Undefined
						End Select
					Else
						writer.WriteLine(prefixStr & "  * No additional Liquid Specific Properties")
					End If
				Else
					writer.WriteLine(prefixStr & "  * No additional Gas Specific Properties")
				End If
			Else
				writer.WriteLine(prefixStr & "  * No additional Basic Properties")
			End If
		Else
			writer.WriteLine(prefixStr & "  * No Properties")
		End If

		writer.WriteLine(prefixStr & "  : End Structural Properties : ")

	End Sub

	Private Sub GetThermAsset(thisAsset As RDB.ThermalAsset, parentXmlNode As XML.XmlElement,
							  writer As StreamWriter, Optional inDent As Integer = 0)

		Dim prefixStr As String = ""
		If inDent > 0 Then
			For cntr = 1 To inDent
				prefixStr = prefixStr & vbTab
			Next
		End If


		writer.WriteLine(prefixStr & "  -  Thermal Asset Name: " & thisAsset.Name &
						 "  |  Class: " & thisAsset.ThermalMaterialType.ToString)

		inDent += 1
		prefixStr = prefixStr & vbTab

		writer.WriteLine(prefixStr & "  : Thermal Properties : ")

		Dim thisXmlNode As XML.XmlElement = thisXMLDocument.CreateElement("Asset")
		thisXmlNode.SetAttribute("Name", thisAsset.Name)
		thisXmlNode.SetAttribute("Class", thisAsset.ThermalMaterialType.ToString)
		thisXmlNode.SetAttribute("SubClass", "")
		parentXmlNode.AppendChild(thisXmlNode)

		Dim propXmlNode As XML.XmlElement

		'16 total possible
		'Common
		'hope we don't need to filter out "Undefined"
		'ThermalConductivity As Double {must be non-negative}
		Dim assetThermConduc As Double = thisAsset.ThermalConductivity
		writer.WriteLine(prefixStr & "  -  Thermal Conductivity: " & assetThermConduc.ToString)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "Thermal Conductivity")
		propXmlNode.SetAttribute("Value", assetThermConduc.ToString)
		thisXmlNode.AppendChild(propXmlNode)
		'SpecificHeat As Double {must be non-negative}
		Dim assetSpecHeat As Double = thisAsset.SpecificHeat
		writer.WriteLine(prefixStr & "  -  Specific Heat: " & assetSpecHeat.ToString)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "Specific Heat")
		propXmlNode.SetAttribute("Value", assetSpecHeat.ToString)
		thisXmlNode.AppendChild(propXmlNode)
		'Density As Double {must be non-negative}
		Dim assetDensity As Double = thisAsset.Density
		writer.WriteLine(prefixStr & "  -  Density: " & assetDensity.ToString)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "Density")
		propXmlNode.SetAttribute("Value", assetDensity.ToString)
		thisXmlNode.AppendChild(propXmlNode)
		'Emissivity As Double {0 to 1}
		Dim assetEmiss As Double = thisAsset.Emissivity
		writer.WriteLine(prefixStr & "  -  Emissivity: " & assetEmiss.ToString)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "Emissivity")
		propXmlNode.SetAttribute("Value", assetEmiss.ToString)
		thisXmlNode.AppendChild(propXmlNode)

		If thisAsset.ThermalMaterialType = RDB.ThermalMaterialType.Gas OrElse
				thisAsset.ThermalMaterialType = RDB.ThermalMaterialType.Liquid Then
			'only one that appears in two distinct Types but not the rest
			'Compressibility As Double {0 to 1}
			Dim assetComp As Double = thisAsset.Compressibility
			writer.WriteLine(prefixStr & "  -  Compressibility: " & assetComp.ToString)
			propXmlNode = thisXMLDocument.CreateElement("Property")
			propXmlNode.SetAttribute("Name", "Compressibility")
			propXmlNode.SetAttribute("Value", assetComp.ToString)
			thisXmlNode.AppendChild(propXmlNode)
		End If

		Select Case thisAsset.ThermalMaterialType
			Case RDB.ThermalMaterialType.Gas
				'GasViscosity As Double {must be non-negative}
				Dim assetGasVis As Double = thisAsset.GasViscosity
				writer.WriteLine(prefixStr & "  -  Gas Viscosity: " & assetGasVis.ToString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Gas Viscosity")
				propXmlNode.SetAttribute("Value", assetGasVis.ToString)
				thisXmlNode.AppendChild(propXmlNode)

			Case RDB.ThermalMaterialType.Liquid
				'LiquidViscosity As Double {must be non-negative}
				Dim assetLiqVis As Double = thisAsset.LiquidViscosity
				writer.WriteLine(prefixStr & "  -  Liquid Viscosity: " & assetLiqVis.ToString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Liquid Viscosity")
				propXmlNode.SetAttribute("Value", assetLiqVis.ToString)
				thisXmlNode.AppendChild(propXmlNode)

				'SpecificHeatOfVaporization As Double {must be non-negative}
				Dim assetSHV As Double = thisAsset.SpecificHeatOfVaporization
				writer.WriteLine(prefixStr & "  -  Specific Heat Of Vaporization: " & assetSHV.ToString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Specific Heat Of Vaporization")
				propXmlNode.SetAttribute("Value", assetSHV.ToString)
				thisXmlNode.AppendChild(propXmlNode)

				'VaporPressure As Double {must be non-negative}
				Dim assetVP As Double = thisAsset.VaporPressure
				writer.WriteLine(prefixStr & "  -  Vapor Pressure: " & assetVP.ToString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Vapor Pressure")
				propXmlNode.SetAttribute("Value", assetVP.ToString)
				thisXmlNode.AppendChild(propXmlNode)

			Case RDB.ThermalMaterialType.Solid
				'TransmitsLight As Boolean
				Dim assetTransLight As Boolean = thisAsset.TransmitsLight
				writer.WriteLine(prefixStr & "  -  Transmits Light: " & assetTransLight.ToString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Transmits Light")
				propXmlNode.SetAttribute("Value", assetTransLight.ToString)
				thisXmlNode.AppendChild(propXmlNode)

				'Behavior As RDB.StructuralBehavior <- No, that isn't a mistake
				Dim assetBehavior As RDB.StructuralBehavior = thisAsset.Behavior
				writer.WriteLine(prefixStr & "  -  Behavior: " & assetBehavior.ToString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Behavior")
				propXmlNode.SetAttribute("Value", assetBehavior.ToString)
				thisXmlNode.AppendChild(propXmlNode)

				'Permeability As Double {must be non-negative}
				Dim assetPerm As Double = thisAsset.Permeability
				writer.WriteLine(prefixStr & "  -  Permeability: " & assetPerm.ToString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Permeability")
				propXmlNode.SetAttribute("Value", assetPerm.ToString)
				thisXmlNode.AppendChild(propXmlNode)

				'Porosity As Double {0 to 1}
				Dim assetPoros As Double = thisAsset.Porosity
				writer.WriteLine(prefixStr & "  -  Porosity: " & assetPoros.ToString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Porosity")
				propXmlNode.SetAttribute("Value", assetPoros.ToString)
				thisXmlNode.AppendChild(propXmlNode)

				'Reflectivity As Double {0 to 1}
				Dim assetRefl As Double = thisAsset.Reflectivity
				writer.WriteLine(prefixStr & "  -  Reflectivity: " & assetRefl.ToString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Reflectivity")
				propXmlNode.SetAttribute("Value", assetRefl.ToString)
				thisXmlNode.AppendChild(propXmlNode)

				'ElectricalResistivity As Double {must be non-negative}
				Dim assetElRes As Double = thisAsset.ElectricalResistivity
				writer.WriteLine(prefixStr & "  -  ElectricalResistivity: " & assetElRes.ToString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "ElectricalResistivity")
				propXmlNode.SetAttribute("Value", assetElRes.ToString)
				thisXmlNode.AppendChild(propXmlNode)

			Case RDB.ThermalMaterialType.Undefined
				writer.WriteLine(prefixStr & "  * Undefined Thermal Type: ")
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Undefined Thermal Type")
				propXmlNode.SetAttribute("Value", "")
				thisXmlNode.AppendChild(propXmlNode)
		End Select

		writer.WriteLine(prefixStr & "  : End Thermal Properties : ")

	End Sub

	Private Sub GetAssetProperties(thisAsset As RDV.Asset, parentXmlNode As XML.XmlElement,
								   writer As StreamWriter, Optional inDent As Integer = 0)

		'Could consider trying to add a UnifiedBitmap and a BumpmapBitmap
		'to see if the property has the option available to it
		'since we can't find any property that will tell us if it's capable
		'would require starting a transaction,
		'Using thisTrans As New RDB.Transaction(thisDoc, "Temp Material")
		'	thisTrans.Start()
		'create an editscope
		'Using editScope As New RDV.AppearanceAssetEditScope(thisDoc)
		'assign the incoming asset to it
		'	Dim editableAsset As RDV.Asset = editScope.Start(thisAsset.Id)

		'enter property loop
		'test for connected Assets, if they're there then determine kind if not then
		'attempt to attach a connected asset of each type to each property in a Try... Catch...
		'we already have the property so we don't need to go get it like we did here
		'Dim diffuseMapProperty As RDV.AssetProperty = editableAsset.FindByName("generic_diffuse")
		'diffuseMapProperty.AddConnectedAsset("UnifiedBitmap")
		'diffuseMapProperty.AddConnectedAsset("BumpMap")
		'then report the success and/or failure of each
		'exit property loop

		For i As Integer = 0 To thisAsset.Size - 1
			Dim assetProp As RDV.AssetProperty = thisAsset.Item(i)
			GetAssetPropValues(assetProp, parentXmlNode, writer, inDent)
		Next

		'then rollback the editscope
		'EditScope.Cancel()
		'End Using
		'then rollback the transaction.
		'thisTrans.RollBack()


	End Sub

	Private Sub GetAssetPropValues(assetProp As RDV.AssetProperty, parentXmlNode As XML.XmlElement,
								   writer As StreamWriter, Optional inDent As Integer = 0)
		Dim printStr As String = ""
		Dim xmlValueStr As String = ""

		Dim prefixStr As String = ""
		If inDent > 0 Then
			For cntr = 1 To inDent
				prefixStr = prefixStr & vbTab
			Next
		End If

		'select case does not require an exit select since it will only process a single match
		Select Case assetProp.Type
			Case RDV.AssetPropertyType.Asset
				'sub class is Asset As RDV.Asset (loop for embedded)
				Dim thisProp As RDV.Asset = TryCast(assetProp, RDV.Asset)
				If thisProp IsNot Nothing Then
					'printStr = "** Embedded Asset **" & "  |  Value: " & thisProp.Name
					writer.WriteLine(vbCrLf & prefixStr & "** Embedded Asset **")
					writer.WriteLine(prefixStr &
							 "Name:  " & thisProp.Name & "  |  Size: " & thisProp.Size.ToString &
							 "  |  Library Name: " & thisProp.LibraryName &
							 "  |  Asset Type: " & thisProp.AssetType.ToString &
							 "  |  Asset Class: " & thisProp.Type.ToString)

					Dim assetXMLNode As XML.XmlElement = thisXMLDocument.CreateElement("Embedded")
					assetXMLNode.SetAttribute("Name", thisProp.Name)
					assetXMLNode.SetAttribute("Size", thisProp.Size.ToString)
					assetXMLNode.SetAttribute("Library", thisProp.LibraryName)
					assetXMLNode.SetAttribute("Type", thisProp.AssetType.ToString)
					assetXMLNode.SetAttribute("Class", thisProp.Type.ToString)
					parentXmlNode.AppendChild(assetXMLNode)

					GetAssetProperties(thisProp, assetXMLNode, writer, inDent + 1)
					writer.WriteLine(prefixStr & "** End Embedded Asset **" & vbCrLf)
					'exit the sub because we don't want the default writeline call at the end
					Exit Sub
				End If
			Case RDV.AssetPropertyType.Boolean
				'sub class is AssetPropertyBoolean As Boolean
				Dim thisProp As RDV.AssetPropertyBoolean = TryCast(assetProp, RDV.AssetPropertyBoolean)
				If thisProp IsNot Nothing Then
					Dim val As Boolean = thisProp.Value
					printStr = "  |  Value: " & val.ToString
					xmlValueStr = val.ToString
				End If
			Case RDV.AssetPropertyType.Distance
				'sub class is AssetPropertyDistance As Double, with method (GetUnitTypeId As ForgeTypeID)
				Dim thisProp As RDV.AssetPropertyDistance = TryCast(assetProp, RDV.AssetPropertyDistance)
				If thisProp IsNot Nothing Then
					Dim val As Double = thisProp.Value
					printStr = "  |  Unit Type: " & thisProp.GetUnitTypeId.TypeId & "  |  Value: " & val.ToString
					xmlValueStr = val.ToString & "  |  Unit Type: " & thisProp.GetUnitTypeId.TypeId
				End If
			Case RDV.AssetPropertyType.Double1
				'sub class is AssetPropertyDouble As Double
				Dim thisProp As RDV.AssetPropertyDouble = TryCast(assetProp, RDV.AssetPropertyDouble)
				If thisProp IsNot Nothing Then
					Dim val As Double = thisProp.Value
					printStr = "  |  Value: " & val.ToString
					xmlValueStr = val.ToString
				End If
			Case RDV.AssetPropertyType.Double2
				'sub class is AssetPropertyDoubleArray2d As RDB.DoubleArray
				Dim thisProp As RDV.AssetPropertyDoubleArray2d = TryCast(assetProp, RDV.AssetPropertyDoubleArray2d)
				If thisProp IsNot Nothing Then
					If thisProp.Value.Size > 0 Then
						Dim val As RDB.DoubleArray = thisProp.Value
						For i As Integer = 0 To val.Size - 1
							printStr += "  |  Value: " & val.Item(i).ToString
							xmlValueStr += " | " & val.Item(i).ToString
						Next
					End If
				End If
			Case RDV.AssetPropertyType.Double3
				'sub class is AssetPropertyDoubleArray3d - no Value property
				'use methods (GetValueAsDoubles As IList(Of Double), GetValueAsXYZ As XYZ)
				Dim thisProp As RDV.AssetPropertyDoubleArray3d = TryCast(assetProp, RDV.AssetPropertyDoubleArray3d)
				If thisProp IsNot Nothing Then
					For Each val As Double In thisProp.GetValueAsDoubles
						printStr += "  |  Value: " & val.ToString
						xmlValueStr += " | " & val.ToString
					Next
					Dim valXYZ As RDB.XYZ = thisProp.GetValueAsXYZ
					printStr += "  |  As XYZ: " & valXYZ.X.ToString & ", " & valXYZ.Y.ToString & ", " & valXYZ.Z.ToString
					xmlValueStr += "  |  As XYZ: " & valXYZ.X.ToString & ", " & valXYZ.Y.ToString & ", " & valXYZ.Z.ToString
				End If
			Case RDV.AssetPropertyType.Double4
				'sub class is AssetPropertyDoubleArray4d - no Value property
				'use methods (GetValueAsDoubles As IList(Of Double), GetValueAsColor As Color)
				Dim thisProp As RDV.AssetPropertyDoubleArray4d = TryCast(assetProp, RDV.AssetPropertyDoubleArray4d)
				If thisProp IsNot Nothing Then
					For Each val As Double In thisProp.GetValueAsDoubles
						printStr += "  |  Value: " & val.ToString
						xmlValueStr += " | " & val.ToString
					Next
					Dim valColor As RDB.Color = thisProp.GetValueAsColor
					printStr += "  |  As RGB: " & valColor.Red.ToString & ", " & valColor.Green.ToString & ", " & valColor.Blue.ToString
					xmlValueStr += "  |  As RGB: " & valColor.Red.ToString & ", " & valColor.Green.ToString & ", " & valColor.Blue.ToString
				End If
			Case RDV.AssetPropertyType.Double44
				'sub class is AssetPropertyDoubleMatrix44 As RDB.DoubleArray
				Dim thisProp As RDV.AssetPropertyDoubleMatrix44 = TryCast(assetProp, RDV.AssetPropertyDoubleMatrix44)
				If thisProp IsNot Nothing Then
					If thisProp.Value.Size > 0 Then
						Dim val As RDB.DoubleArray = thisProp.Value
						For i As Integer = 0 To val.Size - 1
							printStr += "  |  Value: " & val.Item(i).ToString
							xmlValueStr += " | " & val.Item(i).ToString
						Next
					End If
				End If
			Case RDV.AssetPropertyType.Enumeration
				'sub class is AssetPropertyInteger As Integer
				'never returns anything
				Dim thisProp As RDV.AssetPropertyInteger = TryCast(assetProp, RDV.AssetPropertyInteger)
				If thisProp IsNot Nothing Then
					Dim val As Integer = thisProp.Value
					printStr = "  |  Value: " & val.ToString
					xmlValueStr = val.ToString
				End If
			Case RDV.AssetPropertyType.Float
				'sub class is AssetPropertyFloat As Single
				Dim thisProp As RDV.AssetPropertyFloat = TryCast(assetProp, RDV.AssetPropertyFloat)
				If thisProp IsNot Nothing Then
					Dim val As Single = thisProp.Value
					printStr = "  |  Value: " & val.ToString
					xmlValueStr = val.ToString
				End If
			Case RDV.AssetPropertyType.Float3
				'sub class is AssetPropertyFloatArray - no Value property
				'use method (GetValue As IList(Of Single))
				Dim thisProp As RDV.AssetPropertyFloatArray = TryCast(assetProp, RDV.AssetPropertyFloatArray)
				If thisProp IsNot Nothing Then
					For Each val As Single In thisProp.GetValue
						printStr += "  |  Value: " & val.ToString
						xmlValueStr += " | " & val.ToString
					Next
				End If
			Case RDV.AssetPropertyType.Integer
				'sub class is AssetPropertyInteger As Integer
				Dim thisProp As RDV.AssetPropertyInteger = TryCast(assetProp, RDV.AssetPropertyInteger)
				If thisProp IsNot Nothing Then
					Dim val As Integer = thisProp.Value
					printStr = "  |  Value: " & val.ToString
					xmlValueStr = val.ToString
				End If
			Case RDV.AssetPropertyType.List
				'sub class is AssetPropertyList - no Value property
				'use method (GetValue As IList(Of AssetProperty))
				Dim thisProp As RDV.AssetPropertyList = TryCast(assetProp, RDV.AssetPropertyList)
				If thisProp IsNot Nothing Then
					writer.WriteLine(prefixStr & "Property Name:  " & assetProp.Name &
									 "  |  List Count: " & thisProp.GetValue().Count &
									 "  |  Property Type:  " & assetProp.Type.ToString &
									 "  |  API Class Name: " & assetProp.GetType().Name)

					Dim listXMLNode As XML.XmlElement = thisXMLDocument.CreateElement("PropertyList")
					listXMLNode.SetAttribute("Name", assetProp.Name)
					listXMLNode.SetAttribute("Count", thisProp.GetValue().Count)
					'listXMLNode.SetAttribute("ValueType", thisProp.GetValue().FirstOrDefault.Type.ToString)
					listXMLNode.SetAttribute("Type", assetProp.Type.ToString)
					listXMLNode.SetAttribute("Class", assetProp.GetType().Name)
					parentXmlNode.AppendChild(listXMLNode)

					For Each val As RDV.AssetProperty In thisProp.GetValue()
						GetAssetPropValues(val, listXMLNode, writer, inDent + 1)
					Next

					'exit the sub because we don't want the default writeline call at the end
					Exit Sub
				End If
			Case RDV.AssetPropertyType.Longlong
				'sub class is AssetPropertyInt64 As Long
				Dim thisProp As RDV.AssetPropertyInt64 = TryCast(assetProp, RDV.AssetPropertyInt64)
				If thisProp IsNot Nothing Then
					Dim val As Long = thisProp.Value
					printStr = "  |  Value: " & val.ToString
					xmlValueStr = val.ToString
				End If
			Case RDV.AssetPropertyType.Properties
				'sub class is AssetProperties As Set of AssetProperty
				'??Redundant or possibly never occurs?? See AssetPropertyList and connectedproperties
				'printStr = "  |  **AssetPropertyType.Properties"
				'xmlValueStr = "Asset Properties"
			Case RDV.AssetPropertyType.Reference
				'sub class is AssetPropertyReference - no Value property
				'will hold a "connected Property" which will be an embedded Asset
				'printStr = "  |  No Value: Embedded Asset Element"
				'xmlValueStr = "Embedded Asset Element"
			Case RDV.AssetPropertyType.String
				'sub class is AssetPropertyString As String
				Dim thisProp As RDV.AssetPropertyString = TryCast(assetProp, RDV.AssetPropertyString)
				If thisProp IsNot Nothing Then
					Dim val As String = thisProp.Value
					printStr = "  |  Value: " & val.ToString
					xmlValueStr = val.ToString
				End If
			Case RDV.AssetPropertyType.Time
				'sub class is AssetPropertyTime As DateTime
				Dim thisProp As RDV.AssetPropertyTime = TryCast(assetProp, RDV.AssetPropertyTime)
				If thisProp IsNot Nothing Then
					Dim val As DateTime = thisProp.Value
					printStr = "  |  Value: " & val.ToString
					xmlValueStr = val.ToString
				End If
			Case RDV.AssetPropertyType.ULonglong
				'sub class is AssetPropertyUInt64 As ULong
				Dim thisProp As RDV.AssetPropertyUInt64 = TryCast(assetProp, RDV.AssetPropertyUInt64)
				If thisProp IsNot Nothing Then
					Dim val As ULong = thisProp.Value
					printStr = "  |  Value: " & val.ToString
					xmlValueStr = val.ToString
				End If
			Case RDV.AssetPropertyType.Unknown
				'nothing? a catch all? no value to retrieve
		End Select


		writer.WriteLine(prefixStr & "Property Name:  " & assetProp.Name & printStr &
						 "  |  Property Type:  " & assetProp.Type.ToString &
						 "  |  API Class Name: " & assetProp.GetType().Name)

		Dim thisXMLNode As XML.XmlElement = thisXMLDocument.CreateElement("Property")
		thisXMLNode.SetAttribute("Name", assetProp.Name)
		thisXMLNode.SetAttribute("Value", xmlValueStr)
		thisXMLNode.SetAttribute("Type", assetProp.Type.ToString)
		thisXMLNode.SetAttribute("Class", assetProp.GetType().Name)
		parentXmlNode.AppendChild(thisXMLNode)

		'retrieve detachable assets of this property... 
		If assetProp.NumberOfConnectedProperties > 0 Then
			For Each connectedProp As RDV.AssetProperty In assetProp.GetAllConnectedProperties()
				GetAssetPropValues(connectedProp, thisXMLNode, writer, inDent + 1)
			Next
		End If

	End Sub

	Private Sub LogParam(param As RDB.Parameter, doc As RDB.Document, parentXmlNode As XML.XmlElement,
						 logWriter As System.IO.StreamWriter, Optional inDent As Integer = 0)

		logWriter.WriteLine()
		'logWriter.WriteLine(vbTab & "Parameter Name: " & param.ToString)

		Dim prefixStr As String = ""
		If inDent > 0 Then
			For cntr = 1 To inDent
				prefixStr = prefixStr & vbTab
			Next
		End If

		Dim paraDef As RDB.InternalDefinition = param.Definition
		'Too Many Calls to the definition here to try to catch that
		'one Structural parameter that doesn't have a definition so we will
		'put a try... catch... on the calls to this function

		Dim thisXmlNode As XML.XmlElement
		thisXmlNode = thisXMLDocument.CreateElement("Parameter")
		thisXmlNode.SetAttribute("Name", paraDef.Name)
		thisXmlNode.SetAttribute("ElementId", paraDef.Id.ToString)

		'If paraDef IsNot Nothing Then
		'at parameter Definition level
		If paraDef.BuiltInParameter <> RDB.BuiltInParameter.INVALID Then
			logWriter.WriteLine(prefixStr & "Definition IS Built-in: " & paraDef.Id.ToString & " : " & paraDef.Name & " : " & paraDef.GetParameterTypeId().TypeId)
			thisXmlNode.SetAttribute("TypeId", paraDef.GetParameterTypeId().TypeId)
		Else
			logWriter.WriteLine(prefixStr & "Definition Not Built-in: " & paraDef.Id.ToString & " : " & paraDef.Name & " : " & paraDef.GetTypeId().TypeId)
			thisXmlNode.SetAttribute("TypeId", paraDef.GetTypeId().TypeId)
		End If
		parentXmlNode.AppendChild(thisXmlNode)

		Dim propXmlNode As XML.XmlElement

		Dim ftID As RDB.ForgeTypeId
		ftID = paraDef.GetDataType()
		If ftID.Empty = True Then
			logWriter.WriteLine(prefixStr & "Forge Data Type: " & "Empty")
			propXmlNode = thisXMLDocument.CreateElement("Property")
			propXmlNode.SetAttribute("Name", "Forge Data Type")
			propXmlNode.SetAttribute("Value", "Empty")
			thisXmlNode.AppendChild(propXmlNode)
		Else
			logWriter.WriteLine(prefixStr & "Forge Data Type: " & ftID.TypeId)
			propXmlNode = thisXMLDocument.CreateElement("Property")
			propXmlNode.SetAttribute("Name", "Forge Data Type")
			propXmlNode.SetAttribute("Value", ftID.TypeId)
			thisXmlNode.AppendChild(propXmlNode)
			logWriter.WriteLine(prefixStr & "Is Spec : " & RDB.SpecUtils.IsSpec(ftID).ToString)
			propXmlNode = thisXMLDocument.CreateElement("Property")
			propXmlNode.SetAttribute("Name", "Is Spec?")
			propXmlNode.SetAttribute("Value", RDB.SpecUtils.IsSpec(ftID).ToString)
			thisXmlNode.AppendChild(propXmlNode)
			If RDB.SpecUtils.IsSpec(ftID) = True Then
				logWriter.WriteLine(prefixStr & "Spec Display Name: " & RDB.LabelUtils.GetLabelForSpec(ftID))
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Spec Display Name")
				propXmlNode.SetAttribute("Value", RDB.LabelUtils.GetLabelForSpec(ftID))
				thisXmlNode.AppendChild(propXmlNode)
				logWriter.WriteLine(prefixStr & "Is Measurable Spec? : " & RDB.UnitUtils.IsMeasurableSpec(ftID).ToString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Is Measurable Spec?")
				propXmlNode.SetAttribute("Value", RDB.UnitUtils.IsMeasurableSpec(ftID).ToString)
				thisXmlNode.AppendChild(propXmlNode)
				If RDB.UnitUtils.IsMeasurableSpec(ftID) = True Then
					logWriter.WriteLine(prefixStr & "Forge Unit Type ID : " & param.GetUnitTypeId().TypeId)
					propXmlNode = thisXMLDocument.CreateElement("Property")
					propXmlNode.SetAttribute("Name", "Forge Unit Type ID")
					propXmlNode.SetAttribute("Value", param.GetUnitTypeId().TypeId)
					thisXmlNode.AppendChild(propXmlNode)
					logWriter.WriteLine(prefixStr & "Unit Display Name: " & RDB.LabelUtils.GetLabelForUnit(param.GetUnitTypeId()))
					propXmlNode = thisXMLDocument.CreateElement("Property")
					propXmlNode.SetAttribute("Name", "Unit Display Name")
					propXmlNode.SetAttribute("Value", RDB.LabelUtils.GetLabelForUnit(param.GetUnitTypeId()))
					thisXmlNode.AppendChild(propXmlNode)
					'For Each testunit As RDB.ForgeTypeId In RDB.UnitUtils.GetValidUnits(ftID)
					'	logWriter.WriteLine(vbTab & vbTab & "Valid Unit Type: " & testunit.TypeId)
					'	logWriter.WriteLine(vbTab & vbTab & "Unit Display Name: " & RDB.LabelUtils.GetLabelForUnit(testunit))
					'Next
				End If
			Else
				'is Category (only applies to family types)? If so, what category
				If RDB.Category.IsBuiltInCategory(ftID) = True Then
					logWriter.WriteLine(prefixStr & "Is Category : " & RDB.Category.IsBuiltInCategory(ftID).ToString)
					propXmlNode = thisXMLDocument.CreateElement("Property")
					propXmlNode.SetAttribute("Name", "Is Category?")
					propXmlNode.SetAttribute("Value", RDB.Category.IsBuiltInCategory(ftID).ToString)
					thisXmlNode.AppendChild(propXmlNode)
					Dim valBIC As RDB.BuiltInCategory = RDB.Category.GetBuiltInCategory(ftID)
					logWriter.WriteLine(prefixStr & "Builtin Category: " & valBIC.ToString & " : " & RDB.LabelUtils.GetLabelFor(valBIC))
					propXmlNode = thisXMLDocument.CreateElement("Property")
					propXmlNode.SetAttribute("Name", "Built-In Category")
					propXmlNode.SetAttribute("Value", valBIC.ToString & " : " & RDB.LabelUtils.GetLabelFor(valBIC))
					thisXmlNode.AppendChild(propXmlNode)
				Else
					logWriter.WriteLine(prefixStr & "NOT Builtin Category: ")
					propXmlNode = thisXMLDocument.CreateElement("Property")
					propXmlNode.SetAttribute("Name", "Built-In Category")
					propXmlNode.SetAttribute("Value", "No")
					thisXmlNode.AppendChild(propXmlNode)
				End If
			End If
		End If

		logWriter.WriteLine(prefixStr & "Definition Group: " & paraDef.ParameterGroup & " : " & paraDef.ParameterGroup.ToString & " : " &
							RDB.LabelUtils.GetLabelFor(paraDef.ParameterGroup))
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "Parameter Group")
		propXmlNode.SetAttribute("Value", paraDef.ParameterGroup & " : " & paraDef.ParameterGroup.ToString)
		thisXmlNode.AppendChild(propXmlNode)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "Group Name")
		propXmlNode.SetAttribute("Value", RDB.LabelUtils.GetLabelFor(paraDef.ParameterGroup))
		thisXmlNode.AppendChild(propXmlNode)
		logWriter.WriteLine(prefixStr & "Is User Visible: " & paraDef.Visible.ToString)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "Is User Visible")
		propXmlNode.SetAttribute("Value", paraDef.Visible.ToString)
		thisXmlNode.AppendChild(propXmlNode)
		logWriter.WriteLine(prefixStr & "Can Vary in Groups: " & paraDef.VariesAcrossGroups.ToString)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "Can Vary in Groups")
		propXmlNode.SetAttribute("Value", paraDef.VariesAcrossGroups.ToString)
		thisXmlNode.AppendChild(propXmlNode)
		'Else
		'	logWriter.WriteLine(vbTab & "!! No Parameter Definition !!")
		'End If

		'at Parameter Instance Level
		'Tells us where it lives:
		'(Global, Family, Project, External, BuiltIn [by not indicating Parameter Def ID and having it's own description instead])
		If param.IsShared = True Then
			logWriter.WriteLine(prefixStr & "Shared Param: " & param.GUID.ToString)
			propXmlNode = thisXMLDocument.CreateElement("Property")
			propXmlNode.SetAttribute("Name", "Shared Param")
			propXmlNode.SetAttribute("Value", param.GUID.ToString)
			thisXmlNode.AppendChild(propXmlNode)
		Else
			If param.Id.IntegerValue < 0 Then
				Dim bip As RDB.BuiltInParameter = DirectCast(param.Id.IntegerValue, RDB.BuiltInParameter)
				logWriter.WriteLine(prefixStr & "Built-in Param: " & param.Id.ToString & " : " & bip.ToString & " : " &
									RDB.LabelUtils.GetLabelForBuiltInParameter(param.GetTypeId()))
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Built-in Param")
				propXmlNode.SetAttribute("Value", param.Id.ToString & " : " & bip.ToString)
				thisXmlNode.AppendChild(propXmlNode)
			Else
				logWriter.WriteLine(prefixStr & "Local Param: " & param.Id.IntegerValue)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Local Param Id")
				propXmlNode.SetAttribute("Value", param.Id.IntegerValue)
				thisXmlNode.AppendChild(propXmlNode)
			End If
		End If

		logWriter.WriteLine(prefixStr & "Is Read Only? : " & param.IsReadOnly.ToString)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "Is Read Only?")
		propXmlNode.SetAttribute("Value", param.IsReadOnly.ToString)
		thisXmlNode.AppendChild(propXmlNode)
		logWriter.WriteLine(prefixStr & "Is User Modifiable? : " & param.UserModifiable.ToString)
		propXmlNode = thisXMLDocument.CreateElement("Property")
		propXmlNode.SetAttribute("Name", "Is User Modifiable?")
		propXmlNode.SetAttribute("Value", param.UserModifiable.ToString)
		thisXmlNode.AppendChild(propXmlNode)

		Select Case param.StorageType
			Case RDB.StorageType.String
				logWriter.WriteLine(prefixStr & "String Value: " & param.AsString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "String Value")
				propXmlNode.SetAttribute("Value", param.AsString)
				thisXmlNode.AppendChild(propXmlNode)
			Case RDB.StorageType.ElementId
				logWriter.WriteLine(prefixStr & "ElementID Value: " & param.AsElementId.ToString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "ElementID Value")
				propXmlNode.SetAttribute("Value", param.AsElementId.ToString)
				thisXmlNode.AppendChild(propXmlNode)
				Dim refElem As RDB.Element = doc.GetElement(param.AsElementId())
				If refElem IsNot Nothing Then
					logWriter.WriteLine(prefixStr & "Element Class Type:  " & refElem.GetType().ToString)
					propXmlNode = thisXMLDocument.CreateElement("Property")
					propXmlNode.SetAttribute("Name", "Element Class Type")
					propXmlNode.SetAttribute("Value", refElem.GetType().ToString)
					thisXmlNode.AppendChild(propXmlNode)
					Try
						logWriter.WriteLine(prefixStr & "Element Category:  " & refElem.Category.Name)
						propXmlNode = thisXMLDocument.CreateElement("Property")
						propXmlNode.SetAttribute("Name", "Element Category")
						propXmlNode.SetAttribute("Value", refElem.Category.Name)
						thisXmlNode.AppendChild(propXmlNode)
					Catch ex As Exception
						logWriter.WriteLine(prefixStr & "Element Category is NULL:  ")
						propXmlNode = thisXMLDocument.CreateElement("Property")
						propXmlNode.SetAttribute("Name", "Element Category")
						propXmlNode.SetAttribute("Value", "NULL")
						thisXmlNode.AppendChild(propXmlNode)
					End Try
					logWriter.WriteLine(prefixStr & "Element Name:  " & refElem.Name)
					propXmlNode = thisXMLDocument.CreateElement("Property")
					propXmlNode.SetAttribute("Name", "Element Name")
					propXmlNode.SetAttribute("Value", refElem.Name)
					thisXmlNode.AppendChild(propXmlNode)
				Else
					logWriter.WriteLine(prefixStr & "Element Reference Not Assigned")
					propXmlNode = thisXMLDocument.CreateElement("Property")
					propXmlNode.SetAttribute("Name", "Element Reference")
					propXmlNode.SetAttribute("Value", "NULL")
					thisXmlNode.AppendChild(propXmlNode)
				End If
			Case RDB.StorageType.Double
				logWriter.WriteLine(prefixStr & "Double Value: " & param.AsDouble.ToString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Double Value")
				propXmlNode.SetAttribute("Value", param.AsDouble.ToString)
				thisXmlNode.AppendChild(propXmlNode)
				logWriter.WriteLine(prefixStr & "Value String: " & param.AsValueString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Value String")
				propXmlNode.SetAttribute("Value", param.AsValueString)
				thisXmlNode.AppendChild(propXmlNode)
			Case RDB.StorageType.Integer
				logWriter.WriteLine(prefixStr & "Integer Value: " & param.AsInteger.ToString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Integer Value")
				propXmlNode.SetAttribute("Value", param.AsInteger.ToString)
				thisXmlNode.AppendChild(propXmlNode)
				logWriter.WriteLine(prefixStr & "Value String: " & param.AsValueString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Value String")
				propXmlNode.SetAttribute("Value", param.AsValueString)
				thisXmlNode.AppendChild(propXmlNode)
			Case Else
				logWriter.WriteLine(prefixStr & "Value String: " & param.AsValueString)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Value String")
				propXmlNode.SetAttribute("Value", param.AsValueString)
				thisXmlNode.AppendChild(propXmlNode)
		End Select

		If param.CanBeAssociatedWithGlobalParameters = True Then
			If param.GetAssociatedGlobalParameter() <> RDB.ElementId.InvalidElementId Then
				logWriter.WriteLine(prefixStr & "Associated to Global Parameter:  " & doc.GetElement(param.GetAssociatedGlobalParameter()).Name)
				propXmlNode = thisXMLDocument.CreateElement("Property")
				propXmlNode.SetAttribute("Name", "Associated to Global Parameter")
				propXmlNode.SetAttribute("Value", doc.GetElement(param.GetAssociatedGlobalParameter()).Name)
				thisXmlNode.AppendChild(propXmlNode)
			End If
		End If

	End Sub

	Private Function GenerateName(strVal As String, list As IList(Of String)) As String

		Dim numb As Integer = 1
		Dim numbStr As String = "001"
		Dim testStr As String = numbStr & strVal

		Do While list.Contains(testStr) = True
			If numb < 100 Then
				testStr = Mid((1000 + numb).ToString(), 2) & strVal
			Else
				testStr = numb.ToString() & strVal
			End If

			numb = numb + 1
		Loop

		Return testStr

	End Function


End Class
