#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
using System.IO;
using System.Linq;
#endregion

namespace RVT_UnitTypes
{
    [Transaction(TransactionMode.Manual)]
    public class Command : IExternalCommand
    {

        public Result Execute(
          ExternalCommandData commandData,
          ref string message,
          ElementSet elements)
        {
            UIApplication uiapp = commandData.Application;
            UIDocument uidoc = uiapp.ActiveUIDocument;


            CreateInHouseUnitData(uidoc.Document);

            return Result.Succeeded;
        }


        public void CreateInHouseUnitData(Document doc)
        {

            

            string myDocs = Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments);
            string fileName = myDocs + "\\" + "SharedParameterFactory.txt";
            doc.Application.SharedParametersFilename = fileName;

            StreamWriter definitionFileStream = new StreamWriter(doc.Application.SharedParametersFilename, false);
            definitionFileStream.Close();
            DefinitionFile definitionFile = doc.Application.OpenSharedParameterFile();

            List<NVProperty> propsToBeCreated = new List<RVT_UnitTypes.NVProperty>();
            if (definitionFile == null)
                return;

            //get all parametertypes
            Array param_types = Enum.GetValues(typeof(ParameterType));
            foreach (ParameterType pt in param_types)
            {
                if (pt == ParameterType.Invalid)
                    continue;

                if (pt == ParameterType.FamilyType)
                    continue;

                NVProperty prop = new RVT_UnitTypes.NVProperty();
                prop.paramType = pt;
                prop.Name = pt.ToString();
                prop.Description = LabelUtils.GetLabelFor(pt);
                propsToBeCreated.Add(prop);
            }

           

            //create a shared parameter file containing all parameter types
            InhouseParameterFactory(definitionFile, propsToBeCreated);

            List<ExternalDefinition> definitions = null;
            List<Parameter> double_parameters = new List<Parameter>();

            //attach each created parameter to ProjectInformation
            //for storagetype double set value of 1 using internal units
            Element elem = doc.ProjectInformation;

            using (Transaction t = new Transaction(doc, "factory"))
            {
                t.Start(t.GetName());
                definitions = AttachSharedParameters(doc, definitionFile);

                foreach (ExternalDefinition def in definitions)
                {
                    if (def.ParameterType == ParameterType.FixtureUnit)
                        continue;

                    Parameter p = elem.get_Parameter(def.GUID);

                    if (p.StorageType == StorageType.Double)
                    {
                        double d = UnitUtils.ConvertToInternalUnits(1, p.DisplayUnitType);
                        p.Set(d);
                        double_parameters.Add(p);

                    }
                }

                t.Commit();
            }

           
            //create dictionary of DisplayUnitType, abbrevation by reading of the values

            string file_cs = myDocs + "\\" + "UnitAbbrevHandler.cs";
            StreamWriter filestream_cs = new StreamWriter(file_cs, false);

            filestream_cs.WriteLine("using System.Collections.Generic;");
            filestream_cs.WriteLine("using Autodesk.Revit.DB;");
            filestream_cs.WriteLine(System.Environment.NewLine);

            filestream_cs.WriteLine("namespace RVT_UnitTypes");
            filestream_cs.WriteLine("{");

            filestream_cs.WriteLine("public class UnitAbbrevs");
            filestream_cs.WriteLine("{");

            filestream_cs.WriteLine("public Dictionary<DisplayUnitType, string> UnitAbbrevations()");
            filestream_cs.WriteLine("{");

            filestream_cs.WriteLine("Dictionary<DisplayUnitType, string> abbrevations = new Dictionary<DisplayUnitType, string>();");

            Units units = doc.GetUnits();
            List<DisplayUnitType> already_done = new List<DisplayUnitType>();
            foreach (Parameter p_ in double_parameters)
            {

                UnitType ut = p_.Definition.UnitType;
                if (!Units.IsModifiableUnitType(ut))
                {
                    //these will be UT_Number, UT_Electrical_TemperatureDifference, UT_Piping_TemperatureDifference and UT_HVAC_TemperatureDifference.
                    Debug.Print("Non-modifiable UnitType: " + ut.ToString());
                    continue;
                }
                    

                FormatOptions fOpt = units.GetFormatOptions(ut);
                fOpt.UseDefault = false;

                IList<DisplayUnitType> display_units = UnitUtils.GetValidDisplayUnits(ut);
                if (display_units.Count == 0)
                {
                    //nothing enters here
                    Debug.Print("No DisplayUnits for UnitType: " + ut.ToString());
                    continue;
                }
                   

                foreach (DisplayUnitType dut in display_units)
                {
                    if (dut == DisplayUnitType.DUT_CUSTOM)
                        continue;

                    if (already_done.Contains(dut))
                        continue;

                    fOpt.DisplayUnits = dut;

                    IList<UnitSymbolType> lst_ust = fOpt.GetValidUnitSymbols();
                    UnitSymbolType ust_ = fOpt.UnitSymbol;

                    //we want a abbrevivation (if any) so dont use "none".                    
                    foreach (UnitSymbolType ust in lst_ust)
                    {
                        if (ust == UnitSymbolType.UST_NONE)
                            continue;

                        fOpt.UnitSymbol = ust;
                        break;
                    }

                    string value = p_.AsValueString(fOpt);

                    string[] sa = value.Split(' ');
                    if (sa.Count() == 2) //do we need slopes ?
                    {
                        Debug.Print(p_.Definition.ParameterType.ToString() + "," + dut.ToString() + ", " + sa[1]);
                        filestream_cs.WriteLine("abbrevations.Add(DisplayUnitType." + dut.ToString() +", \"" + sa[1] + "\");");
                    }
                    else
                    {
                        //what might land here ?
                        Debug.Print("Not unit-based? " + value + "___" + p_.Definition.ParameterType.ToString());
                        
                    }

                    already_done.Add(dut);
                }
            }
            filestream_cs.WriteLine("return abbrevations;");
            filestream_cs.WriteLine("}");
            filestream_cs.WriteLine("}");
            filestream_cs.WriteLine("}");

            filestream_cs.Close();

            string s = "Created shared parameterfile and dictionary .cs file for abbrevations.\n(The parameters are attached to ProjectInformation.)";
            TaskDialog.Show("ParameterFactory", s);
        }



        public void InhouseParameterFactory(DefinitionFile definitionFile, List<NVProperty> properties)
        {

            DefinitionGroup defGroup = definitionFile.Groups.Create("ParameterFactory");

            foreach (NVProperty property in properties)
            {
                Definition definition = null;

                if (property.paramType == ParameterType.Invalid)
                {
                    Debug.WriteLine(property.Name + " has an invalid ParameterType");
                    continue;
                }

                ExternalDefinitionCreationOptions option = new ExternalDefinitionCreationOptions(property.Name, property.paramType);

                option.GUID = Guid.NewGuid();
                option.Description = property.Description;

                definition = defGroup.Definitions.Create(option);
            }

        }

        public List<ExternalDefinition> AttachSharedParameters(Document rvtDoc, DefinitionFile defFile)
        {
            BindingMap map = rvtDoc.ParameterBindings;
            DefinitionGroup dg = defFile.Groups.get_Item("ParameterFactory");
            CategorySet paramCategories = new CategorySet();
            paramCategories.Insert(Category.GetCategory(rvtDoc, BuiltInCategory.OST_ProjectInformation));
            List<ExternalDefinition> parameters = new List<ExternalDefinition>();
            foreach (ExternalDefinition def in dg.Definitions)
            {
                Autodesk.Revit.DB.Binding binding = rvtDoc.Application.Create.NewInstanceBinding(paramCategories);
                map.Insert(def, binding, BuiltInParameterGroup.PG_GENERAL);
                parameters.Add(def);
            }

            return parameters;
        }

    }

    public class NVProperty
    {
        public string Name;
        public string Description;
        public ParameterType paramType;
    }

}
