﻿using System;
using System.Collections.Generic;

using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
using Autodesk.Revit.DB;
using Autodesk.Revit.Attributes;
using System.Diagnostics;
using System.IO;
using Autodesk.Revit.ApplicationServices;

namespace TempDemoSample
{

    [TransactionAttribute(TransactionMode.Manual)]

    public class AnalyzeOpenings : IExternalCommand
    {

        public Result Execute(ExternalCommandData commandData,
                              ref string message,
                              ElementSet elements)
        {
            UIApplication rvtUIApp = commandData.Application;
            UIDocument rvtUIDoc = rvtUIApp.ActiveUIDocument;

            Application rvtApp = rvtUIApp.Application;
            Document rvtDoc = rvtUIDoc.Document;


            ICollection<ElementId> idElems = new List<ElementId>();

            try
            {

                if (rvtUIDoc.Selection.GetElementIds().Count > 0)
                    idElems = GetPickFirsts(rvtDoc);
                else
                    idElems = GetAllElements(rvtDoc);

                rvtApp.DocumentChanged += new EventHandler<Autodesk.Revit.DB.Events.DocumentChangedEventArgs>(application_DocumentChanged);

                Dictionary<ElementId, Solid> openingData = new Dictionary<ElementId, Solid>();

                using (TransactionGroup transGroup = new TransactionGroup(rvtDoc))
                {
                    transGroup.Start("VisibleUndoName");
                    openingData = GetOpeningData(rvtDoc, idElems);

                    transGroup.RollBack();                    
                }

                rvtApp.DocumentChanged -= new EventHandler<Autodesk.Revit.DB.Events.DocumentChangedEventArgs>(application_DocumentChanged);

                Options opt = new Options();
                opt.IncludeNonVisibleObjects = true;
                string s = string.Empty;

                using (Transaction t1 = new Transaction(rvtDoc, "_"))
                {
                    t1.Start("vizualise opening geometry");
                    Creator creator = new TempDemoSample.Creator(rvtDoc);
                    
                    foreach (KeyValuePair<ElementId, Solid> pair in openingData)
                    {
                        FamilyInstance fi = rvtDoc.GetElement(pair.Key) as FamilyInstance;
                        GeometryElement fi_geom = fi.get_Geometry(opt);
                        List<Curve> openingCurves = GetOpeningCurves(fi_geom);

                        foreach (Curve curve in openingCurves)
                            creator.CreateModelCurves(curve);

                        //move out 1m for clarity
                        Transform tr = Transform.CreateTranslation(fi.FacingOrientation * Utils.MeterToFeet(1));
                        Solid solid = SolidUtils.CreateTransformed(pair.Value,tr);
                        creator.CreateDirectShape(rvtDoc, solid, "demolish_fill");

                        s += "Opening: " + fi.Symbol.Name + "("+ fi.Id + ") has a volume of " + pair.Value.Volume + System.Environment.NewLine;
                    }

                    t1.Commit();
                }

                TaskDialog.Show("OpeningData", s);

            }

            catch (Autodesk.Revit.Exceptions.OperationCanceledException)
            {

                return Result.Cancelled;
            }

            catch (Exception ex)
            {
                message = ex.Message;

                return Result.Failed;
            }

            return Autodesk.Revit.UI.Result.Succeeded;
        }

        public Dictionary<ElementId, Solid> GetOpeningData(Document doc, ICollection<ElementId> elementIds)
        {

            Dictionary<ElementId, Solid> openingData = new Dictionary<ElementId, Solid>();

            Autodesk.Revit.DB.Options geoOptCompRef = doc.Application.Create.NewGeometryOptions();
            geoOptCompRef.DetailLevel = ViewDetailLevel.Medium;

            idDemolishInfill = new List<ElementId>();

            using (Transaction t = new Transaction(doc, "temp"))
            {
                foreach (ElementId id in elementIds)
                {

                    idDemolishInfill.Clear();
                    FamilyInstance fi = doc.GetElement(id) as FamilyInstance;

                    if (fi.GroupId != ElementId.InvalidElementId)
                        continue;
                    t.Start("demolish");
                    fi.DemolishedPhaseId = fi.CreatedPhaseId;
                    t.Commit();

                    //Revit has now created an "infill" wall in the opening
                    //and the idDemolishInfill collection now contains the Id of that wall.
                    Wall wall = null;
                    foreach (ElementId idWall in idDemolishInfill)
                    {
                        Element e = doc.GetElement(idWall) as Element;
                        if (e is Wall)
                            wall = e as Wall;
                    }

                    if (wall == null)
                        continue;
                    
                    WallKind wallKind = wall.WallType.Kind;
                    if (wallKind == WallKind.Curtain || wallKind == WallKind.Unknown)
                        continue;

                    Solid solid = null;

                    GeometryElement geomWall = wall.get_Geometry(geoOptCompRef) as GeometryElement;
                    List<Solid> solids = GetElementSolids(geomWall);

                    if (solids.Count != 1) //it really cant be anything else than 1 in this case
                        continue;

                    solid = solids[0];

                    //the original solid doesnt return edges for some reason, if you need them generate those through copy
                    Solid solidWithEdges = Autodesk.Revit.DB.SolidUtils.CreateTransformed(solid, Transform.Identity);
                    openingData.Add(fi.Id, solidWithEdges);

                }
            }

            return openingData;
        }


        private List<Curve> GetOpeningCurves(GeometryElement geomElem)
        {
            List<Curve> curvesInFamily = new List<Curve>();
        
            GeometryInstance inst = null;

            foreach (GeometryObject geomObj in geomElem)
            {
                inst = geomObj as GeometryInstance;
                if (inst == null)
                    continue;

                GeometryElement ge = inst.GetSymbolGeometry() as GeometryElement;

                foreach (GeometryObject o in ge)
                {
                    Curve curves = o as Curve;
                    if (curves == null)
                        continue;

                    //the lines we are after are not physical lines and so they dont have a graphic style
                    if (curves.GraphicsStyleId != ElementId.InvalidElementId)
                        continue;

                    curvesInFamily.Add(curves);

                }
            }

            //offset out for clarity
            //in using the non-transformed SymbolGeometry we are inside the family editor - sort of.
            //and being a door or window family we know the wall inside the family is always horizontal
            //therefore XYZ.BasisY will always be perpendicular to the wall
            double offset_dist = Utils.MeterToFeet(.5);
            Transform transformOut = Transform.CreateTranslation(XYZ.BasisY * offset_dist);
            curvesInFamily = GetTransformedCurves(curvesInFamily, transformOut);

            //now transform from family to project
            return GetTransformedCurves(curvesInFamily, inst.Transform);           
            
        }

        private List<Curve> GetTransformedCurves(List<Curve> curves, Transform transform)
        {
            List<Curve> curve_transformed = new List<Curve>();
            foreach(Curve c in curves)
                curve_transformed.Add(c.CreateTransformed(transform));

            return curve_transformed;
        }

        private List<Arc> GetArcsFromSymbol(GeometryElement geomElem)
        {

            List<Arc> arcsInFamily = new List<Arc>();

            GeometryInstance inst = null;

            foreach (GeometryObject geomObj in geomElem)
            {
                inst = geomObj as GeometryInstance;
                if (inst == null)
                    continue;

                GeometryElement ge = inst.GetSymbolGeometry() as GeometryElement;

                foreach (GeometryObject o in ge)
                {

                    Arc arc = o as Arc;
                    if (arc == null)
                        continue;

                    //the lines we are after are not physical lines and so they dont have a graphic style
                    if (arc.GraphicsStyleId != ElementId.InvalidElementId)
                        continue;

                    arcsInFamily.Add(arc);
                }
            }

            return arcsInFamily;
        }



        public ICollection<ElementId> GetPickFirsts(Document doc)
        {

            UIDocument uiDoc = new UIDocument(doc);
            Selection selPickFirst = uiDoc.Selection;
            ICollection<ElementId> pickFirstIds = selPickFirst.GetElementIds();
            ICollection<ElementId> validPickFirstIds = new List<ElementId>();
            ICollection<ElementId> allowedPickFirstIds_ = GetElementFilter().GetCategoryIds();

            foreach (ElementId id in pickFirstIds)
            {
                if (allowedPickFirstIds_.Contains(id))
                {
                    validPickFirstIds.Add(id);
                }
            }

            return validPickFirstIds;
        }

        public ICollection<ElementId> GetAllElements(Document doc)
        {
            FilteredElementCollector clElems = new FilteredElementCollector(doc);
            clElems.WherePasses(GetElementFilter());
            clElems.WhereElementIsNotElementType();
            return clElems.ToElementIds();
        }
    
        private static ElementMulticategoryFilter GetElementFilter()
        {

            List<BuiltInCategory> builtInCats = new List<BuiltInCategory>();

            builtInCats.Add(BuiltInCategory.OST_Doors);
            builtInCats.Add(BuiltInCategory.OST_Windows);
            //builtInHostCats.Add(BuiltInCategory.OST_Floors);
            //builtInHostCats.Add(BuiltInCategory.OST_Conduit);
            //builtInHostCats.Add(BuiltInCategory.OST_PipeCurves);
            //builtInHostCats.Add(BuiltInCategory.OST_DuctCurves);

            return new ElementMulticategoryFilter(builtInCats);
        }

        private static List<Solid> GetElementSolids(GeometryElement geomElem)
        {

            List<Solid> solidsInFamily = new List<Solid>();
            Solid geomSolid;

            foreach (GeometryObject geomObj in geomElem)
            {
                GeometryInstance geomInst = geomObj as GeometryInstance;

                if (null != geomInst) //second level solids could be a nested FamilyInstance
                {
                    GeometryElement instGeomElem = geomInst.GetInstanceGeometry();

                    foreach (GeometryObject _geomObj in instGeomElem)
                    {
                        geomSolid = _geomObj as Solid;

                        if (geomSolid != null)
                            solidsInFamily.Add(geomSolid);

                        //if this was a familyinstance we might get zero or negative volumes from voids created in the family                          
                    }
                }

                else // try first level solids likely a system family
                {
                    geomSolid = geomObj as Solid;

                    if (geomSolid != null)
                    solidsInFamily.Add(geomSolid);
                }
            }

            return solidsInFamily;
        }

        private static ICollection<ElementId> idDemolishInfill
        {
            get;
            set;
        }

        private static void application_DocumentChanged(object sender, Autodesk.Revit.DB.Events.DocumentChangedEventArgs args)
        {
            idDemolishInfill = args.GetAddedElementIds();
        }
    }

    
}
