﻿using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using ClipperLib;
using System;
using System.Collections.Generic;
using System.Linq;
using Polygon = System.Collections.Generic.List<ClipperLib.IntPoint>;
using Polygons = System.Collections.Generic.List<System.Collections.Generic.List<ClipperLib.IntPoint>>;

namespace RvtClipper
{
    public partial class UIOptions : System.Windows.Forms.Form
    {
        public ClipType clipType { get; set; }

        public PolyFillType polyFillTypeSubj { get; set; }

        public PolyFillType polyFillTypeClip { get; set; }

        bool isInitializing { get; set; } = false;

        Polygons intersection = new Polygons();

        Clipper clipper { get; set; }

        FilledRegion subject { get; set; }

        List<Element> elemsToReset { get; set; } = new List<Element>();

        FilledRegionType regionTypeForResult { get; set; }
        UIDocument uidoc { get; set; }
        Polygons clipitem { get; set; }

        public UIOptions(UIDocument _uidoc, Clipper _clipper, Polygons _clip)
        {
            InitializeComponent();
            uidoc = _uidoc;
            clipper = _clipper;
            clipitem = _clip;

            isInitializing = true;

            List<FilledRegionType> allRegions = new FilteredElementCollector(uidoc.Document)
                .WhereElementIsElementType()
                .OfClass(typeof(FilledRegionType))
                .Cast<FilledRegionType>()
                .ToList();

            var regions = allRegions.Where(t => t.Name == "Solid_Black");
            if (regions.Count() == 1)
                regionTypeForResult = regions.First();
            else
                regionTypeForResult = allRegions.First();


            comboBoxClipType.DataSource = Enum.GetNames(typeof(ClipType));
            comboBoxClipType.SelectedIndex = 0;

            comboBoxPolyFillTypeSubj.DataSource = Enum.GetNames(typeof(PolyFillType));
            comboBoxPolyFillTypeSubj.SelectedIndex = 0;

            comboBoxPolyFillTypeClip.DataSource = Enum.GetNames(typeof(PolyFillType));
            comboBoxPolyFillTypeClip.SelectedIndex = 0;
            
            clipper.Execute(clipType, intersection, polyFillTypeSubj, polyFillTypeClip);
            CreateIntersectRegion();

            isInitializing = false;

        }

        private void CreateIntersectRegion()
        {
            if (intersection.Count > 0)
            {
                // Convert the Clipper intersection to IList<CurveLoop>
                IList<CurveLoop> curveLoops = ConvertToCurveloops(intersection);
                if(curveLoops.Count == 0)
                {
                    TaskDialog.Show("CurveLoop", "No CurveLoop converted");
                    return;
                }

                using (Transaction tx = new Transaction(uidoc.Document))
                {
                    tx.Start("Create Intersection Region");

                    ResetElements();

                    try
                    {
                        subject = FilledRegion.Create(uidoc.Document, regionTypeForResult.Id, uidoc.ActiveGraphicalView.Id, curveLoops);
                        uidoc.Document.Regenerate();
                        elemsToReset.Add(subject);
                    }
                    catch
                    {

                        foreach (CurveLoop loop in curveLoops)
                        {
                            elemsToReset.AddRange(Util.CreateModelLine(uidoc.Document, loop, false));
                        }

                        TaskDialog.Show("Boundaries", "Boundaries are probably self-intersecting");
                    }

                    tx.Commit();
                }
            }
            else
            {
                TaskDialog.Show("Intersection", "No intersection found");
                using (Transaction tx = new Transaction(uidoc.Document, "reset"))
                {
                    tx.Start();

                    ResetElements();
                    tx.Commit();
                }
            }
        }

        private void ResetElements()
        {
            if (elemsToReset.Count > 0)
            {
                List<Element> valid = elemsToReset.Where(t => t.IsValidObject).ToList();

                if (valid.Count > 0)
                    uidoc.Document.Delete(valid.Select(t => t.Id).ToList());

            }

            elemsToReset = new List<Element>();
        }

        private IList<CurveLoop> ConvertToCurveloops(Polygons intersection)
        {
            IList<CurveLoop> curveLoops = new List<CurveLoop>();
            foreach (Polygon poly in intersection)
            {
                IList<Curve> curves = new List<Curve>();

                IntPoint? p0 = null; // first
                IntPoint? p = null; // previous

                foreach (IntPoint q in poly)
                {
                    if (null == p0)
                    {
                        p0 = q;
                    }
                    if (null != p)
                    {
                        curves.Add(Line.CreateBound(Util.GetXyzPoint(p.Value), Util.GetXyzPoint(q)));
                    }
                    p = q;
                }

                //close the loop
                curves.Add(Line.CreateBound(Util.GetXyzPoint(p.Value), Util.GetXyzPoint(p0.Value)));

                CurveLoop loop = CurveLoop.Create(curves);
                curveLoops.Add(loop);
            }

            return curveLoops;
        }

        private void comboBoxClipType_SelectedIndexChanged(object sender, EventArgs e)
        {

            clipType = (ClipType)Enum.Parse(typeof(ClipType), comboBoxClipType.Text);

            if (isInitializing)
                return;

            clipper.Execute(clipType, intersection, polyFillTypeSubj, polyFillTypeClip);
            CreateIntersectRegion();
        }

        private void comboBoxPolyFillType_SelectedIndexChanged(object sender, EventArgs e)
        {
            polyFillTypeSubj = (PolyFillType)Enum.Parse(typeof(PolyFillType), comboBoxPolyFillTypeSubj.Text);

            if (isInitializing)
                return;

            clipper.Execute(clipType, intersection, polyFillTypeSubj, polyFillTypeClip);
            CreateIntersectRegion();
        }

        private void comboBoxPolyFillTypeClip_SelectedIndexChanged(object sender, EventArgs e)
        {

            polyFillTypeClip = (PolyFillType)Enum.Parse(typeof(PolyFillType), comboBoxPolyFillTypeClip.Text);

            if (isInitializing)
                return;

            clipper.Execute(clipType, intersection, polyFillTypeSubj, polyFillTypeClip);
            CreateIntersectRegion();
        }


    }
}
