﻿using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
using ClipperLib;
using Polygon = System.Collections.Generic.List<ClipperLib.IntPoint>;
using Polygons = System.Collections.Generic.List<System.Collections.Generic.List<ClipperLib.IntPoint>>;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;


namespace RvtClipper
{


    public class Util
    {


        public static List<ModelCurve> CreateModelLine(Document doc, CurveLoop loop, bool useTransaction)
        {
            List<ModelCurve> curves = new List<ModelCurve>();
            foreach (Curve curve in loop)
            {
                ModelCurve mc = CreateModelLine(doc, (Line)curve, useTransaction);
                if (mc != null)
                    curves.Add(mc);

            }

            return curves;
        }

        public static ModelCurve CreateModelLine(Document doc, Line line, bool useTransaction)
        {
            XYZ p = line.GetEndPoint(0);
            XYZ q = line.GetEndPoint(1);


            if (p.DistanceTo(q) < doc.Application.ShortCurveTolerance)
                return null;

            XYZ v = q - p;

            double dxy = Math.Abs(v.X) + Math.Abs(v.Y);

            XYZ w = (dxy > _eps)
              ? XYZ.BasisZ
              : XYZ.BasisY;

            XYZ norm = v.CrossProduct(w).Normalize();

            Plane plane = Plane.CreateByNormalAndOrigin(norm, p);
            ModelCurve curve = null;
            using (Transaction t = new Transaction(doc, "debug_line"))
            {
                if (useTransaction)
                    t.Start();

                SketchPlane sketchPlane = SketchPlane.Create(doc, plane);

                curve = doc.IsFamilyDocument
                  ? doc.FamilyCreate.NewModelCurve(line, sketchPlane)
                  : doc.Create.NewModelCurve(line, sketchPlane);



                if (t.HasStarted())
                    t.Commit();
            }

            return curve;
        }


        /// <summary>
        /// Retrieve the boundary loops of the given sketch 
        /// </summary>
        public static Polygons GetBoundaryLoops(Sketch sketch)
        {
            int n;
            Polygons polys = null;

            CurveArrArray loops = sketch.Profile;

            n = loops.Size;
            polys = new Polygons(n);

            foreach (CurveArray loop in loops)
            {
                n = loop.Size;
                Polygon poly = new Polygon(n);

                foreach (Curve edge in loop)
                {
                    IList<XYZ> pts = edge.Tessellate();

                    n = pts.Count;

                    foreach (XYZ p in pts)
                    {
                        poly.Add(Util.GetIntPoint(p));
                    }
                }
                polys.Add(poly);
            }

            return polys;
        }


        //This method only returns the first Sketch it encounters
        //for more advanced geometry some simple elimination would be needed
        //such as when a sketchbased opening is connected to a floor
        public static Sketch GetSketch(UIDocument uidoc, Element e = null)
        {
            ElementClassFilter filter = new ElementClassFilter(typeof(Sketch));
            Reference rf = null;
            Sketch sketch = null;

            if(e == null)
            {
                try
                {
                    rf = uidoc.Selection.PickObject(ObjectType.Element, "Pick a sketchbased element");
                }
                catch
                {
                    return sketch;
                }

                e = uidoc.Document.GetElement(rf);
            }
         
            IList<ElementId> sketches = e.GetDependentElements(filter);

            if (sketches.Count == 1)
            {
                sketch = (Sketch)uidoc.Document.GetElement(sketches[0]);
            }

            return sketch;
        }


        /// <summary>
        /// Consider a Revit length zero 
        /// if is smaller than this.
        /// </summary>
        const double _eps = 1.0e-9;

        /// <summary>
        /// Conversion factor from feet to millimetres.
        /// </summary>
        const double _feet_to_mm = 25.4 * 12;

        /// <summary>
        /// Conversion a given length value 
        /// from feet to millimetres.
        /// </summary>
        static long ConvertFeetToMillimetres(double d)
        {
            if (0 < d)
            {
                return _eps > d
                  ? 0
                  : (long)(_feet_to_mm * d + 0.5);

            }
            else
            {
                return _eps > -d
                  ? 0
                  : (long)(_feet_to_mm * d - 0.5);

            }
        }

        /// <summary>
        /// Conversion a given length value 
        /// from millimetres to feet.
        /// </summary>
        static double ConvertMillimetresToFeet(long d)
        {
            return d / _feet_to_mm;
        }

        /// <summary>
        /// Return a clipper integer point 
        /// from a Revit model space one.
        /// Do so by dropping the Z coordinate
        /// and converting from imperial feet 
        /// to millimetres.
        /// </summary>
        public static IntPoint GetIntPoint(XYZ p)
        {
            return new IntPoint(
              ConvertFeetToMillimetres(p.X),
              ConvertFeetToMillimetres(p.Y));
        }

        /// <summary>
        /// Return a Revit model space point 
        /// from a clipper integer one.
        /// Do so by adding a zero Z coordinate
        /// and converting from millimetres to
        /// imperial feet.
        /// </summary>
        public static XYZ GetXyzPoint(IntPoint p)
        {
            return new XYZ(
              ConvertMillimetresToFeet(p.X),
              ConvertMillimetresToFeet(p.Y),
              0.0);
        }



    }
}
