﻿' 
' (C) Copyright 2003-2009 by Autodesk, Inc. 
' 
' Permission to use, copy, modify, and distribute this software in 
' object code form for any purpose and without fee is hereby granted, 
' provided that the above copyright notice appears in all copies and 
' that both that copyright notice and the limited warranty and 
' restricted rights notice below appear in all supporting 
' documentation. 
' 
' AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
' AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF 
' MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE. AUTODESK, INC. 
' DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE 
' UNINTERRUPTED OR ERROR FREE. 
' 
' Use, duplication, or disclosure by the U.S. Government is subject to 
' restrictions set forth in FAR 52.227-19 (Commercial Computer 
' Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii) 
' (Rights in Technical Data and Computer Software), as applicable. 
' 

Imports System
Imports System.Text
Imports System.Reflection
Imports System.Collections.Generic
Imports System.Collections.ObjectModel
Imports System.IO

Imports Autodesk.Revit
Imports Autodesk.Revit.Elements

''' <summary> 
''' Base data class which stores the basic information for export 
''' </summary> 
Public Class ExportData
#Region "Class Memeber Variables"
    ''' <summary> 
    ''' Revit command data 
    ''' </summary> 
    Protected m_commandData As ExternalCommandData

    ''' <summary> 
    ''' Active document 
    ''' </summary> 
    Protected m_activeDoc As Document

    ''' <summary> 
    ''' File Name or Prefix to be used 
    ''' </summary> 
    Protected m_exportFileName As [String]

    ''' <summary> 
    ''' Directory to store the exported file 
    ''' </summary> 
    Protected m_exportFolder As [String]

    ''' <summary> 
    ''' Whether current view is a 3D view 
    ''' </summary> 
    Protected m_is3DView As Boolean

    ''' <summary> 
    ''' The format to be exported 
    ''' </summary> 
    Protected m_exportFormat As ExportFormat

    ''' <summary> 
    ''' The filter which will be used in file saving dialog 
    ''' </summary> 
    Protected m_filter As [String]

    ''' <summary> 
    ''' The title of exporting dialog 
    ''' </summary> 
    Protected m_title As [String]
#End Region

#Region "Class Properties"
    ''' <summary> 
    ''' Revit command data 
    ''' </summary> 
    Public ReadOnly Property CommandData() As ExternalCommandData
        Get
            Return m_commandData
        End Get
    End Property

    ''' <summary> 
    ''' File Name or Prefix to be used 
    ''' </summary> 
    Public Property ExportFileName() As [String]
        Get
            Return m_exportFileName
        End Get
        Set(ByVal value As [String])
            m_exportFileName = value
        End Set
    End Property

    ''' <summary> 
    ''' Directory to store the exported file 
    ''' </summary> 
    Public Property ExportFolder() As [String]
        Get
            Return m_exportFolder
        End Get
        Set(ByVal value As [String])
            m_exportFolder = value
        End Set
    End Property

    ''' <summary> 
    ''' Whether current view is a 3D view 
    ''' </summary> 
    Public Property Is3DView() As Boolean
        Get
            Return m_is3DView
        End Get
        Set(ByVal value As Boolean)
            m_is3DView = value
        End Set
    End Property

    ''' <summary> 
    ''' The format to be exported 
    ''' </summary> 
    Public Property ExportFormat() As ExportFormat
        Get
            Return m_exportFormat
        End Get
        Set(ByVal value As ExportFormat)
            m_exportFormat = value
        End Set
    End Property

    ''' <summary> 
    ''' The filter which will be used in file saving dialog 
    ''' </summary> 
    Public ReadOnly Property Filter() As [String]
        Get
            Return m_filter
        End Get
    End Property

    ''' <summary> 
    ''' The title of exporting dialog 
    ''' </summary> 
    Public ReadOnly Property Title() As [String]
        Get
            Return m_title
        End Get
    End Property
#End Region


#Region "Class Member Methods"
    ''' <summary> 
    ''' Constructor 
    ''' </summary> 
    ''' <param name="commandData">Revit command data</param> 
    ''' <param name="exportFormat">Format to export</param> 
    Public Sub New(ByVal commandData As ExternalCommandData, ByVal exportFormat As ExportFormat)
        m_commandData = commandData
        m_activeDoc = commandData.Application.ActiveDocument
        m_exportFormat = exportFormat
        m_filter = [String].Empty
        Initialize()
    End Sub

    ''' <summary> 
    ''' Initialize the variables 
    ''' </summary> 
    Private Sub Initialize()
        'The directory into which the file will be exported 
        Dim dllFilePath As [String] = Assembly.GetExecutingAssembly().Location
        m_exportFolder = Path.GetDirectoryName(dllFilePath)

        'The file name to be used by export 
        Dim docTitle As [String] = m_activeDoc.Title
        Dim viewName As [String] = m_activeDoc.ActiveView.Name
        Dim viewType As [String] = m_activeDoc.ActiveView.ViewType.ToString()
        m_exportFileName = (((docTitle & "-") + viewType & "-") + viewName & ".") + getExtension().ToString()

        'Whether current active view is 3D view 
        If m_activeDoc.ActiveView.ViewType = Autodesk.Revit.Enums.ViewType.ThreeD Then
            m_is3DView = True
        Else
            m_is3DView = False
        End If
    End Sub

    ''' <summary> 
    ''' Get the extension of the file to export 
    ''' </summary> 
    ''' <returns></returns> 
    Private Function getExtension() As [String]
        Dim extension As [String] = Nothing
        Select Case m_exportFormat
            Case ExportFormat.DWG
                extension = "dwg"
                Exit Select
            Case ExportFormat.DWF2D, ExportFormat.DWF3D
                extension = "dwf"
                Exit Select
            Case ExportFormat.DWFx2D, ExportFormat.DWFx3D
                extension = "dwfx"
                Exit Select
            Case ExportFormat.GBXML
                extension = "xml"
                Exit Select
            Case ExportFormat.FBX
                extension = "fbx"
                Exit Select
            Case ExportFormat.DGN
                extension = "dgn"
                Exit Select
            Case ExportFormat.Civil3D
                extension = "adsk"
                Exit Select
            Case Else
                Exit Select
        End Select

        Return extension
    End Function

    ''' <summary> 
    ''' Collect the parameters and export 
    ''' </summary> 
    ''' <returns></returns> 
    Public Overridable Function Export() As Boolean
        If m_exportFolder Is Nothing OrElse m_exportFileName Is Nothing Then
            Throw New NullReferenceException()
        End If

        Return True
    End Function
#End Region
End Class
