﻿' 
' (C) Copyright 2003-2009 by Autodesk, Inc. 
' 
' Permission to use, copy, modify, and distribute this software in 
' object code form for any purpose and without fee is hereby granted, 
' provided that the above copyright notice appears in all copies and 
' that both that copyright notice and the limited warranty and 
' restricted rights notice below appear in all supporting 
' documentation. 
' 
' AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
' AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF 
' MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE. AUTODESK, INC. 
' DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE 
' UNINTERRUPTED OR ERROR FREE. 
' 
' Use, duplication, or disclosure by the U.S. Government is subject to 
' restrictions set forth in FAR 52.227-19 (Commercial Computer 
' Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii) 
' (Rights in Technical Data and Computer Software), as applicable. 
' 

Imports System
Imports System.Text
Imports System.Collections.Generic
Imports System.Collections.ObjectModel

Imports Autodesk.Revit
Imports Autodesk.Revit.Elements


''' <summary> 
''' Data class which stores the information for importing dwg format 
''' </summary> 
Public Class ImportDWGData
    Inherits ImportData
#Region "Class Memeber Variables"
    ''' <summary> 
    ''' ThisViewOnly 
    ''' </summary> 
    Private m_importThisViewOnly As Boolean

    ''' <summary> 
    ''' All views 
    ''' </summary> 
    Private m_views As ViewSet

    ''' <summary> 
    ''' Import view 
    ''' </summary> 
    Private m_importView As View

    ''' <summary> 
    ''' ColorMode for import 
    ''' </summary> 
    Private m_colorMode As List(Of [String])

    ''' <summary> 
    ''' All available import color modes 
    ''' </summary> 
    Private m_enumColorMode As List(Of Autodesk.Revit.Enums.ImportColorMode)

    ''' <summary> 
    ''' Import color mode 
    ''' </summary> 
    Private m_importColorMode As Autodesk.Revit.Enums.ImportColorMode

    ''' <summary> 
    ''' Custom scale for import 
    ''' </summary> 
    Private m_importCustomScale As Double

    ''' <summary> 
    ''' OrientToView 
    ''' </summary> 
    Private m_importOrientToView As Boolean

    ''' <summary> 
    ''' Placement 
    ''' </summary> 
    Private m_placement As List(Of [String])

    ''' <summary> 
    ''' All placement for layers to be imported 
    ''' </summary> 
    Private m_enumPlacement As List(Of Autodesk.Revit.Enums.ImportPlacement)

    ''' <summary> 
    ''' Placement for import 
    ''' </summary> 
    Private m_importPlacement As Autodesk.Revit.Enums.ImportPlacement

    ''' <summary> 
    ''' All units for layer to be imported 
    ''' </summary> 
    Private m_unit As List(Of [String])

    ''' <summary> 
    ''' All import unit for import layers 
    ''' </summary> 
    Private m_enumUnit As List(Of Autodesk.Revit.Enums.ImportUnit)

    ''' <summary> 
    ''' Import unit 
    ''' </summary> 
    Private m_importUnit As Autodesk.Revit.Enums.ImportUnit

    ''' <summary> 
    ''' All available layers only 
    ''' </summary> 
    Private m_visibleLayersOnly As List(Of [String])

    ''' <summary> 
    ''' All boolean values for available visible layers 
    ''' </summary> 
    Private m_enumVisibleLayersOnly As List(Of Boolean)

    ''' <summary> 
    ''' Whether import visible layer only 
    ''' </summary> 
    Private m_importVisibleLayersOnly As Boolean

    ''' <summary> 
    ''' Whether active view is 3D 
    ''' </summary> 
    Private m_is3DView As Boolean
#End Region


#Region "Class Properties"
    ''' <summary> 
    ''' Get or set whether import this view only 
    ''' </summary> 
    Public Property ImportThisViewOnly() As Boolean
        Get
            Return m_importThisViewOnly
        End Get
        Set(ByVal value As Boolean)
            m_importThisViewOnly = value
        End Set
    End Property


    ''' <summary> 
    ''' all views for import 
    ''' </summary> 
    Public Property Views() As ViewSet
        Get
            Return m_views
        End Get
        Set(ByVal value As ViewSet)
            m_views = value
        End Set
    End Property


    ''' <summary> 
    ''' Import view 
    ''' </summary> 
    Public Property ImportView() As View
        Get
            Return m_importView
        End Get
        Set(ByVal value As View)
            m_importView = value
        End Set
    End Property


    ''' <summary> 
    ''' All available color modes for import 
    ''' </summary> 
    Public ReadOnly Property ColorMode() As ReadOnlyCollection(Of [String])
        Get
            Return New ReadOnlyCollection(Of [String])(m_colorMode)
        End Get
    End Property


    ''' <summary> 
    ''' All available import color modes 
    ''' </summary> 
    Public ReadOnly Property EnumColorMode() As ReadOnlyCollection(Of Autodesk.Revit.Enums.ImportColorMode)
        Get
            Return New ReadOnlyCollection(Of Autodesk.Revit.Enums.ImportColorMode)(m_enumColorMode)
        End Get
    End Property


    ''' <summary> 
    ''' Import color mode 
    ''' </summary> 
    Public Property ImportColorMode() As Autodesk.Revit.Enums.ImportColorMode
        Get
            Return m_importColorMode
        End Get
        Set(ByVal value As Autodesk.Revit.Enums.ImportColorMode)
            m_importColorMode = value
        End Set
    End Property


    ''' <summary> 
    ''' Custom scale for import 
    ''' </summary> 
    Public Property ImportCustomScale() As Double
        Get
            Return m_importCustomScale
        End Get
        Set(ByVal value As Double)
            m_importCustomScale = value
        End Set
    End Property


    ''' <summary> 
    ''' Whether import orient to view 
    ''' </summary> 
    Public Property ImportOrientToView() As Boolean
        Get
            Return m_importOrientToView
        End Get
        Set(ByVal value As Boolean)
            m_importOrientToView = value
        End Set
    End Property


    ''' <summary> 
    ''' All placement for layers to be imported 
    ''' </summary> 
    Public ReadOnly Property Placement() As ReadOnlyCollection(Of [String])
        Get
            Return New ReadOnlyCollection(Of [String])(m_placement)
        End Get
    End Property


    ''' <summary> 
    ''' All ImportPlacements for all layers to be imported 
    ''' </summary> 
    Public ReadOnly Property EnumPlacement() As ReadOnlyCollection(Of Autodesk.Revit.Enums.ImportPlacement)
        Get
            Return New ReadOnlyCollection(Of Autodesk.Revit.Enums.ImportPlacement)(m_enumPlacement)
        End Get
    End Property


    ''' <summary> 
    ''' Import placement for import 
    ''' </summary> 
    Public Property ImportPlacement() As Autodesk.Revit.Enums.ImportPlacement
        Get
            Return m_importPlacement
        End Get
        Set(ByVal value As Autodesk.Revit.Enums.ImportPlacement)
            m_importPlacement = value
        End Set
    End Property


    ''' <summary> 
    ''' All units for layer to be imported 
    ''' </summary> 
    Public ReadOnly Property Unit() As ReadOnlyCollection(Of [String])
        Get
            Return New ReadOnlyCollection(Of String)(m_unit)
        End Get
    End Property


    ''' <summary> 
    ''' All import unit for import layers 
    ''' </summary> 
    Public ReadOnly Property EnumUnit() As ReadOnlyCollection(Of Autodesk.Revit.Enums.ImportUnit)
        Get
            Return New ReadOnlyCollection(Of Autodesk.Revit.Enums.ImportUnit)(m_enumUnit)
        End Get
    End Property


    ''' <summary> 
    ''' Get or set import unit 
    ''' </summary> 
    Public Property ImportUnit() As Autodesk.Revit.Enums.ImportUnit
        Get
            Return m_importUnit
        End Get
        Set(ByVal value As Autodesk.Revit.Enums.ImportUnit)
            m_importUnit = value
        End Set
    End Property


    ''' <summary> 
    ''' All available layers only 
    ''' </summary> 
    Public ReadOnly Property VisibleLayersOnly() As ReadOnlyCollection(Of [String])
        Get
            Return New ReadOnlyCollection(Of [String])(m_visibleLayersOnly)
        End Get
    End Property


    ''' <summary> 
    ''' All boolean values for available visible layers 
    ''' </summary> 
    Public ReadOnly Property EnumVisibleLayersOnly() As ReadOnlyCollection(Of Boolean)
        Get
            Return New ReadOnlyCollection(Of Boolean)(m_enumVisibleLayersOnly)
        End Get
    End Property


    ''' <summary> 
    ''' Whether import visible layer only 
    ''' </summary> 
    Public Property ImportVisibleLayersOnly() As Boolean
        Get
            Return m_importVisibleLayersOnly
        End Get
        Set(ByVal value As Boolean)
            m_importVisibleLayersOnly = value
        End Set
    End Property

    ''' <summary> 
    ''' Whether active view is 3D 
    ''' </summary> 
    Public Property Is3DView() As Boolean
        Get
            Return m_is3DView
        End Get
        Set(ByVal value As Boolean)
            m_is3DView = value
        End Set
    End Property
#End Region


#Region "Class Member Methods"
    ''' <summary> 
    ''' Constructor 
    ''' </summary> 
    ''' <param name="commandData">Revit command data</param> 
    ''' <param name="format">Format to import</param> 
    Public Sub New(ByVal commandData As ExternalCommandData, ByVal format As ImportFormat)
        MyBase.New(commandData, format)
        Initialize()
    End Sub


    ''' <summary> 
    ''' Collect the parameters and export 
    ''' </summary> 
    ''' <returns></returns> 
    Public Overloads Overrides Function Import() As Boolean
        Dim imported As Boolean = False

        'parameter: DWGImportOptions 
        Dim dwgImportOption As New DWGImportOptions()
        dwgImportOption.ColorMode = m_importColorMode
        dwgImportOption.CustomScale = m_importCustomScale
        dwgImportOption.OrientToView = m_importOrientToView
        dwgImportOption.Placement = m_importPlacement
        dwgImportOption.ThisViewOnly = m_importThisViewOnly
        dwgImportOption.Unit = m_importUnit
        dwgImportOption.View = m_importView
        dwgImportOption.VisibleLayersOnly = m_importVisibleLayersOnly

        'parameter: Element 
        Dim element As Autodesk.Revit.Element = Nothing

        'Import 
        imported = m_activeDoc.Import(m_importFileFullName, dwgImportOption, element)

        Return imported
    End Function
#End Region


#Region "Class Implementation"
    ''' <summary> 
    ''' Initialize the variables 
    ''' </summary> 
    Private Sub Initialize()
        'ColorMode 
        m_colorMode = New List(Of [String])()
        m_enumColorMode = New List(Of Autodesk.Revit.Enums.ImportColorMode)()
        m_colorMode.Add("Black and white")
        m_enumColorMode.Add(Autodesk.Revit.Enums.ImportColorMode.BlackAndWhite)
        m_colorMode.Add("Preserve colors")
        m_enumColorMode.Add(Autodesk.Revit.Enums.ImportColorMode.Preserved)
        m_colorMode.Add("Invert colors")
        m_enumColorMode.Add(Autodesk.Revit.Enums.ImportColorMode.Inverted)

        'Placement 
        m_placement = New List(Of [String])()
        m_enumPlacement = New List(Of Autodesk.Revit.Enums.ImportPlacement)()
        m_placement.Add("Center-to-center")
        m_enumPlacement.Add(Autodesk.Revit.Enums.ImportPlacement.Centered)
        m_placement.Add("Origin-to-origin")
        m_enumPlacement.Add(Autodesk.Revit.Enums.ImportPlacement.Origin)

        'Unit 
        m_unit = New List(Of [String])()
        m_enumUnit = New List(Of Autodesk.Revit.Enums.ImportUnit)()
        m_unit.Add("Auto-Detect")
        m_enumUnit.Add(Autodesk.Revit.Enums.ImportUnit.[Default])
        m_unit.Add(Autodesk.Revit.Enums.ImportUnit.Foot.ToString())
        m_enumUnit.Add(Autodesk.Revit.Enums.ImportUnit.Foot)
        m_unit.Add(Autodesk.Revit.Enums.ImportUnit.Inch.ToString())
        m_enumUnit.Add(Autodesk.Revit.Enums.ImportUnit.Inch)
        m_unit.Add(Autodesk.Revit.Enums.ImportUnit.Meter.ToString())
        m_enumUnit.Add(Autodesk.Revit.Enums.ImportUnit.Meter)
        m_unit.Add(Autodesk.Revit.Enums.ImportUnit.Decimeter.ToString())
        m_enumUnit.Add(Autodesk.Revit.Enums.ImportUnit.Decimeter)
        m_unit.Add(Autodesk.Revit.Enums.ImportUnit.Centimeter.ToString())
        m_enumUnit.Add(Autodesk.Revit.Enums.ImportUnit.Centimeter)
        m_unit.Add(Autodesk.Revit.Enums.ImportUnit.Millimeter.ToString())
        m_enumUnit.Add(Autodesk.Revit.Enums.ImportUnit.Millimeter)
        m_unit.Add("Custom")
        m_enumUnit.Add(Autodesk.Revit.Enums.ImportUnit.[Default])

        'VisibleLayersOnly 
        m_visibleLayersOnly = New List(Of [String])()
        m_enumVisibleLayersOnly = New List(Of Boolean)()
        m_visibleLayersOnly.Add("All")
        m_enumVisibleLayersOnly.Add(False)
        m_visibleLayersOnly.Add("Visible")
        m_enumVisibleLayersOnly.Add(True)

        'Whether active view is 3D 
        m_is3DView = False
        If m_activeDoc.ActiveView.ViewType = Autodesk.Revit.Enums.ViewType.ThreeD Then
            m_is3DView = True
        End If

        'Views 
        m_views = New ViewSet()
        GetViews()

        m_importCustomScale = 0.0R
        m_importOrientToView = True
        m_importUnit = Autodesk.Revit.Enums.ImportUnit.[Default]
        m_importThisViewOnly = False
        m_importView = m_activeDoc.ActiveView
        m_importColorMode = Autodesk.Revit.Enums.ImportColorMode.Inverted
        m_importPlacement = Autodesk.Revit.Enums.ImportPlacement.Centered
        m_importVisibleLayersOnly = False

        m_filter = "DWG Files (*.dwg)|*.dwg"
        m_title = "Import DWG"
    End Sub


    ''' <summary> 
    ''' Get all the views to be displayed 
    ''' </summary> 
    Private Sub GetViews()
        Dim itor As ElementIterator = m_activeDoc.Elements
        itor.Reset()
        Dim views As New ViewSet()
        Dim floorPlans As New ViewSet()
        Dim ceilingPlans As New ViewSet()
        Dim engineeringPlans As New ViewSet()
        While itor.MoveNext()
            Dim view As View = TryCast(itor.Current, View)
            If view Is Nothing Then
                Continue While
            ElseIf view.ViewType = Autodesk.Revit.Enums.ViewType.FloorPlan Then
                floorPlans.Insert(view)
            ElseIf view.ViewType = Autodesk.Revit.Enums.ViewType.CeilingPlan Then
                ceilingPlans.Insert(view)
            ElseIf view.ViewType = Autodesk.Revit.Enums.ViewType.EngineeringPlan Then
                engineeringPlans.Insert(view)
            End If
        End While

        For Each floorPlan As View In floorPlans
            For Each ceilingPlan As View In ceilingPlans
                If floorPlan.ViewName = ceilingPlan.ViewName Then
                    views.Insert(floorPlan)
                End If
            Next
        Next

        For Each engineeringPlan As View In engineeringPlans
            If engineeringPlan.ViewName = engineeringPlan.GenLevel.Name Then
                views.Insert(engineeringPlan)
            End If
        Next

        Dim activeView As View = m_activeDoc.ActiveView
        Dim viewType As Autodesk.Revit.Enums.ViewType = activeView.ViewType
        If viewType = Autodesk.Revit.Enums.ViewType.FloorPlan OrElse viewType = Autodesk.Revit.Enums.ViewType.CeilingPlan Then
            m_views.Insert(activeView)
            For Each view As View In views
                If view.GenLevel.Elevation < activeView.GenLevel.Elevation Then
                    m_views.Insert(view)
                End If
            Next
        ElseIf viewType = Autodesk.Revit.Enums.ViewType.EngineeringPlan Then
            If views.Contains(activeView) Then
                m_views.Insert(activeView)
            End If
            For Each view As View In views
                If view.GenLevel.Elevation < activeView.GenLevel.Elevation Then
                    m_views.Insert(view)
                End If
            Next
        Else
            'Get view of the lowest elevation 
            Dim i As Integer = 0
            Dim elevation As Double = 0
            Dim viewLowestElevation As View = Nothing
            For Each view As View In views
                If i = 0 Then
                    elevation = view.GenLevel.Elevation
                    viewLowestElevation = view
                Else
                    If view.GenLevel.Elevation <= elevation Then
                        elevation = view.GenLevel.Elevation
                        viewLowestElevation = view
                    End If
                End If

                i += 1
            Next
            m_views.Insert(viewLowestElevation)
        End If
    End Sub
#End Region
End Class
