﻿' 
' (C) Copyright 2003-2009 by Autodesk, Inc. 
' 
' Permission to use, copy, modify, and distribute this software in 
' object code form for any purpose and without fee is hereby granted, 
' provided that the above copyright notice appears in all copies and 
' that both that copyright notice and the limited warranty and 
' restricted rights notice below appear in all supporting 
' documentation. 
' 
' AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
' AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF 
' MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE. AUTODESK, INC. 
' DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE 
' UNINTERRUPTED OR ERROR FREE. 
' 
' Use, duplication, or disclosure by the U.S. Government is subject to 
' restrictions set forth in FAR 52.227-19 (Commercial Computer 
' Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii) 
' (Rights in Technical Data and Computer Software), as applicable. 
' 

Imports System
Imports System.Collections.Generic
Imports System.ComponentModel
Imports System.Data
Imports System.Drawing
Imports System.Text
Imports System.Windows.Forms
Imports System.Reflection
Imports System.IO

Imports Autodesk.Revit
Imports Autodesk.Revit.Elements

''' <summary> 
''' It contains a dialog which provides the options of importing dwg format 
''' </summary> 
Partial Public Class ImportDWGForm
    Inherits System.Windows.Forms.Form
    ''' <summary> 
    ''' Data class 
    ''' </summary> 
    Private m_importData As ImportDWGData

    ''' <summary> 
    ''' Constructor 
    ''' </summary> 
    ''' <param name="importData"></param> 
    Public Sub New(ByVal importData As ImportDWGData)
        InitializeComponent()
        m_importData = importData
        InitializeControls()
    End Sub

    ''' <summary> 
    ''' Change the status of checkBoxOrient2View and comboBoxLevel according to the selection 
    ''' </summary> 
    ''' <param name="sender"></param> 
    ''' <param name="e"></param> 
    Private Sub checkBoxCurrentViewOnly_CheckedChanged(ByVal sender As Object, ByVal e As EventArgs)
        Dim currentViewOnly As Boolean = checkBoxCurrentViewOnly.Checked
        checkBoxOrient2View.Enabled = Not currentViewOnly
        comboBoxLevel.Enabled = Not currentViewOnly
    End Sub

    ''' <summary> 
    ''' Initialize values and status of controls 
    ''' </summary> 
    Private Sub InitializeControls()
        'Layers 
        For Each layer As [String] In m_importData.VisibleLayersOnly
            comboBoxLayers.Items.Add(layer)
        Next
        comboBoxLayers.SelectedIndex = 0

        'unit 
        For Each unit As [String] In m_importData.Unit
            comboBoxUnits.Items.Add(unit)
        Next
        comboBoxUnits.SelectedIndex = 0

        'Level 
        For Each view As Autodesk.Revit.Elements.View In m_importData.Views
            comboBoxLevel.Items.Add(view.ViewName)
        Next
        comboBoxLevel.SelectedIndex = 0

        'If active view is 3D 
        If m_importData.Is3DView Then
            checkBoxCurrentViewOnly.Checked = False
            checkBoxCurrentViewOnly.Enabled = False
        End If

        Me.Text = m_importData.Title
    End Sub

    ''' <summary> 
    ''' Specify a file to export from 
    ''' </summary> 
    ''' <param name="sender"></param> 
    ''' <param name="e"></param> 
    Private Sub buttonBrowser_Click(ByVal sender As Object, ByVal e As EventArgs)
        Dim returnFileFullName As [String] = [String].Empty
        If MainData.ShowOpenDialog(m_importData, returnFileFullName) <> DialogResult.Cancel Then
            textBoxFileSource.Text = returnFileFullName
        End If
    End Sub

    ''' <summary> 
    ''' Change the status of textBoxScale according to the selection 
    ''' </summary> 
    ''' <param name="sender"></param> 
    ''' <param name="e"></param> 
    Private Sub comboBoxUnits_SelectedIndexChanged(ByVal sender As Object, ByVal e As EventArgs)
        Dim [custom] As Boolean = comboBoxUnits.SelectedIndex = 7
        textBoxScale.Enabled = [custom]
    End Sub

    ''' <summary> 
    ''' Transfer information back to ImportData class and execute IMPORT operation 
    ''' </summary> 
    ''' <param name="sender"></param> 
    ''' <param name="e"></param> 
    Private Sub buttonOpen_Click(ByVal sender As Object, ByVal e As EventArgs)
        If ValidateFileName() Then
            'Color Mode 
            SetImportColorMode()
            'Placement 
            SetImportPlacement()
            'Current view only, Orient to view, view to import into 
            SetImportViewsRelated()
            'Visible layers only 
            SetImportLayers()
            'Unit, scaling 
            SetImportUnitsAndScaling()

            'Import 
            Try
                If Not m_importData.Import() Then
                    MessageBox.Show("Cannot import " & Path.GetFileName(m_importData.ImportFileFullName) & " in current settings.", "Import", MessageBoxButtons.OK, MessageBoxIcon.Warning)
                End If
            Catch ex As Exception
                MessageBox.Show(ex.ToString(), "Import Failed", MessageBoxButtons.OK, MessageBoxIcon.[Error])
            End Try

            Me.DialogResult = DialogResult.OK
            Me.Close()
        End If
    End Sub

    ''' <summary> 
    ''' Validate the file to import 
    ''' </summary> 
    ''' <returns></returns> 
    Private Function ValidateFileName() As [Boolean]
        Dim fileNameFull As [String] = textBoxFileSource.Text
        'If the textBoxFileSource is empty 
        If [String].IsNullOrEmpty(fileNameFull) Then
            MessageBox.Show("Please specify the folder and file name!", "Information", MessageBoxButtons.OK, MessageBoxIcon.Information)
            textBoxFileSource.Focus()
            Return False
        End If

        If Not File.Exists(fileNameFull) Then
            MessageBox.Show("Please specify an existing file!", "Information", MessageBoxButtons.OK, MessageBoxIcon.Information)
            textBoxFileSource.Focus()
            Return False
        End If

        m_importData.ImportFileFullName = fileNameFull

        Return True
    End Function

    ''' <summary> 
    ''' Set the value of color mode 
    ''' </summary> 
    Private Sub SetImportColorMode()
        Dim colorMode As [String]
        If radioButtonBlackWhite.Checked Then
            colorMode = radioButtonBlackWhite.Text
        ElseIf radioButtonPreserve.Checked Then
            colorMode = radioButtonPreserve.Text
        Else
            colorMode = radioButtonInvertColor.Text
        End If

        Dim indexColor As Integer = 0
        For i As Integer = 0 To m_importData.ColorMode.Count - 1
            If colorMode = m_importData.ColorMode(i) Then
                indexColor = i
                Exit For
            End If
        Next

        m_importData.ImportColorMode = m_importData.EnumColorMode(indexColor)
    End Sub

    ''' <summary> 
    ''' Set the value of placement 
    ''' </summary> 
    Private Sub SetImportPlacement()
        Dim placement As [String]
        If radioButtonCenter2Center.Checked Then
            placement = radioButtonCenter2Center.Text
        Else
            placement = radioButtonOrigin2Origin.Text
        End If

        Dim indexPlacement As Integer = 0
        For i As Integer = 0 To m_importData.Placement.Count - 1
            If placement = m_importData.Placement(i) Then
                indexPlacement = i
                Exit For
            End If
        Next

        m_importData.ImportPlacement = m_importData.EnumPlacement(indexPlacement)
    End Sub

    ''' <summary> 
    ''' Set the value of ImportThisViewOnly, ImportOrientToView, ImportView 
    ''' </summary> 
    Private Sub SetImportViewsRelated()
        If checkBoxCurrentViewOnly.Checked Then
            m_importData.ImportThisViewOnly = True
        Else
            m_importData.ImportThisViewOnly = False
            m_importData.ImportOrientToView = checkBoxOrient2View.Checked
            Dim viewName As [String] = comboBoxLevel.SelectedItem.ToString()
            For Each view As Autodesk.Revit.Elements.View In m_importData.Views
                If viewName = view.ViewName Then
                    m_importData.ImportView = view
                    Exit For
                End If
            Next
        End If
    End Sub

    ''' <summary> 
    ''' Set the value of ImportVisibleLayersOnly 
    ''' </summary> 
    Private Sub SetImportLayers()
        'comboBoxLayers 
        m_importData.ImportVisibleLayersOnly = m_importData.EnumVisibleLayersOnly(comboBoxLayers.SelectedIndex)
    End Sub

    ''' <summary> 
    ''' Set the value of unit, scaling 
    ''' </summary> 
    Private Sub SetImportUnitsAndScaling()
        Dim [custom] As Boolean = comboBoxUnits.SelectedIndex = 7

        'Custom is selected 
        If [custom] Then
            'Set the scaling 
            m_importData.ImportCustomScale = [Double].Parse(textBoxScale.Text)
        Else
            'Do not change the scaling 
            'Set unit 
            m_importData.ImportUnit = m_importData.EnumUnit(comboBoxUnits.SelectedIndex)
        End If
    End Sub

    ''' <summary> 
    ''' Only numbers and '.' permitted 
    ''' </summary> 
    ''' <param name="sender"></param> 
    ''' <param name="e"></param> 
    Private Sub textBoxScale_KeyPress(ByVal sender As Object, ByVal e As KeyPressEventArgs)
        If (e.KeyChar < Chr(46) OrElse e.KeyChar > Chr(57)) AndAlso (e.KeyChar <> Chr(8) OrElse e.KeyChar = Chr(47)) Then
            e.Handled = True
        Else
            e.Handled = False
        End If
    End Sub

    ''' <summary> 
    ''' Only one '.' permitted 
    ''' </summary> 
    ''' <param name="sender"></param> 
    ''' <param name="e"></param> 
    Private Sub textBoxScale_TextChanged(ByVal sender As Object, ByVal e As EventArgs)
        Dim newPoint As Boolean = False
        Dim tmp As Char = "0"c
        Dim text As String = textBoxScale.Text.Trim()
        Dim textAfter As New StringBuilder()
        For i As Integer = 0 To text.Length - 1
            tmp = Char.Parse(text.Substring(i, 1).ToString())
            If Not newPoint AndAlso (tmp = "."c) Then
                textAfter.Append(tmp)
                newPoint = True
            Else
                If Char.IsDigit(tmp) Then
                    textAfter.Append(tmp)
                End If

            End If
        Next
        textBoxScale.Text = textAfter.ToString()
    End Sub
End Class