﻿' 
' (C) Copyright 2003-2009 by Autodesk, Inc. 
' 
' Permission to use, copy, modify, and distribute this software in 
' object code form for any purpose and without fee is hereby granted, 
' provided that the above copyright notice appears in all copies and 
' that both that copyright notice and the limited warranty and 
' restricted rights notice below appear in all supporting 
' documentation. 
' 
' AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
' AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF 
' MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE. AUTODESK, INC. 
' DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE 
' UNINTERRUPTED OR ERROR FREE. 
' 
' Use, duplication, or disclosure by the U.S. Government is subject to 
' restrictions set forth in FAR 52.227-19 (Commercial Computer 
' Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii) 
' (Rights in Technical Data and Computer Software), as applicable. 
' 

Imports System
Imports System.Collections.Generic
Imports System.Text
Imports System.IO
Imports System.Windows.Forms

Imports Autodesk.Revit

''' <summary> 
''' Export formats 
''' </summary> 
Public Enum ExportFormat
    ''' <summary> 
    ''' DWG format 
    ''' </summary> 
    DWG
    ''' <summary> 
    ''' 2D DWF format 
    ''' </summary> 
    DWF2D
    ''' <summary> 
    ''' 2D DWFx format 
    ''' </summary> 
    DWFx2D
    ''' <summary> 
    ''' 3D DWF format 
    ''' </summary> 
    DWF3D
    ''' <summary> 
    ''' 3D DWFx format 
    ''' </summary> 
    DWFx3D
    ''' <summary> 
    ''' GBXML format 
    ''' </summary> 
    GBXML
    ''' <summary> 
    ''' FBX format 
    ''' </summary> 
    FBX
    ''' <summary> 
    ''' DGN format 
    ''' </summary> 
    DGN
    ''' <summary> 
    ''' Civil3D format 
    ''' </summary> 
    Civil3D
End Enum

''' <summary> 
''' Import formats 
''' </summary> 
Public Enum ImportFormat
    ''' <summary> 
    ''' DWF format 
    ''' </summary> 
    DWG
    ''' <summary> 
    ''' IMAGE format 
    ''' </summary> 
    IMAGE
    ''' <summary> 
    ''' GBXML format 
    ''' </summary> 
    GBXML
    ''' <summary> 
    ''' Inventor format 
    ''' </summary> 
    Inventor
End Enum

''' <summary> 
''' Data class contains the external command data. 
''' </summary> 
Public Class MainData
    ' Revit command data 
    Private m_commandData As ExternalCommandData

    ' Whether current view is a 3D view 
    Private m_is3DView As Boolean

    ''' <summary> 
    ''' Revit command data 
    ''' </summary> 
    Public ReadOnly Property CommandData() As ExternalCommandData
        Get
            Return m_commandData
        End Get
    End Property

    ''' <summary> 
    ''' Whether current view is a 3D view 
    ''' </summary> 
    Public Property Is3DView() As Boolean
        Get
            Return m_is3DView
        End Get
        Set(ByVal value As Boolean)
            m_is3DView = value
        End Set
    End Property

    ''' <summary> 
    ''' Constructor 
    ''' </summary> 
    ''' <param name="commandData">Revit command data</param> 
    Public Sub New(ByVal commandData As ExternalCommandData)
        m_commandData = commandData

        'Whether current active view is 3D view 
        If commandData.Application.ActiveDocument.ActiveView.ViewType = Autodesk.Revit.Enums.ViewType.ThreeD Then
            m_is3DView = True
        Else
            m_is3DView = False
        End If
    End Sub

    ''' <summary> 
    ''' Get the format to export 
    ''' </summary> 
    ''' <param name="selectedFormat">Selected format in format selecting dialog</param> 
    ''' <returns>The format to export</returns> 
    Private Shared Function GetSelectedExportFormat(ByVal selectedFormat As String) As ExportFormat
        Dim format As ExportFormat = ExportFormat.DWG
        Select Case selectedFormat
            Case "DWG"
                format = ExportFormat.DWG
                Exit Select
            Case "2D DWF"
                format = ExportFormat.DWF2D
                Exit Select
            Case "2D DWFx"
                format = ExportFormat.DWFx2D
                Exit Select
            Case "GBXML"
                format = ExportFormat.GBXML
                Exit Select
            Case "3D DWFx"
                format = ExportFormat.DWFx3D
                Exit Select
            Case "3D DWF"
                format = ExportFormat.DWF3D
                Exit Select
            Case "FBX"
                format = ExportFormat.FBX
                Exit Select
            Case "DGN"
                format = ExportFormat.DGN
                Exit Select
            Case "Civil3D"
                format = ExportFormat.Civil3D
                Exit Select
            Case Else
                Exit Select
        End Select

        Return format
    End Function

    ''' <summary> 
    ''' Export according to selected format 
    ''' </summary> 
    ''' <param name="selectedFormat">Selected format</param> 
    Public Function Export(ByVal selectedFormat As String) As DialogResult
        Dim format As ExportFormat = GetSelectedExportFormat(selectedFormat)
        Dim dialogResult__1 As DialogResult = DialogResult.OK

        Try
            Select Case format
                Case ExportFormat.DWG
                    Dim exportDWGData As New ExportDWGData(m_commandData, format)
                    Using exportForm As New ExportWithViewsForm(exportDWGData)
                        dialogResult__1 = exportForm.ShowDialog()
                    End Using
                    Exit Select
                    'Case ExportFormat.DWF2D, ExportFormat.DWFx2D
                    '    Dim export2DDWFData As New Export2DDWFData(m_commandData, format)
                    '    Using exportForm As New ExportWithViewsForm(export2DDWFData)
                    '        dialogResult__1 = exportForm.ShowDialog()
                    '    End Using
                    '    Exit Select
                    'Case ExportFormat.GBXML
                    '    Dim exportGBXMLData As New ExportGBXMLData(m_commandData, format)
                    '    dialogResult__1 = Export(exportGBXMLData)
                    '    Exit Select
                    'Case ExportFormat.DWFx3D, ExportFormat.DWF3D
                    '    Dim export3DDWFData As New Export3DDWFData(m_commandData, format)
                    '    dialogResult__1 = Export(export3DDWFData)
                    '    Exit Select
                    'Case ExportFormat.FBX
                    '    Dim exportFBXData As New ExportFBXData(m_commandData, format)
                    '    dialogResult__1 = Export(exportFBXData)
                    '    Exit Select
                    'Case ExportFormat.DGN
                    '    Dim exportDGNData As New ExportDGNData(m_commandData, format)
                    '    Using exportForm As New ExportWithViewsForm(exportDGNData)
                    '        dialogResult__1 = exportForm.ShowDialog()
                    '    End Using
                    '    Exit Select
                    'Case ExportFormat.Civil3D
                    '    Dim exportCivil3DData As New ExportCivil3DData(m_commandData, format)
                    '    If exportCivil3DData.DataValidated Then
                    '        Using exportForm As New ExportCivil3DForm(exportCivil3DData)
                    '            dialogResult__1 = exportForm.ShowDialog()
                    '        End Using
                    '    End If
                    '    Exit Select
                Case Else
                    Exit Select
            End Select
        Catch generatedExceptionName As Exception
            Dim errorMessage As [String] = "Failed to export " & format & " format"
            MessageBox.Show(errorMessage, "Error", MessageBoxButtons.OK, MessageBoxIcon.[Error])
        End Try

        Return dialogResult__1
    End Function

    ''' <summary> 
    ''' Export 
    ''' </summary> 
    ''' <param name="data"></param> 
    Private Shared Function Export(ByVal data As ExportData) As DialogResult
        Dim returnFilename As [String] = [String].Empty
        Dim filterIndex As Integer = -1

        Dim result As DialogResult = ShowSaveDialog(data, returnFilename, filterIndex)
        If result <> DialogResult.Cancel Then
            data.ExportFileName = Path.GetFileName(returnFilename)
            data.ExportFolder = Path.GetDirectoryName(returnFilename)
            If Not data.Export() Then
                MessageBox.Show("This project cannot be exported to " & data.ExportFileName & " in current settings.", "Export", MessageBoxButtons.OK, MessageBoxIcon.Warning)
            End If
        End If

        Return result
    End Function

    ''' <summary> 
    ''' Show Save dialog 
    ''' </summary> 
    ''' <param name="exportData">Data to export</param> 
    ''' <param name="returnFileName">File name will be returned</param> 
    ''' <param name="filterIndex">Selected filter index will be returned</param> 
    ''' <returns></returns> 
    Public Shared Function ShowSaveDialog(ByVal exportData As ExportData, ByRef returnFileName As [String], ByRef filterIndex As Integer) As DialogResult
        Using saveDialog As New SaveFileDialog()
            saveDialog.Title = exportData.Title
            saveDialog.InitialDirectory = exportData.ExportFolder
            saveDialog.FileName = exportData.ExportFileName
            saveDialog.Filter = exportData.Filter
            saveDialog.FilterIndex = 1
            saveDialog.RestoreDirectory = True

            Dim result As DialogResult = saveDialog.ShowDialog()
            If result <> DialogResult.Cancel Then
                returnFileName = saveDialog.FileName
                filterIndex = saveDialog.FilterIndex
            End If

            Return result
        End Using
    End Function

    ''' <summary> 
    ''' Get the format to import 
    ''' </summary> 
    ''' <param name="selectedFormat">Selected format in format selecting dialog</param> 
    ''' <returns>The format to import</returns> 
    Private Shared Function GetSelectedImportFormat(ByVal selectedFormat As String) As ImportFormat
        Dim format As ImportFormat = ImportFormat.DWG
        Select Case selectedFormat
            Case "DWG"
                format = ImportFormat.DWG
                Exit Select
            Case "IMAGE"
                format = ImportFormat.IMAGE
                Exit Select
            Case "GBXML"
                format = ImportFormat.GBXML
                Exit Select
            Case "Inventor"
                format = ImportFormat.Inventor
                Exit Select
            Case Else
                Exit Select
        End Select

        Return format
    End Function

    ''' <summary> 
    ''' Export according to selected format 
    ''' </summary> 
    ''' <param name="selectedFormat">Selected format</param> 
    ''' <returns></returns> 
    Public Function Import(ByVal selectedFormat As String) As DialogResult
        Dim dialogResult__1 As DialogResult = DialogResult.OK
        Dim format As ImportFormat = GetSelectedImportFormat(selectedFormat)

        Try
            Select Case format
                Case ImportFormat.DWG
                    Dim importDWGData As New ImportDWGData(m_commandData, format)
                    Using importForm As New ImportDWGForm(importDWGData)
                        dialogResult__1 = importForm.ShowDialog()
                    End Using
                    Exit Select
                    'Case ImportFormat.IMAGE
                    '    Dim importIMAGEData As New ImportImageData(m_commandData, format)
                    '    dialogResult__1 = Import(importIMAGEData)
                    '    Exit Select
                    'Case ImportFormat.GBXML
                    '    Dim importGBXMLData As New ImportGBXMLData(m_commandData, format)
                    '    dialogResult__1 = Import(importGBXMLData)
                    '    Exit Select
                    'Case ImportFormat.Inventor
                    '    Dim importInventorData As New ImportInventorData(m_commandData, format)
                    '    dialogResult__1 = Import(importInventorData)
                    '    Exit Select
                Case Else
                    Exit Select
            End Select
        Catch generatedExceptionName As Exception
            Dim errorMessage As [String] = "Failed to import " & format & " format"
            MessageBox.Show(errorMessage, "Error", MessageBoxButtons.OK, MessageBoxIcon.[Error])
        End Try

        Return dialogResult__1
    End Function

    ''' <summary> 
    ''' Import 
    ''' </summary> 
    ''' <param name="data"></param> 
    Private Shared Function Import(ByVal data As ImportData) As DialogResult
        Dim returnFilename As [String] = [String].Empty
        Dim result As DialogResult = ShowOpenDialog(data, returnFilename)
        If result <> DialogResult.Cancel Then
            data.ImportFileFullName = returnFilename
            If Not data.Import() Then
                MessageBox.Show("Cannot import " & Path.GetFileName(data.ImportFileFullName) & " in current settings.", "Import", MessageBoxButtons.OK, MessageBoxIcon.Warning)
            End If
        End If

        Return result
    End Function

    ''' <summary> 
    ''' Show Open File dialog 
    ''' </summary> 
    ''' <param name="importData">Data to import</param> 
    ''' <param name="returnFileName">File name will be returned</param> 
    ''' <returns>Dialog result</returns> 
    Public Shared Function ShowOpenDialog(ByVal importData As ImportData, ByRef returnFileName As [String]) As DialogResult
        Using importDialog As New OpenFileDialog()
            importDialog.Title = importData.Title
            importDialog.InitialDirectory = importData.ImportFolder
            importDialog.Filter = importData.Filter
            importDialog.RestoreDirectory = True

            Dim result As DialogResult = importDialog.ShowDialog()
            If result <> DialogResult.Cancel Then
                returnFileName = importDialog.FileName
            End If

            Return result
        End Using
    End Function
End Class
