﻿using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;

namespace RevitJigSample.ExternalGraphics
{
    public abstract class DrawJigBase : IDisposable
    {
        public UIApplication HostApplication { get; set; }
        public JigDrawingServer DrawingServer { get; set; }
        public UserActivityHook UserActivityHook { get; set; }
        public List<XYZ> PickedPoints { get; set; }
        public string UserInput { get; set; }

        private IntPtr revitWindow = IntPtr.Zero;
        protected DrawJigBase(UIApplication uiApplication)
        {
            this.UserInput = "";
            this.PickedPoints = new List<XYZ>();
            this.HostApplication = uiApplication;
            this.UserActivityHook = new UserActivityHook(true, true);
            this.UserActivityHook.OnMouseActivity += OnMouseActivity;
            this.UserActivityHook.KeyPress += OnKeyPressActivity;

            this.DrawingServer = new JigDrawingServer(this.HostApplication.ActiveUIDocument.Document);
            var externalGraphics = new DrawingServerHost();
            externalGraphics.RegisterServer(this.DrawingServer);

            this.revitWindow = GetActiveWindow();
        }

        public abstract void DrawJig();

        public virtual void OnKeyPressActivity(object sender, KeyPressEventArgs e)
        {
            var topWindow = GetActiveWindow();
            if (this.revitWindow != topWindow)
            {
                e.Handled = false;
                return;
            }

            if (this.DrawingServer != null && e.KeyChar == 27 && this.DrawingServer.BasePoint != null)
            {
                this.DrawingServer.BasePoint = null;
                this.DrawingServer.NextPoint = null;
                this.UserInput = "";
            }
            else if (this.DrawingServer != null && e.KeyChar == 27)
            {
                StopJig();
            }
            if (this.DrawingServer != null && e.KeyChar == 8 && this.UserInput.Length > 0)
            {
                this.UserInput = this.UserInput.Substring(0, this.UserInput.Length - 1);
            }
            else
            {
                if (char.IsLetterOrDigit(e.KeyChar))
                {
                    this.UserInput += e.KeyChar.ToString();
                }
            }

            e.Handled = false;
        }

        public void StopJig()
        {
            if (this.UserActivityHook != null)
            {
                this.UserActivityHook.OnMouseActivity -= OnMouseActivity;
                this.UserActivityHook.KeyPress -= OnKeyPressActivity;
                this.UserActivityHook.Stop();
            }

            if (this.DrawingServer != null)
            {
                this.DrawingServer.BasePoint = null;
                this.DrawingServer.NextPoint = null;

                var externalGraphics = new DrawingServerHost();
                externalGraphics.UnRegisterServer(this.DrawingServer.Document);

                this.DrawingServer = null;
            }
        }

        public virtual void OnMouseActivity(object sender, MouseEventArgs e)
        {
            var topWindow = GetActiveWindow();
            if (this.revitWindow != topWindow)
            {
                return;
            }

            try
            {
                var currPoint = GetMousePoint();
                //add points to the list:
                if (this.PickedPoints != null && e.Clicks > 0 && e.Button == MouseButtons.Left)
                {
                    this.PickedPoints.Add(currPoint);
                }

                if (this.DrawingServer.BasePoint == null && e.Clicks > 0 && e.Button == MouseButtons.Left)
                {
                    //start server
                    this.DrawingServer.BasePoint = currPoint;
                }
                else if (this.DrawingServer != null && e.Clicks > 0 && e.Button == MouseButtons.Left)
                {
                    this.DrawingServer.BasePoint = currPoint;
                    this.DrawingServer.NextPoint = null;
                }
                else if (this.DrawingServer != null)
                {
                    //mouse is moving
                    if (this.DrawingServer.NextPoint != null)
                    {
                        if (currPoint.DistanceTo(this.DrawingServer.NextPoint) > 0.01)
                        {
                            this.DrawingServer.NextPoint = currPoint;
                            this.DrawJig();
                            HostApplication.ActiveUIDocument.RefreshActiveView();
                        }
                    }
                    else
                    {
                        this.DrawingServer.NextPoint = currPoint;
                        this.DrawJig();
                        HostApplication.ActiveUIDocument.RefreshActiveView();
                    }
                }
            }
            catch (Exception ex)
            {

            }
        }

        private UIView GetActiveUiView(UIDocument uidoc)
        {
            var doc = uidoc.Document;
            var view = doc.ActiveView;
            var uiviews = uidoc.GetOpenUIViews();
            UIView uiview = null;

            foreach (var uv in uiviews)
            {
                if (uv.ViewId.Equals(view.Id))
                {
                    uiview = uv;
                    break;
                }
            }
            return uiview;
        }

        protected XYZ GetMousePoint()
        {
            var uiView = GetActiveUiView(this.HostApplication.ActiveUIDocument);
            var corners = uiView.GetZoomCorners();
            var rect = uiView.GetWindowRectangle();
            var p = Cursor.Position;
            var dx = (double)(p.X - rect.Left) / (rect.Right - rect.Left);
            var dy = (double)(p.Y - rect.Bottom) / (rect.Top - rect.Bottom);
            var a = corners[0];
            var b = corners[1];
            var v = b - a;
            var q = a
                    + dx * v.X * XYZ.BasisX
                    + dy * v.Y * XYZ.BasisY;

            return q;
        }

        public void Dispose()
        {
            this.StopJig();
        }

        [DllImport("user32.dll")]
        static extern IntPtr GetActiveWindow();
    }
}
