﻿using System;
using System.Collections.Generic;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using RevitJigSample.ExternalGraphics;

namespace RevitJigSample
{
    [Transaction(TransactionMode.Manual)]
    [Regeneration(RegenerationOption.Manual)]
    public class RectangleJigCommand : IExternalCommand
    {
        public Result Execute(ExternalCommandData commandData, ref string message, ElementSet elements)
        {
            if (commandData.Application.ActiveUIDocument == null)
            {
                return Result.Succeeded;
            }

            var uiDocument = commandData.Application.ActiveUIDocument;
            var document = uiDocument.Document;
            XYZ p1 = null;
            try
            {
                p1 = uiDocument.Selection.PickPoint("Pick Base Point:");
            }
            catch (Exception ex)
            {
                //
            }

            if (p1 == null)
            {
                return Result.Succeeded;
            }

            XYZ p3 = null;
            RectangleJig rectangleJig = null;
            try
            {
                rectangleJig = new RectangleJig(commandData.Application) {DrawingServer = {BasePoint = p1}};
                rectangleJig.DrawJig();

                p3 = uiDocument.Selection.PickPoint("Pick Next Point:");
            }
            catch (Exception ex)
            {
                //
            }
            finally
            {
                if (rectangleJig != null)
                {
                    rectangleJig.Dispose();
                }
            }

            if (p3 == null)
            {
                return Result.Succeeded;
            }

            var mpt = (p1 + p3) * 0.5;
            var currView = document.ActiveView;
            var plane = Plane.CreateByNormalAndOrigin(currView.RightDirection, mpt);
            var mirrorMat = Transform.CreateReflection(plane);

            var p2 = mirrorMat.OfPoint(p1);
            var p4 = mirrorMat.OfPoint(p3);

            var points = new List<XYZ> {p1, p2, p3, p4, p1};
            using (var trans = new Transaction(document, "Draw Rectangle"))
            {
                trans.Start();
                var lpt = points[0];
                for (var k = 1; k < points.Count; k++)
                {
                    var cpt = points[k];
                    if (lpt.DistanceTo(cpt) > 0.001)
                    {
                        CreateModelLine(document, lpt, cpt);
                    }

                    lpt = cpt;
                }

                trans.Commit();
            }

            return Result.Succeeded;
        }

        public static ModelCurve CreateModelLine(Document doc, XYZ p, XYZ q)
        {
            if (p.DistanceTo(q) < doc.Application.ShortCurveTolerance)
            {
                return null;
            }

            var v = q - p;
            var dxy = Math.Abs(v.X) + Math.Abs(v.Y);
            var w = (dxy > doc.Application.ShortCurveTolerance)
                ? XYZ.BasisZ
                : XYZ.BasisY;

            var norm = v.CrossProduct(w)
                .Normalize();
            var plane = Plane.CreateByNormalAndOrigin(norm, p);
            var sketchPlane = SketchPlane.Create(doc, plane);
            var line = Line.CreateBound(p, q);
            var curve = doc.IsFamilyDocument
                ? doc.FamilyCreate.NewModelCurve(line, sketchPlane)
                : doc.Create.NewModelCurve(line, sketchPlane);

            return curve;
        }
    }
}
