#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;

#endregion

namespace sample
{
    /// <summary>
    /// given a planar face on a solid, generate two solids, 
    /// the bounding box of the input solid split in two by the plane of the selected planar face
    /// </summary>
    [Transaction(TransactionMode.Manual)]
    public class PlanarSplitBoundingBox : IExternalCommand
    {
        public Result Execute(
            ExternalCommandData commandData,
            ref string message,
            ElementSet elements)
        {
            // basic intro setup

            UIApplication uiapp = commandData.Application;
            UIDocument uidoc = uiapp.ActiveUIDocument;
            Application app = uiapp.Application;
            Document doc = uidoc.Document;

            // select face for split
            Reference objectRef = uidoc.Selection.PickObject(
                ObjectType.Face,
                new PlanarFaceFilter(doc),
                "Select object"
                );

            // get solid from face
            Solid parentSolid = singleSolidFromElement(doc.GetElement(objectRef));

            // input as PlanarFace:
            GeometryObject inputGeoObj = doc.GetElement(objectRef).GetGeometryObjectFromReference(objectRef);
            PlanarFace inputPlanarFace = inputGeoObj as PlanarFace;


            Solid fullBox = solidBoundingBox(parentSolid);
            Plane splitPlane = Plane.CreateByNormalAndOrigin(inputPlanarFace.FaceNormal, inputPlanarFace.Origin);
            Plane flipPlane = Plane.CreateByNormalAndOrigin(inputPlanarFace.FaceNormal.Negate(), inputPlanarFace.Origin);

            Solid outHalf = BooleanOperationsUtils.CutWithHalfSpace(fullBox, splitPlane);
            Solid inHalf = BooleanOperationsUtils.CutWithHalfSpace(fullBox, flipPlane);

            // Create freeform element in current document
            if (doc.IsFamilyDocument)
            {
                using (Transaction t = new Transaction(doc, "Add element"))
                {
                    t.Start();
                    FreeFormElement elemOut = FreeFormElement.Create(doc, outHalf);
                    FreeFormElement elemIn = FreeFormElement.Create(doc, inHalf);
                    t.Commit();
                }
                return Result.Succeeded;
            }
            else
            {

                message = " this only works in family (.rfa) file context";
                return Result.Failed;
            }

        }


        /// <summary>
        /// return Solid based on bounding box coordinates retrieved from input solid
        /// </summary>
        /// <param name="inputSolid"></param>
        /// <returns></returns>
        public static Solid solidBoundingBox(Solid inputSolid)
        {
            BoundingBoxXYZ bbox = inputSolid.GetBoundingBox();

            // corners in BBox coords
            XYZ pt0 = new XYZ(bbox.Min.X, bbox.Min.Y, bbox.Min.Z);
            XYZ pt1 = new XYZ(bbox.Max.X, bbox.Min.Y, bbox.Min.Z);
            XYZ pt2 = new XYZ(bbox.Max.X, bbox.Max.Y, bbox.Min.Z);
            XYZ pt3 = new XYZ(bbox.Min.X, bbox.Max.Y, bbox.Min.Z);
            //edges in BBox coords
            Line edge0 = Line.CreateBound(pt0, pt1);
            Line edge1 = Line.CreateBound(pt1, pt2);
            Line edge2 = Line.CreateBound(pt2, pt3);
            Line edge3 = Line.CreateBound(pt3, pt0);
            //create loop, still in BBox coords
            List<Curve> edges = new List<Curve>();
            edges.Add(edge0);
            edges.Add(edge1);
            edges.Add(edge2);
            edges.Add(edge3);
            Double height = bbox.Max.Z - bbox.Min.Z;
            CurveLoop baseLoop = CurveLoop.Create(edges);
            List<CurveLoop> loopList = new List<CurveLoop>();
            loopList.Add(baseLoop);
            Solid preTransformBox = GeometryCreationUtilities.CreateExtrusionGeometry(loopList, XYZ.BasisZ, height);

            Solid transformBox = SolidUtils.CreateTransformed(preTransformBox, bbox.Transform);

            return transformBox;

        }

        /// <summary>
        /// Creates a Solid by merging all solids of an input Element
        /// </summary>
        /// <param name="inputElement"> the element to merge to single solid</param>
        /// <returns></returns>
        public static Solid singleSolidFromElement(Element inputElement)
        {

            Document doc = inputElement.Document;
            Autodesk.Revit.ApplicationServices.Application app = doc.Application;

            // create solid from Element:

            IList<Solid> fromElement = GetTargetSolids(inputElement);

            int solidCount = fromElement.Count;

            // Merge all found solids into single one
            Solid solidResult = null;
            if (solidCount == 1)
            {
                solidResult = fromElement[0];
            }

            else if (solidCount > 1)
            {
                solidResult =
                    BooleanOperationsUtils.ExecuteBooleanOperation(fromElement[0], fromElement[1], BooleanOperationsType.Union);
            }

            if (solidCount > 2)
            {
                for (int i = 2; i < solidCount; i++)
                {
                    solidResult = BooleanOperationsUtils.ExecuteBooleanOperation(solidResult, fromElement[i],
                                                                                         BooleanOperationsType.Union);
                }
            }

            return solidResult;


        }

        /// <summary>
        /// Gets all target solids in a given element.
        /// </summary>
        /// <remarks>Includes solids and solids in first level instances only.  Deeper levels are ignored.  Empty solids are not returned.</remarks>
        /// <param name="element">The element.</param>
        /// <returns>The list of solids.</returns>
        public static IList<Solid> GetTargetSolids(Element element)
        {
            List<Solid> solids = new List<Solid>();


            Options options = new Options();
            options.DetailLevel = ViewDetailLevel.Fine;
            GeometryElement geomElem = element.get_Geometry(options);
            foreach (GeometryObject geomObj in geomElem)
            {
                if (geomObj is Solid)
                {
                    Solid solid = (Solid)geomObj;
                    if (solid.Faces.Size > 0 && solid.Volume > 0.0)
                    {
                        solids.Add(solid);
                    }
                    // Single-level recursive check of instances. If viable solids are more than
                    // one level deep, this example ignores them.
                }
                else if (geomObj is GeometryInstance)
                {
                    GeometryInstance geomInst = (GeometryInstance)geomObj;
                    GeometryElement instGeomElem = geomInst.GetInstanceGeometry();
                    foreach (GeometryObject instGeomObj in instGeomElem)
                    {
                        if (instGeomObj is Solid)
                        {
                            Solid solid = (Solid)instGeomObj;
                            if (solid.Faces.Size > 0 && solid.Volume > 0.0)
                            {
                                solids.Add(solid);
                            }
                        }
                    }
                }
            }
            return solids;
        }
    }
    
    /// <summary>
    /// A Filter for planar face.
    /// Only planar faces are allowed to be picked.
    /// </summary>
    public class PlanarFaceFilter : ISelectionFilter
    {
        // Revit document.
        Document m_doc = null;

        /// <summary>
        /// Constructor the filter and initialize the document.
        /// </summary>
        /// <param name="doc">The document.</param>
        public PlanarFaceFilter(Document doc)
        {
            m_doc = doc;
        }

        /// <summary>
        /// Allow all the element to be selected
        /// </summary>
        /// <param name="element">A candidate element in selection operation.</param>
        /// <returns>Return true to allow the user to select this candidate element.</returns>
        public bool AllowElement(Element element)
        {
            return true;
        }

        /// <summary>
        /// Allow planar face reference to be selected
        /// </summary>
        /// <param name="refer">A candidate reference in selection operation.</param>
        /// <param name="point">The 3D position of the mouse on the candidate reference.</param>
        /// <returns>Return true for planar face reference. Return false for non planar face reference.</returns>
        public bool AllowReference(Reference refer, XYZ point)
        {
            GeometryObject geoObject = m_doc.GetElement(refer).GetGeometryObjectFromReference(refer);
            return geoObject != null && geoObject is PlanarFace;
        }
    }


}


