﻿#region Header
//
// p2c/Command.cs - Pipe to Conduit Converter
//
// Copyright (C) 2010 by Jeremy Tammik, Autodesk Inc. All rights reserved.
//
#endregion // Header

#region Namespaces
using System;
using System.Diagnostics;
using System.Collections.Generic;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.Electrical;
using Autodesk.Revit.DB.Plumbing;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
using InvalidOperationException 
  = Autodesk.Revit.Exceptions
    .InvalidOperationException;
#endregion // Namespaces

namespace p2c
{
  [Transaction(TransactionMode.Automatic)]
  [Regeneration(RegenerationOption.Manual)]
  public class Command : IExternalCommand
  {
    const string _caption = "Pipe to Conduit Converter";
    const string _prompt = "Please pick some pipes to convert to conduits. Click Finish to complete. Click Cancel or hit ESC to cancel.";

    const BuiltInParameter _bipConduitDiameter = BuiltInParameter.RBS_CONDUIT_DIAMETER_PARAM;
    const BuiltInParameter _bipCurveDiameter = BuiltInParameter.RBS_CURVE_DIAMETER_PARAM;
    const BuiltInParameter _bipPipeDiameter = BuiltInParameter.RBS_PIPE_DIAMETER_PARAM;

    #region Formatting and message handlers
    /// <summary>
    /// Return an English plural suffix 's' or
    /// nothing for the given number of items.
    /// </summary>
    public static string PluralSuffix( int n )
    {
      return 1 == n ? "" : "s";
    }

    /// <summary>
    /// MessageBox or Revit TaskDialog 
    /// wrapper for informational message.
    /// </summary>
    public static void InfoMsg( string msg )
    {
      Debug.WriteLine( msg );

      //W.MessageBox.Show( msg, _caption, W.MessageBoxButtons.OK, W.MessageBoxIcon.Information );

      TaskDialog.Show( _caption, msg, 
        TaskDialogCommonButtons.Ok );
    }

    /// <summary>
    /// MessageBox or Revit TaskDialog 
    /// wrapper for error message.
    /// </summary>
    public static void ErrorMsg( string msg )
    {
      Debug.WriteLine( msg );

      //W.MessageBox.Show( msg, _caption, W.MessageBoxButtons.OK, W.MessageBoxIcon.Error );

      TaskDialog d = new TaskDialog( _caption );
      d.MainIcon = TaskDialogIcon.TaskDialogIconWarning;
      d.MainInstruction = msg;
      d.Show();
    }
    #endregion // Message handlers

    /// <summary>
    /// A selection filter for pipe elements.
    /// </summary>
    public class PipeFilter : ISelectionFilter
    {
      const BuiltInCategory _bic = BuiltInCategory.OST_PipeCurves;

      /// <summary>
      /// Allow pipe to be selected.
      /// </summary>
      /// <param name="element">A candidate element in selection operation.</param>
      /// <returns>Return true for pipe, false for all other elements.</returns>
      public bool AllowElement( Element e )
      {
        return null != e.Category
          && e.Category.Id.IntegerValue == ( int ) _bic;
      }

      /// <summary>
      /// Allow all the reference to be selected
      /// </summary>
      /// <param name="refer">A candidate reference in selection operation.</param>
      /// <param name="point">The 3D position of the mouse on the candidate reference.</param>
      /// <returns>Return true to allow the user to select this candidate reference.</returns>
      public bool AllowReference( Reference r, XYZ p )
      {
        return true;
      }
    }

    /// <summary>
    /// Convert the given pipe element to a Revit MEP 
    /// conduit using the specified conduit type.
    /// </summary>
    Conduit ConvertPipeToConduit( 
      ElementId idConduitType, 
      Pipe pipe )
    {
      Document doc = pipe.Document;

      ElementId idLevel = ( null == pipe.Level )
        ? ElementId.InvalidElementId
        : pipe.Level.Id;

      LocationCurve lc = pipe.Location as LocationCurve;
      XYZ startPoint = lc.Curve.get_EndPoint( 0 );
      XYZ endPoint = lc.Curve.get_EndPoint( 1 );

      double diameter = pipe.Diameter;
      Parameter p;

      #region Debug code
    #if DEBUG
      p = pipe.get_Parameter( _bipCurveDiameter );
      Debug.Assert( null != p, "expected valid RBS_CURVE_DIAMETER_PARAM value on pipe" );
      Debug.Assert( StorageType.Double == p.StorageType, "expected double RBS_CURVE_DIAMETER_PARAM value" );
      Debug.Assert( p.AsDouble().Equals( diameter ), "expected RBS_CURVE_DIAMETER_PARAM value to equal diameter" );

      p = pipe.get_Parameter( _bipPipeDiameter );
      Debug.Assert( null != p, "expected valid RBS_PIPE_DIAMETER_PARAM value on pipe" );
      Debug.Assert( StorageType.Double == p.StorageType, "expected double RBS_PIPE_DIAMETER_PARAM value" );
      Debug.Assert( p.AsDouble().Equals( diameter ), "expected RBS_PIPE_DIAMETER_PARAM value to equal diameter" );
    #endif // DEBUG
      #endregion // Debug code

      //
      // new second generation element creation approach.
      // use a static creation method on the element class
      // instead of the Autodesk.Revit.Creation.Document
      // creation methods. This second generation API is
      // automatically generated with RIDL, the Revit 
      // Interface Definition Language:
      //
      // idLevel argument:
      //
      // The element id of the level on which this conduit 
      // is based. If the input level id equals InvalidElementId, 
      // the nearest level is used. 
      //
      // The create method will throw an ArgumentNullException
      // if a null value is provided. 
      //
      // ElementId.InvalidElementId property:
      // public static ElementId InvalidElementId { get; }
      // Get the invalid ElementId whose IntegerValue is -1.
      //

      Conduit conduit = Conduit.Create( doc, idConduitType, 
        startPoint, endPoint, idLevel );

      p = conduit.get_Parameter( _bipConduitDiameter );
      Debug.Assert( null != p, "expected RBS_CONDUIT_DIAMETER_PARAM parameter on conduit" );
      Debug.Assert( StorageType.Double == p.StorageType, "expected double RBS_CONDUIT_DIAMETER_PARAM value" );
      p.Set( diameter );

      doc.Delete( pipe );

      return conduit;
    }

    public Result Execute(
      ExternalCommandData commandData, 
      ref string message, 
      ElementSet elements )
    {
      Result result = Result.Failed;

      UIApplication app = commandData.Application;
      UIDocument uidoc = app.ActiveUIDocument;
      Document doc = uidoc.Document;

      try
      {
        // determine conduit type to use:

        FilteredElementCollector collector 
          = new FilteredElementCollector( doc );

        collector.OfCategory( 
          BuiltInCategory.OST_Conduit );

        collector.OfClass( typeof( 
          ElementType ) );

        ElementId idConduitType 
          = collector.FirstElementId();

        if( null == idConduitType 
          || ElementId.InvalidElementId == idConduitType )
        {
          ErrorMsg( "No conduit types found." );
        }
        else
        {
          //
          // select pipe elements to convert.
          // 
          // One possibility for additional user interaction would be to
          // determine all available conduit types before starting the pipe 
          // selection and displaying a modeless dialogue box in parallel 
          // in whihc the user can select the desired target conduit type.
          //

          IList<Reference> refs 
            = uidoc.Selection.PickObjects(
              ObjectType.Element, 
              new PipeFilter(), 
              _prompt );

          List<Pipe> pipes = new List<Pipe>();

          foreach( Reference r in refs )
          {
            Debug.Assert( null != r, 
              "expected non-null reference from PickObjects" );

            Debug.Assert( null != r.Element, 
              "expected non-null element from PickObjects" );

            Debug.Assert( r.Element is Pipe, 
              "expected PickObjects to return only Pipe elements" );

            pipes.Add( r.Element as Pipe );
          }

          int n = pipes.Count;

          if( 0 == n )
          {
            ErrorMsg( "No pipes selected." );
          }
          else
          {
            n = 0;

            // convert pipe elements to conduit:

            foreach( Pipe pipe in pipes )
            {
              ConvertPipeToConduit( 
                idConduitType, pipe );

              ++n;
            }

            string msg = string.Format( 
              "Converted {0} pipe{1}.",
              n, PluralSuffix( n ) );

            if( 0 == n )
            {
              ErrorMsg( msg );
            }
            else
            {
              doc.Regenerate();
              InfoMsg( msg );
              result = Result.Succeeded;
            }
          }
        }
      }
      catch( InvalidOperationException )
      {
        // selection cancelled

        result = Result.Cancelled;
      }
      catch( Exception ex )
      {
        // if any error occurs, display error 
        // information and return failed

        message = ex.Message;
      }
      return result;
    }
  }
}
