﻿using System;
using System.Collections.Generic;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;

[TransactionAttribute( TransactionMode.ReadOnly )]
[RegenerationAttribute( RegenerationOption.Manual )]
public class GetSlantedWallAndAngle : IExternalCommand
{
  public Result Execute( 
    ExternalCommandData commandData, 
    ref string messages, 
    ElementSet elements )
  {
    UIApplication app = commandData.Application;
    Document doc = app.ActiveUIDocument.Document;

    FilteredElementCollector collector 
      = new FilteredElementCollector( doc )
        //.OfClass( typeof( FaceWall ) )
        .OfCategory( BuiltInCategory.OST_Walls );

    FaceWall slantedWall = null;

    foreach( Element elem in collector )
    {
      // use the class to identify slanted wall

      if( elem is FaceWall ) 
      {
        slantedWall = elem as FaceWall;
        break;
      }
    }

    if( slantedWall == null )
    {
      messages = "There is no slanted wall in this model";
      return Result.Failed;
    }

    // retrieve geometry

    Options options = app.Application.Create
      .NewGeometryOptions();

    GeometryElement geoElement 
      = slantedWall.get_Geometry( options );

    // get the only solid

    Solid wallSolid = null;

    foreach( GeometryObject geoObject 
      in geoElement.Objects )
    {
      if( geoObject is Solid )
      {
        wallSolid = geoObject as Solid;
        break;
      }
    }

    // get the wall width

    ElementId typeId = slantedWall.GetTypeId();
    
    WallType type = doc.get_Element( typeId ) 
      as WallType;

    double dWidth = type.Width;

    double dTolerance = 0.01;
    double dAngle = 0.0;
    foreach( PlanarFace face in wallSolid.Faces )
    {
      if( dTolerance > Math.Abs( 
        face.Area * dWidth - wallSolid.Volume ) )
      {
        if( face.Normal.Z > dTolerance )
        {
          // calculate the angle

          XYZ verticalVector = XYZ.BasisZ;

          dAngle = verticalVector.AngleTo( 
            face.Normal ) * 180 / 3.1415926;

          TaskDialog.Show( 
            "Slanted wall angle to XOY plane", 
            dAngle.ToString() );

          break;
        }
      }
    }
    return Result.Succeeded;
  }
}