﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpatialAnalysis.ExternalConnection;

namespace SpatialAnalysis.Geometry
{
    public class UVLine
    {
        public UV Start { get; set; }
        public UV End { get; set; }
        public UVLine(UV start, UV end)
        {
            this.Start = start;
            this.End = end;
        }
        #region utility functions
        public double GetLength()
        {
            return this.Start.DistanceTo(this.End);
        }
        public UV FindPoint(double u)
        {
            UV p = this.Start + (u / (this.End.DistanceTo(this.Start))) * (this.End - this.Start);
            return p;
        }
        public void Invert()
        {
            UV p = new UV(this.Start.U, this.Start.V);
            this.Start = this.End;
            this.End = p;
        }
        /// <summary>
        /// return the direction of the line which is not normalized
        /// </summary>
        /// <returns></returns>
        public UV GetDirection()
        {
            return this.End - this.Start;
        }
        public double? Intersection(UVLine l, double tolerance = .000001f)
        {
            UV lineVector = this.End - this.Start;
            double area1 = lineVector.CrossProductValue(l.Start - this.Start);
            double area2 = lineVector.CrossProductValue(l.End - this.Start);
            if (area1 * area2 > tolerance)
            {
                lineVector = null;
                return null;
            }
            lineVector = l.End - l.Start;
            area1 = lineVector.CrossProductValue(this.Start - l.Start);
            area2 = lineVector.CrossProductValue(this.End - l.Start);
            if (area1 * area2 > tolerance)
            {
                lineVector = null;
                return null;
            }
            double lengthL = l.GetLength();
            double a1 = (l.Start - this.Start).CrossProductValue(l.End - this.Start);
            if (a1 == 0)
            {
                return Math.Min(l.Start.DistanceTo(this.Start), l.End.DistanceTo(this.Start));
            }
            lineVector = null;
            double u = this.GetLength() * Math.Abs(a1) / (Math.Abs(area1) + Math.Abs(area2));
            return u;
        }
        public bool Intersects(UVLine l, double tolerance = 0)
        {
            UV lineVector = this.End - this.Start;
            double area1 = lineVector.CrossProductValue(l.Start - this.Start);
            double area2 = lineVector.CrossProductValue(l.End - this.Start);
            if (area1 * area2 > tolerance)
            {
                lineVector = null;
                return false;
            }
            lineVector = l.End - l.Start;
            area1 = lineVector.CrossProductValue(this.Start - l.Start);
            area2 = lineVector.CrossProductValue(this.End - l.Start);
            if (area1 * area2 > tolerance)
            {
                lineVector = null;
                return false;
            }
            return true;
        }
        public void Visualize(IVisualize visualizer, double elevation)
        {
            visualizer.VisualizeLine(this, elevation);
        }
        #endregion

        public override int GetHashCode()
        {
            UV less = new UV(), more = new UV();
            if (Start.CompareTo(End) > 0)
            {
                less = End;
                more = Start;
            }
            else
            {
                less = Start;
                more = End;
            }
            int hash = less.GetHashCode();
            hash = 71 * hash + more.U.GetHashCode();
            hash = 71 * hash + more.V.GetHashCode();
            return hash;
        }
        public override bool Equals(object obj)
        {
            UVLine l = obj as UVLine;
            if (l==null)
            {
                return false;
            }
            else
            {
                if ((l.Start == this.Start && l.End == this.End) ||
                    (l.End == this.Start && l.Start == this.End))
                {
                    return true;
                }
            }
            return false;
        }
        public override string ToString()
        {
            return string.Format("Start ({0}); End ({1})", this.Start.ToString(), this.End.ToString());
        }

    }
}
