﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpatialAnalysis.ExternalConnection;

namespace SpatialAnalysis.Geometry
{
    public class UV : IComparable<UV>
    {
        public double U { get; set; }
        public double V { get; set; }
        public UV() { }
        public UV(double x, double y)
        {
            this.U = x;
            this.V = y;
        }

        #region Utility functions
        public double CrossProductValue(UV vector2D)
        {
            return this.U * vector2D.V - this.V * vector2D.U;
        }

        public UV RotateNew(double angle)
        {
            double x = Math.Cos(angle) * this.U - Math.Sin(angle) * this.V;
            double y = Math.Sin(angle) * this.U + Math.Cos(angle) * this.V;
            return new UV(x, y);
        }
        public void Rotate(double angle)
        {
            double x = Math.Cos(angle) * this.U - Math.Sin(angle) * this.V;
            double y = Math.Sin(angle) * this.U + Math.Cos(angle) * this.V;
            this.U = x;
            this.V = y;
        }
        public double Length()
        {
            return Math.Sqrt(this.U * this.U + this.V * this.V);
        }
        public double LengthSquared()
        {
            return this.U * this.U + this.V * this.V;
        }
        public static double DistanceSquared(UV a, UV b)
        {
            return (a.U - b.U) * (a.U - b.U) + (a.V - b.V) * (a.V - b.V);
        }
        public static double DistanceBetween(UV a, UV b)
        {
            return Math.Sqrt((a.U - b.U) * (a.U - b.U) + (a.V - b.V) * (a.V - b.V));
        }
        public void Unitize()
        {
            double length = this.Length();
            if (length != 0)
            {
                this.U /= length;
                this.V /= length;
            }
        }

        public bool AlmostEqualsTo(object p, double DistTolerance = .0001)
        {
            UV xy = p as UV;
            if (xy != null)
            {
                if (this.DistanceTo(xy) < DistTolerance)
                {
                    return true;
                }
            }
            return false;
        }
        public double Angle()
        {

            if (this.V == 0)
            {
                if (U > 0)
                {
                    return 0;
                }
                if (U < 0)
                {
                    return Math.PI;
                }
            }

            if (U == 0)
            {
                if (V > 0)
                {
                    return .5 * Math.PI;
                }
                if (V < 0)
                {
                    return 1.5 * Math.PI;
                }
            }
            if (U == 0 && V == 0)
            {
                return double.NaN;
            }
            double sin = this.V / Math.Sqrt(this.U * this.U + this.V * this.V);
            double a = Math.Asin(Math.Abs(sin));
            if (this.U > 0 && V > 0)
            {
                return a;
            }
            if (this.U > 0 && V < 0)
            {
                return Math.PI * 2 - a;
            }
            if (this.U < 0 && V < 0)
            {
                return Math.PI + a;
            }
            if (U < 0 && V > 0)
            {
                return Math.PI - a;
            }

            return 0;
        }

        public double AngleTo(UV xy)
        {
            return xy.Angle() - this.Angle();
        }
        public double DistanceTo(UV xy)
        {
            return Math.Sqrt((this.U - xy.U) * (this.U - xy.U) + (this.V - xy.V) * (this.V - xy.V));
        }

        public double DotProduct(UV xy)
        {
            return this.U * xy.U + this.V * xy.V;
        }
        #endregion

        #region Defining operators
        public static bool operator ==(UV a, UV b)
        {
            if (System.Object.ReferenceEquals(a, b))
            {
                return true;
            }
            if ((object)a == null || (object)b == null)
            {
                return false;
            }
            return a.U == b.U && a.V == b.V;
        }
        public static bool operator !=(UV a, UV b)
        {
            return !(a == b);
        }
        public static UV operator +(UV a, UV b)
        {
            return new UV(a.U + b.U, a.V + b.V);
        }
        public static UV operator -(UV a, UV b)
        {
            return new UV(a.U - b.U, a.V - b.V);
        }
        public static UV operator /(UV a, double k)
        {
            return new UV(a.U / k, a.V / k);
        }
        public static UV operator *(UV a, double k)
        {
            return new UV(a.U * k, a.V * k);
        }
        public static UV operator *(double k, UV a)
        {
            return new UV(a.U * k, a.V * k);
        }
        public static UV Unit_U()
        {
            return new UV(1, 0);
        }
        public static UV Unit_V()
        {
            return new UV(0, 1);
        }
        public static UV Zero()
        {
            return new UV(0, 0);
        }
        #endregion

        public override bool Equals(object p)
        {
            UV xy = p as UV;
            if (xy != null)
            {
                if (this.U == xy.U && this.V == xy.V)
                {
                    return true;
                }
            }
            return false;
        }
        public override int GetHashCode()
        {
            int hash = 7;
            hash = 71 * hash + this.U.GetHashCode();
            hash = 71 * hash + this.V.GetHashCode();
            return hash;
        }
        public override string ToString()
        {
            return string.Format("U = {0}; V = {1}", this.U.ToString(), this.V.ToString());
        }
        public double DistanceTo(UVLine line)
        {
            double area = (line.Start - line.End).CrossProductValue(line.Start - this);
            area = Math.Abs(area);
            return area / line.GetLength();
        }
        public UV Projection(UVLine line)
        {
            double u = (line.End - line.Start).DotProduct(this - line.Start);
            u /= line.GetLength();
            return line.FindPoint(u);
        }
        public double ClosestDistance(UVLine line)
        {
            UV p = new UV();
            double length = line.GetLength();
            double u = (line.End - line.Start).DotProduct(this - line.Start);
            u /= length;
            if (u < 0)
            {
                p = line.Start;
            }
            else if (u > length)
            {
                p = line.End;
            }
            else
            {
                p = this.ClosestPoint(line);
            }
            return p.DistanceTo(this);
        }
        public UV ClosestPoint(UVLine line)
        {
            double length = line.GetLength();
            double u = (line.End - line.Start).DotProduct(this - line.Start);
            u /= length;
            if (u < 0)
            {
                return line.Start;
            }
            if (u > length)
            {
                return line.End;
            }
            return line.FindPoint(u);
        }
        public void ShowPoint(IVisualize visualizer, double size, double height = 0.0d)
        {
            visualizer.VisualizePoint(this, size, height);
        }
        public int CompareTo(UV other)
        {
            int uCompared = this.U.CompareTo(other.U);
            if (uCompared != 0)
            {
                return uCompared;
            }
            return this.V.CompareTo(other.V);
        }
        public UV GetReflection(UV vector)
        {
            if (vector.LengthSquared() != 1)
            {
                vector.Unitize();
            }
            double dotPro = this.DotProduct(vector);
            UV horizontal_Component = vector * dotPro;
            UV normal_Component = this - horizontal_Component;
            var result = horizontal_Component - normal_Component;
            return result;
        }
        public UV GetReflection(UVLine line)
        {
            var vector = line.GetDirection();
            vector.Unitize();
            double dotPro = this.DotProduct(vector);
            UV horizontal_Component = vector * dotPro;
            UV normal_Component = this - horizontal_Component;
            var result = horizontal_Component - normal_Component;
            return result;
        }
        public UV Copy()
        {
            return new UV(this.U, this.V);
        }


    }
}
