﻿using System;
using System.Data;
using System.Diagnostics;
using System.IO;
using System.IO.Packaging;
using System.Linq;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Text.RegularExpressions;
using System.Windows.Forms;

namespace RvtVer
{
    public partial class frmMain : Form
    {
        private const string StreamName = "BasicFileInfo";

        private static Regex FoundYear = new Regex(@"\s\d{4}\s");
        private static Regex FoundYear2019 = new Regex(@"^(\d{4})\u0012");


        public frmMain(string file)
        {
            InitializeComponent();
            f = file;
        }

        private string f;

        private void Form1_Load(object sender, EventArgs e)
        {

        }

        private void GetRevitVersion()
        {
            var rf = new RevitFile(f);
            GetYearFromFileInfo(f, rf);
            lblRevitFile.Text = "Opening " + Path.GetFileName(f); lblRevitFile.Refresh();
            lblRevitVersion.Text = "With Revit ver: " + rf.Version; lblRevitVersion.Refresh();
            OpenRvt(rf);
            System.Threading.Thread.Sleep(5000);
            Environment.Exit(0);
        }

        private static void GetYearFromFileInfo(string pathToRevitFile, RevitFile rf)
        {
            try
            {
                var rawData = GetRawBasicFileInfo(
                    pathToRevitFile);
                if (rawData != null)
                {


                    var rawString = System.Text.Encoding.Unicode
                        .GetString(rawData);

                    var fileInfoData = rawString.Split(
                        new string[] { "\0", "\r\n" },
                        StringSplitOptions.RemoveEmptyEntries);

                    foreach (var info in fileInfoData)
                    {
                        //Console.WriteLine(info);
                        if (info.Contains("Autodesk"))
                        {
                            Match match = FoundYear.Match(info);
                            if (match.Success)
                            {
                                rf.Version = match.Value.Replace(" ", "");
                                rf.AdditionalInfo = info.Replace("Revit Build: ", "");
                                break;
                            }

                        }

                    }

                    //2019+ don't use the conventions above :(
                    if (string.IsNullOrEmpty(rf.Version))
                    {
                        foreach (var info in fileInfoData.Select((val, i) => new { i, val }))
                        {
                            Match yearMatch = FoundYear2019.Match(info.val);

                            if (yearMatch.Success)
                            {
                                rf.Version = yearMatch.Groups[1].Value;
                                rf.AdditionalInfo = "Build: " + fileInfoData[info.i + 1];
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private static byte[] GetRawBasicFileInfo(string revitFileName)
        {
            try
            {
                if (!StructuredStorageUtils.IsFileStucturedStorage(revitFileName))
                {
                    _ = MessageBox.Show("File is not a structured storage file");
                    throw new NotSupportedException("File is not a structured storage file");
                }

                using (StructuredStorageRoot ssRoot = new StructuredStorageRoot(revitFileName))
                {
                   
                    if (!ssRoot.BaseRoot.StreamExists(StreamName))
                    {
                        _ = MessageBox.Show(string.Format("File doesn't contain {0} stream", StreamName));
                        throw new NotSupportedException(string.Format("File doesn't contain {0} stream", StreamName));
                    }

                    StreamInfo imageStreamInfo = ssRoot.BaseRoot.GetStreamInfo(StreamName);

                    using (Stream stream = imageStreamInfo.GetStream(FileMode.Open, FileAccess.Read))
                    {
                        byte[] buffer = new byte[stream.Length];
                        stream.Read(buffer, 0, buffer.Length);
                        return buffer;
                    }
                }
            }
            catch (System.Exception ex1)
            {
               // MessageBox.Show("exception: " + ex1);
            }
            return null;
        }


        public static class StructuredStorageUtils
        {
            [DllImport("ole32.dll")]
            private static extern int StgIsStorageFile(
                [MarshalAs(UnmanagedType.LPWStr)] string pwcsName);

            public static bool IsFileStucturedStorage(
                string fileName)
            {
                int res = StgIsStorageFile(fileName);

                if (res == 0)
                {
                    return true;
                }

                if (res == 1)
                {
                    return false;
                }

                MessageBox.Show("File not found");
                throw new FileNotFoundException(
                    "File not found", fileName);
            }
        }

        public class StructuredStorageException : Exception
        {
            public StructuredStorageException()
            {
            }

            public StructuredStorageException(string message)
              : base(message)
            {
            }

            public StructuredStorageException(
                string message,
                Exception innerException)
              : base(message, innerException)
            {
            }
        }

        public class StructuredStorageRoot : IDisposable
        {
            private StorageInfo _storageRoot;

            public StructuredStorageRoot(Stream stream)
            {
                try
                {
                    _storageRoot
                        = (StorageInfo)InvokeStorageRootMethod(
                            null, "CreateOnStream", stream);
                }
                catch (Exception ex)
                {
                    //MessageBox.Show("Cannot get StructuredStorageRoot: " + ex.Message);
                    //throw new StructuredStorageException(
                    //    "Cannot get StructuredStorageRoot", ex);
                }
            }

            public StructuredStorageRoot(string fileName)
            {
                try
                {
                    _storageRoot
                        = (StorageInfo)InvokeStorageRootMethod(
                            null, "Open", fileName, FileMode.Open,
                            FileAccess.Read, FileShare.Read);
                }
                catch (Exception ex)
                {
                    //MessageBox.Show("Cannot get StructuredStorageRoot: " + ex.Message);
                    //throw new StructuredStorageException(
                    //    "Cannot get StructuredStorageRoot", ex);
                }
            }

            private static object InvokeStorageRootMethod(StorageInfo storageRoot, string methodName, params object[] methodArgs)
            {
                try
                {
                    Type storageRootType = typeof(StorageInfo).Assembly.GetType("System.IO.Packaging.StorageRoot", false, false);

                    object result = storageRootType.InvokeMember(
                      methodName,
                      BindingFlags.Static | BindingFlags.Instance
                      | BindingFlags.Public | BindingFlags.NonPublic
                      | BindingFlags.InvokeMethod,
                      null, storageRoot, methodArgs);

                    return result;
                }
                catch (Exception ex)
                {
                }
                return null;
            }

            private
            void CloseStorageRoot()
            {
                InvokeStorageRootMethod(_storageRoot, "Close");
            }

            #region Implementation of IDisposable

            public void Dispose()
            {
                CloseStorageRoot();
            }

            #endregion

            public StorageInfo BaseRoot
            {
                get { return _storageRoot; }
            }

        }


        private void OpenRvt(RevitFile rvtFile)
        {
            if (rvtFile == null)
            {
                return;
            }

            var programFiles = Environment.GetFolderPath(Environment.SpecialFolder.ProgramFiles) + "\\Autodesk\\";

            if (programFiles.Contains("Program Files (x86)"))
            {
                programFiles = programFiles.Replace("Program Files (x86)", "Program Files");
            }

        lookagain:

            var appPath = String.Empty;
            if (rvtFile.Version != null)
                if (rvtFile.Version.Contains("2016"))
                {
                    appPath = programFiles + "Revit Architecture 2016\\Revit.exe";
                }
                else if (rvtFile.Version.Contains("2017"))
                {
                    appPath = programFiles + "Revit 2017\\Revit.exe";
                }
                else if (rvtFile.Version.Contains("2018"))
                {
                    appPath = programFiles + "Revit 2018\\Revit.exe";
                }
                else if (rvtFile.Version.Contains("2019"))
                {
                    appPath = programFiles + "Revit 2019\\Revit.exe";
                }
                else if (rvtFile.Version.Contains("2020"))
                {
                    appPath = programFiles + "Revit 2020\\Revit.exe";
                }
                else if (rvtFile.Version.Contains("2021"))
                {
                    appPath = programFiles + "Revit 2021\\Revit.exe";
                }

            if (string.IsNullOrEmpty(appPath))
            {
                if (!programFiles.Contains("D:"))
                {
                    programFiles = programFiles.Replace("C:", "D:");
                    goto lookagain;
                }
                else
                {
                    Hide();
                    _ = MessageBox.Show("Project could not be opened", "Revit file is locked");
                    Environment.Exit(0);
                }
            }
            else
            if (File.Exists(appPath))
            {
                ProcessStartInfo startInfo = new ProcessStartInfo(appPath);
                startInfo.WindowStyle = ProcessWindowStyle.Maximized;
                startInfo.Arguments = "\"" + rvtFile.FullPath + "\"";
                Process.Start(startInfo);
            }

        }

        private void frmMain_Shown(object sender, EventArgs e)
        {
            BringToFront();
            Refresh();
            GetRevitVersion();
        }
    }
}
